// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "device/power_save_blocker/power_save_blocker.h"

#include "base/bind.h"
#include "base/location.h"
#include "base/logging.h"
#include "base/macros.h"
#include "base/memory/ref_counted.h"
#include "device/base/device_client.h"
#include "device/power_save_blocker/power_save_blocker_service.h"

namespace device {

class PowerSaveBlocker::Delegate
    : public base::RefCountedThreadSafe<PowerSaveBlocker::Delegate> {
 public:
  Delegate(PowerSaveBlockerType type,
           Reason reason,
           const std::string& description,
           scoped_refptr<base::SequencedTaskRunner> ui_task_runner);

  void CreateBlocker();
  void CancelBlocker();

 private:
  friend class base::RefCountedThreadSafe<Delegate>;
  ~Delegate();

  void CreateBlockerOnUIThread();
  void CancelBlockerOnUIThread();

  PowerSaveBlockerType type_;
  Reason reason_;
  std::string description_;
  scoped_refptr<base::SequencedTaskRunner> ui_task_runner_;

  int block_id_;

  DISALLOW_COPY_AND_ASSIGN(Delegate);
};

PowerSaveBlocker::Delegate::~Delegate() {
  DCHECK_EQ(block_id_, -1);
}

void PowerSaveBlocker::Delegate::CreateBlockerOnUIThread() {
  if (!DeviceClient::Get() ||
      !DeviceClient::Get()->GetPowerSaveBlockerService()) {
    LOG(ERROR)
        << "Cannot create PowerSaveBlocker - no DeviceClient or "
        << "PowerSaveBlockerService provided by embedder";
  }

  block_id_ =
      DeviceClient::Get()->GetPowerSaveBlockerService()
          ->CreatePowerSaveBlocker(type_, reason_, description_);
}

void PowerSaveBlocker::Delegate::CancelBlockerOnUIThread() {
  if (block_id_ == -1 || !DeviceClient::Get()) {
    return;
  }

  DeviceClient::Get()->GetPowerSaveBlockerService()
      ->CancelPowerSaveBlocker(block_id_);
  block_id_ = -1;
}

PowerSaveBlocker::Delegate::Delegate(
    PowerSaveBlockerType type,
    Reason reason,
    const std::string& description,
    scoped_refptr<base::SequencedTaskRunner> ui_task_runner)
    : type_(type),
      reason_(reason),
      description_(description),
      ui_task_runner_(ui_task_runner),
      block_id_(-1) {}

void PowerSaveBlocker::Delegate::CreateBlocker() {
  ui_task_runner_->PostTask(
      FROM_HERE,
      base::Bind(&Delegate::CreateBlockerOnUIThread, this));
}

void PowerSaveBlocker::Delegate::CancelBlocker() {
  ui_task_runner_->PostTask(
      FROM_HERE,
      base::Bind(&Delegate::CancelBlockerOnUIThread, this));
}

PowerSaveBlocker::PowerSaveBlocker(
    PowerSaveBlockerType type,
    Reason reason,
    const std::string& description,
    scoped_refptr<base::SequencedTaskRunner> ui_task_runner,
    scoped_refptr<base::SingleThreadTaskRunner> blocking_task_runner)
    : delegate_(new Delegate(type, reason, description, ui_task_runner)),
      ui_task_runner_(ui_task_runner),
      blocking_task_runner_(blocking_task_runner) {
  delegate_->CreateBlocker();
}

PowerSaveBlocker::~PowerSaveBlocker() {
  delegate_->CancelBlocker();
}

} // namespace device
