/*
 * Copyright (C) 2022 Michael Brown <mbrown@fensystems.co.uk>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * You can also choose to distribute this program under the terms of
 * the Unmodified Binary Distribution Licence (as given in the file
 * COPYING.UBDL), provided that you have satisfied its requirements.
 */

FILE_LICENCE ( GPL2_OR_LATER_OR_UBDL );

/** @file
 *
 * Ephemeral Diffie-Hellman self-tests
 *
 * NIST test vectors are taken from
 *
 *   https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/keymgmt/KASTestVectorsFFC2014.zip
 *
 * using the files in the "KASTestVectorsFFC2014/Test of 800-56A
 * excluding KDF/FFC Ephemeral Scheme" subdirectory
 */

/* Forcibly enable assertions */
#undef NDEBUG

#include <stdint.h>
#include <string.h>
#include <ipxe/dhe.h>
#include <ipxe/test.h>

/** Define inline prime modulus data */
#define MODULUS(...) { __VA_ARGS__ }

/** Define inline generator data */
#define GENERATOR(...) { __VA_ARGS__ }

/** Define inline partner public key data */
#define PARTNER(...) { __VA_ARGS__ }

/** Define inline private key data */
#define PRIVATE(...) { __VA_ARGS__ }

/** Define inline public key data */
#define PUBLIC(...) { __VA_ARGS__ }

/** Define inline shared secret data */
#define SHARED(...) { __VA_ARGS__ }

/** An Ephemeral Diffie-Hellman self-test */
struct dhe_test {
	/** Prime modulus */
	const void *modulus;
	/** Length of prime modulus */
	size_t len;
	/** Generator */
	const void *generator;
	/** Length of generator */
	size_t generator_len;
	/** Partner public key */
	const void *partner;
	/** Length of partner public key */
	size_t partner_len;
	/** Private key */
	const void *private;
	/** Length of private key */
	size_t private_len;
	/** Expected public key */
	const void *public;
	/** Length of expected public key */
	size_t public_len;
	/** Expected shared secret */
	const void *shared;
	/** Length of shared secret */
	size_t shared_len;
};

/**
 * Define an Ephemeral Diffie-Hellman test
 *
 * @v name		Test name
 * @v MODULUS		Prime modulus
 * @v GENERATOR		Generator
 * @v PARTNER		Partner public key
 * @v PRIVATE		Private key
 * @v PUBLIC		Expected public key
 * @v SHARED		Expected shared secret
 * @ret test		Ephemeral Diffie-Hellman test
 */
#define DHE_TEST( name, MODULUS, GENERATOR, PARTNER, PRIVATE, PUBLIC,	\
		  SHARED )						\
	static const uint8_t name ## _modulus[] = MODULUS;		\
	static const uint8_t name ## _generator[] = GENERATOR;		\
	static const uint8_t name ## _partner[] = PARTNER;		\
	static const uint8_t name ## _private[] = PRIVATE;		\
	static const uint8_t name ## _public[] = PUBLIC;		\
	static const uint8_t name ## _shared[] = SHARED;		\
	static struct dhe_test name = {					\
		.modulus = name ## _modulus,				\
		.len = sizeof ( name ## _modulus ),			\
		.generator = name ## _generator,			\
		.generator_len = sizeof ( name ## _generator ),		\
		.partner = name ## _partner,				\
		.partner_len = sizeof ( name ## _partner ),		\
		.private = name ## _private,				\
		.private_len = sizeof ( name ## _private ),		\
		.public = name ## _public,				\
		.public_len = sizeof ( name ## _public ),		\
		.shared = name ## _shared,				\
		.shared_len = sizeof ( name ## _shared ),		\
	}

/**
 * Report an Ephemeral Diffie-Hellman test result
 *
 * @v test		Ephemeral Diffie-Hellman test
 * @v file		Test code file
 * @v line		Test code line
 */
static void dhe_key_okx ( struct dhe_test *test, const char *file,
			  unsigned int line ) {
	uint8_t public[test->len];
	uint8_t shared[test->len];

	/* Sanity checks */
	okx ( test->generator_len <= test->len, file, line );
	okx ( test->partner_len <= test->len, file, line );
	okx ( test->private_len <= test->len, file, line );
	okx ( test->public_len == test->len, file, line );
	okx ( test->shared_len == test->len, file, line );

	/* Calculate Diffie-Hellman key */
	okx ( dhe_key ( test->modulus, test->len, test->generator,
			test->generator_len, test->partner, test->partner_len,
			test->private, test->private_len, public,
			shared ) == 0, file, line );

	/* Compare against expected result */
	okx ( memcmp ( public, test->public, test->len ) == 0, file, line );
	okx ( memcmp ( shared, test->shared, test->len ) == 0, file, line );
}
#define dhe_key_ok( test ) dhe_key_okx ( test, __FILE__, __LINE__ )

/* KASValidityTest_FFCEphem_NOKC_ZZOnly_init.fax test 0 */
DHE_TEST ( kasvaliditytest_ffcephem_nokc_zzonly_init_fb_0,
	   MODULUS ( 0xc5, 0x7c, 0xa2, 0x4f, 0x4b, 0xd6, 0x8c, 0x3c, 0xda,
		     0xc7, 0xba, 0xaa, 0xea, 0x2e, 0x5c, 0x1e, 0x18, 0xb2,
		     0x7b, 0x8c, 0x55, 0x65, 0x9f, 0xea, 0xe0, 0xa1, 0x36,
		     0x53, 0x2b, 0x36, 0xe0, 0x4e, 0x3e, 0x64, 0xa9, 0xe4,
		     0xfc, 0x8f, 0x32, 0x62, 0x97, 0xe4, 0xbe, 0xf7, 0xc1,
		     0xde, 0x07, 0x5a, 0x89, 0x28, 0xf3, 0xfe, 0x4f, 0xfe,
		     0x68, 0xbc, 0xfb, 0x0a, 0x7c, 0xa4, 0xb3, 0x14, 0x48,
		     0x89, 0x9f, 0xaf, 0xb8, 0x43, 0xe2, 0xa0, 0x62, 0x5c,
		     0xb4, 0x88, 0x3f, 0x06, 0x50, 0x11, 0xfe, 0x65, 0x8d,
		     0x49, 0xd2, 0xf5, 0x4b, 0x74, 0x79, 0xdb, 0x06, 0x62,
		     0x92, 0x89, 0xed, 0xda, 0xcb, 0x87, 0x37, 0x16, 0xd2,
		     0xa1, 0x7a, 0xe8, 0xde, 0x92, 0xee, 0x3e, 0x41, 0x4a,
		     0x91, 0x5e, 0xed, 0xf3, 0x6c, 0x6b, 0x7e, 0xfd, 0x15,
		     0x92, 0x18, 0xfc, 0xa7, 0xac, 0x42, 0x85, 0x57, 0xe9,
		     0xdc, 0xda, 0x55, 0xc9, 0x8b, 0x28, 0x9e, 0xc1, 0xc4,
		     0x46, 0x4d, 0x88, 0xed, 0x62, 0x8e, 0xdb, 0x3f, 0xb9,
		     0xd7, 0xc8, 0xe3, 0xcf, 0xb8, 0x34, 0x2c, 0xd2, 0x6f,
		     0x28, 0x06, 0x41, 0xe3, 0x66, 0x8c, 0xfc, 0x72, 0xff,
		     0x26, 0x3b, 0x6b, 0x6c, 0x6f, 0x73, 0xde, 0xf2, 0x90,
		     0x29, 0xe0, 0x61, 0x32, 0xc4, 0x12, 0x74, 0x09, 0x52,
		     0xec, 0xf3, 0x1b, 0xa6, 0x45, 0x98, 0xac, 0xf9, 0x1c,
		     0x65, 0x8e, 0x3a, 0x91, 0x84, 0x4b, 0x23, 0x8a, 0xb2,
		     0x3c, 0xc9, 0xfa, 0xea, 0xf1, 0x38, 0xce, 0xd8, 0x05,
		     0xe0, 0xfa, 0x44, 0x68, 0x1f, 0xeb, 0xd9, 0x57, 0xb8,
		     0x4a, 0x97, 0x5b, 0x88, 0xc5, 0xf1, 0xbb, 0xb0, 0x49,
		     0xc3, 0x91, 0x7c, 0xd3, 0x13, 0xb9, 0x47, 0xbb, 0x91,
		     0x8f, 0xe5, 0x26, 0x07, 0xab, 0xa9, 0xc5, 0xd0, 0x3d,
		     0x95, 0x41, 0x26, 0x92, 0x9d, 0x13, 0x67, 0xf2, 0x7e,
		     0x11, 0x88, 0xdc, 0x2d ),
	   GENERATOR ( 0x4a, 0x1a, 0xf3, 0xa4, 0x92, 0xe9, 0xee, 0x74, 0x6e,
		       0x57, 0xd5, 0x8c, 0x2c, 0x5b, 0x41, 0x41, 0x5e, 0xd4,
		       0x55, 0x19, 0xdc, 0xd9, 0x32, 0x91, 0xf7, 0xfd, 0xc2,
		       0x57, 0xff, 0x03, 0x14, 0xdb, 0xf1, 0xb7, 0x60, 0x0c,
		       0x43, 0x59, 0x3f, 0xff, 0xac, 0xf1, 0x80, 0x9a, 0x15,
		       0x6f, 0xd8, 0x6e, 0xb7, 0x85, 0x18, 0xc8, 0xec, 0x4e,
		       0x59, 0x4a, 0xe2, 0x91, 0x43, 0x4c, 0xeb, 0x95, 0xb6,
		       0x2e, 0x9a, 0xea, 0x53, 0x68, 0x80, 0x64, 0x69, 0x40,
		       0xf9, 0xec, 0xbd, 0x85, 0x89, 0x26, 0x97, 0x67, 0xaf,
		       0xb0, 0xad, 0x00, 0x1b, 0xd4, 0xfd, 0x94, 0xd3, 0xe9,
		       0x92, 0xb1, 0xb4, 0xbc, 0x5a, 0xaa, 0x92, 0x80, 0x89,
		       0x3b, 0x39, 0x05, 0x6c, 0x22, 0x26, 0xfe, 0x5a, 0x28,
		       0x6c, 0x37, 0x50, 0x5a, 0x38, 0x99, 0xcf, 0xf3, 0xc1,
		       0x96, 0x45, 0xdc, 0x01, 0xcb, 0x20, 0x87, 0xa5, 0x00,
		       0x8c, 0xf5, 0x4d, 0xc2, 0xef, 0xb8, 0x9b, 0xd1, 0x87,
		       0xbe, 0xed, 0xd5, 0x0a, 0x29, 0x15, 0x34, 0x59, 0x4c,
		       0x3a, 0x05, 0x22, 0x05, 0x44, 0x4f, 0x9f, 0xc8, 0x47,
		       0x12, 0x24, 0x8e, 0xa8, 0x79, 0xe4, 0x67, 0xba, 0x4d,
		       0x5b, 0x75, 0x56, 0x95, 0xeb, 0xe8, 0x8a, 0xfa, 0x8e,
		       0x01, 0x8c, 0x1b, 0x74, 0x63, 0xd9, 0x2f, 0xf7, 0xd3,
		       0x44, 0x8f, 0xa8, 0xf5, 0xaf, 0x6c, 0x4f, 0xdb, 0xe7,
		       0xc9, 0x6c, 0x71, 0x22, 0xa3, 0x1d, 0xf1, 0x40, 0xb2,
		       0xe0, 0x9a, 0xb6, 0x72, 0xc9, 0xc0, 0x13, 0x16, 0xa2,
		       0x4a, 0xe1, 0x92, 0xc7, 0x54, 0x23, 0xab, 0x9d, 0xa1,
		       0xa1, 0xe5, 0x0b, 0xed, 0xba, 0xe8, 0x84, 0x37, 0xb2,
		       0xe7, 0xfe, 0x32, 0x8d, 0xfa, 0x1c, 0x53, 0x77, 0x97,
		       0xc7, 0xf3, 0x48, 0xc9, 0xdb, 0x2d, 0x75, 0x52, 0x9d,
		       0x42, 0x51, 0x78, 0x62, 0x68, 0x05, 0x45, 0x15, 0xf8,
		       0xa2, 0x4e, 0xf3, 0x0b ),
	   PARTNER ( 0x7d, 0x06, 0x60, 0xbf, 0xce, 0x9e, 0x72, 0x63, 0xf6,
		     0xbf, 0xc7, 0x9b, 0xca, 0x58, 0xfe, 0xca, 0x71, 0x31,
		     0x07, 0x7d, 0x31, 0xa9, 0x64, 0xdd, 0x3a, 0x2a, 0x9a,
		     0x49, 0x34, 0x20, 0xfb, 0x49, 0x62, 0xbe, 0x61, 0xae,
		     0xfa, 0x37, 0x63, 0x53, 0x54, 0x56, 0x36, 0x71, 0xf6,
		     0x61, 0x86, 0xe8, 0x09, 0x6e, 0x6e, 0x88, 0x81, 0x60,
		     0xce, 0x1d, 0xdc, 0xf1, 0xb5, 0xfe, 0x4b, 0xf5, 0xc9,
		     0xf6, 0xc9, 0x9a, 0x63, 0x01, 0x5d, 0x17, 0x23, 0xad,
		     0xcd, 0x8a, 0x2a, 0xeb, 0xd4, 0x84, 0x7a, 0xb0, 0x22,
		     0x89, 0x6f, 0x81, 0x07, 0x11, 0x4c, 0xbc, 0xf3, 0x4e,
		     0xa3, 0x24, 0x35, 0xd5, 0xa2, 0x68, 0x9f, 0x73, 0x56,
		     0xd3, 0x89, 0x4a, 0xea, 0xfd, 0x9a, 0xd8, 0x0b, 0xaa,
		     0xb0, 0xfd, 0xd2, 0x67, 0x15, 0x40, 0xa5, 0x9b, 0x2f,
		     0xc7, 0x89, 0xfd, 0x0b, 0xe4, 0x15, 0x43, 0x57, 0xdf,
		     0x8d, 0x7f, 0x99, 0x6f, 0x20, 0x76, 0xe9, 0x63, 0xfa,
		     0x59, 0xd5, 0xff, 0xd9, 0xfe, 0x8a, 0x00, 0x6c, 0xe0,
		     0x9c, 0x05, 0x2e, 0xbe, 0xbc, 0xc6, 0xdb, 0x71, 0xb7,
		     0x77, 0x8f, 0xb0, 0xd3, 0x03, 0x0f, 0x18, 0xad, 0x26,
		     0x04, 0xd3, 0x15, 0x2a, 0x20, 0x7f, 0x66, 0x25, 0xa6,
		     0x31, 0x21, 0xbd, 0x02, 0x17, 0xa4, 0x42, 0x48, 0x56,
		     0xd2, 0x16, 0x7a, 0xf8, 0xfe, 0xa3, 0xe7, 0x7d, 0x20,
		     0xeb, 0xe1, 0x61, 0x24, 0xd0, 0xe7, 0x4f, 0x95, 0xfb,
		     0x8b, 0x5e, 0xd6, 0x19, 0x44, 0x71, 0x52, 0xe4, 0x88,
		     0x3d, 0x00, 0xfc, 0x2e, 0x3d, 0x14, 0xf2, 0x6a, 0x72,
		     0x4f, 0xde, 0x7e, 0x87, 0x0f, 0x19, 0x6e, 0xc5, 0xa0,
		     0x40, 0xbb, 0xba, 0xf4, 0xda, 0x12, 0x03, 0x4e, 0x5c,
		     0xbf, 0x94, 0x62, 0x4e, 0x2d, 0xc2, 0x80, 0xb1, 0xd4,
		     0x44, 0x57, 0xc8, 0x82, 0x0d, 0x3c, 0x48, 0x5a, 0x80,
		     0x23, 0xd6, 0xbb, 0x44 ),
	   PRIVATE ( 0xc7, 0xb3, 0x46, 0x82, 0x6d, 0xd8, 0xd9, 0x40, 0x6d,
		     0xda, 0xbf, 0x92, 0x6e, 0x71, 0xee, 0xf1, 0xec, 0xe5,
		     0xbb, 0x78, 0xd4, 0x32, 0x12, 0xc5, 0xa0, 0x38, 0xd8,
		     0x35 ),
	   PUBLIC ( 0x78, 0x04, 0x58, 0x69, 0x39, 0x2d, 0x56, 0xf3, 0x52,
		    0x07, 0x12, 0xa6, 0x8f, 0x29, 0x46, 0x6d, 0x8a, 0x89,
		    0xcf, 0x41, 0x91, 0x92, 0x50, 0x4c, 0x45, 0x3c, 0x22,
		    0x4d, 0xbc, 0x9a, 0x0f, 0xfd, 0x29, 0x7d, 0x6c, 0xc1,
		    0xa3, 0x70, 0xee, 0xe9, 0x39, 0x81, 0x39, 0x9d, 0xa4,
		    0xc7, 0x08, 0x97, 0xaa, 0xb4, 0x8f, 0x33, 0x4f, 0x05,
		    0xa5, 0x56, 0x27, 0x33, 0xe4, 0xe7, 0x31, 0xc3, 0x7f,
		    0x17, 0x38, 0x80, 0x76, 0x00, 0x88, 0xed, 0xb1, 0xe9,
		    0x98, 0x6a, 0x5d, 0x43, 0x08, 0x06, 0xd5, 0x14, 0x64,
		    0x24, 0xd9, 0x3a, 0x7f, 0xa4, 0xa3, 0x91, 0x65, 0x9f,
		    0xf9, 0x66, 0x67, 0x55, 0xe7, 0x5a, 0x14, 0x38, 0x81,
		    0x61, 0x13, 0xe1, 0x44, 0x8e, 0x6e, 0x14, 0xb4, 0x6c,
		    0xe8, 0xad, 0x79, 0x08, 0xfe, 0xc3, 0xb8, 0xe5, 0x02,
		    0x25, 0x72, 0x63, 0xea, 0xba, 0xfe, 0xfc, 0x9a, 0x3b,
		    0x9c, 0x64, 0x52, 0x21, 0x50, 0xfb, 0xc2, 0x11, 0xf4,
		    0x5e, 0xef, 0x43, 0x35, 0xfb, 0xc6, 0xdc, 0x01, 0xa9,
		    0x15, 0x69, 0x43, 0xab, 0xae, 0x05, 0xc3, 0xb1, 0x77,
		    0xac, 0x9d, 0x7e, 0x3b, 0xd3, 0xc5, 0x7e, 0xf5, 0x8d,
		    0xf9, 0x52, 0x3b, 0xcf, 0xc5, 0x53, 0x2d, 0x67, 0xac,
		    0x61, 0x17, 0x4f, 0x6c, 0x9c, 0x09, 0xa9, 0x38, 0x92,
		    0xdf, 0xbf, 0x49, 0x0d, 0x15, 0x0b, 0x02, 0xf2, 0x24,
		    0x38, 0x56, 0x19, 0xcf, 0x6d, 0xb9, 0x0a, 0x6f, 0x79,
		    0x04, 0x2e, 0xf9, 0xef, 0xdb, 0xbc, 0xdb, 0xf2, 0xa3,
		    0x8b, 0x01, 0x12, 0xec, 0x40, 0xdc, 0x6b, 0xff, 0x2b,
		    0xc7, 0xf5, 0x96, 0x41, 0x78, 0x40, 0xe3, 0xac, 0x4a,
		    0xa5, 0xa5, 0xd0, 0x44, 0xe7, 0x1a, 0x87, 0x6a, 0x10,
		    0xa2, 0x04, 0xdf, 0x71, 0x34, 0x47, 0xf2, 0x92, 0x0d,
		    0x92, 0x18, 0x0e, 0x14, 0x43, 0x18, 0xce, 0x0e, 0x18,
		    0xb8, 0x7e, 0xef, 0x6e ),
	   SHARED ( 0x0d, 0xf1, 0xd1, 0xb0, 0xfa, 0xae, 0x8b, 0x8a, 0xfb,
		    0x7c, 0x47, 0x88, 0x48, 0x49, 0xf2, 0x3a, 0x2d, 0x8f,
		    0x01, 0xee, 0x47, 0x14, 0x1e, 0x91, 0xd5, 0x49, 0x49,
		    0xff, 0x7f, 0xd0, 0xd1, 0x10, 0xac, 0x4c, 0xcc, 0xc6,
		    0x7f, 0x42, 0x8a, 0x04, 0xee, 0x64, 0x41, 0xf8, 0x1d,
		    0x1a, 0x04, 0x26, 0x3b, 0x99, 0xb5, 0xee, 0xf6, 0x17,
		    0x94, 0xfd, 0x3d, 0x58, 0x4b, 0xd7, 0xe1, 0xf4, 0xf6,
		    0x10, 0xb8, 0xdc, 0xb7, 0x8d, 0x04, 0x5f, 0x72, 0x13,
		    0x19, 0xf6, 0xa8, 0x33, 0x3e, 0x82, 0x8c, 0x56, 0xb1,
		    0x97, 0x5c, 0x4f, 0xa3, 0xd3, 0x1e, 0xeb, 0x61, 0xa4,
		    0xc2, 0x04, 0x2c, 0xc9, 0x22, 0x6c, 0x6e, 0xee, 0xa7,
		    0x5b, 0x66, 0x8b, 0xfb, 0xb5, 0x0d, 0x1b, 0x7c, 0xa1,
		    0x88, 0xc7, 0x95, 0x85, 0xa4, 0x4c, 0xe5, 0x38, 0x04,
		    0x1e, 0x94, 0x1b, 0x03, 0xe6, 0xcc, 0xa4, 0x36, 0x58,
		    0x02, 0xbe, 0x79, 0x89, 0x5a, 0xd2, 0x60, 0x1e, 0x47,
		    0xe6, 0x26, 0x35, 0x15, 0x3a, 0x91, 0xef, 0x92, 0xa9,
		    0x0f, 0x89, 0x58, 0xd3, 0xf7, 0xd2, 0xc5, 0x03, 0x61,
		    0x37, 0x78, 0x51, 0x1e, 0x39, 0x4d, 0x92, 0x8f, 0xdd,
		    0xb0, 0x7d, 0xc7, 0xad, 0xf4, 0x34, 0x85, 0x4e, 0x94,
		    0x85, 0x19, 0xf1, 0xbf, 0xb7, 0xca, 0x02, 0x3b, 0xb0,
		    0x46, 0x45, 0x96, 0xb6, 0x01, 0x0e, 0x38, 0x0d, 0xa5,
		    0xd7, 0x13, 0x5c, 0x5a, 0x0d, 0x41, 0xc5, 0x6d, 0x37,
		    0x92, 0xf6, 0xdc, 0x5e, 0x09, 0xa4, 0x2f, 0x7e, 0x2a,
		    0x94, 0x86, 0xd8, 0x59, 0x0b, 0x01, 0x54, 0x2a, 0xe6,
		    0x9f, 0xa1, 0x4f, 0xa5, 0x82, 0xca, 0x73, 0x44, 0x43,
		    0x47, 0x05, 0x64, 0x9d, 0x68, 0x78, 0x85, 0xcc, 0x24,
		    0x77, 0xaa, 0x4c, 0x08, 0x8d, 0x47, 0x33, 0x95, 0x48,
		    0x92, 0x6b, 0x9f, 0x7a, 0x17, 0x13, 0x82, 0x67, 0xf3,
		    0xe4, 0x55, 0x89, 0xdb ) );

/* KASValidityTest_FFCEphem_NOKC_ZZOnly_init.fax test 0 */
DHE_TEST ( kasvaliditytest_ffcephem_nokc_zzonly_init_fc_0,
	   MODULUS ( 0xc7, 0x37, 0xde, 0x69, 0x33, 0x3e, 0xb6, 0xd7, 0xe8,
		     0x3f, 0x8c, 0xfc, 0x69, 0xbb, 0x39, 0xcf, 0x2d, 0xd1,
		     0xe6, 0xe3, 0xd0, 0x03, 0x9a, 0x3b, 0x25, 0x53, 0x39,
		     0x4b, 0xbb, 0xe2, 0x7a, 0x25, 0xa3, 0x5b, 0x76, 0xf4,
		     0xd5, 0x6b, 0xb6, 0x95, 0xf3, 0x6e, 0x87, 0x1d, 0x68,
		     0x0f, 0x62, 0xfc, 0xce, 0xef, 0x57, 0x0f, 0xd1, 0x24,
		     0xfe, 0x55, 0xe1, 0xfe, 0xfa, 0x70, 0xb4, 0x7a, 0x38,
		     0x8f, 0x26, 0x4a, 0xa5, 0x28, 0x2d, 0x11, 0x51, 0x62,
		     0x17, 0x4f, 0x27, 0xe1, 0x80, 0x3b, 0xf4, 0x2b, 0xde,
		     0x54, 0x91, 0xe5, 0x8c, 0xfe, 0x15, 0x0e, 0x76, 0xbf,
		     0xbd, 0x83, 0x62, 0xdf, 0xc4, 0xc1, 0x22, 0x39, 0x65,
		     0x7d, 0x7c, 0x65, 0x74, 0xb1, 0x81, 0xb1, 0xfc, 0xf8,
		     0x8c, 0xc0, 0xc8, 0x91, 0xec, 0xd9, 0xcc, 0x43, 0xb4,
		     0x10, 0xa6, 0x50, 0xfd, 0xaf, 0x26, 0xb8, 0x01, 0xdc,
		     0x88, 0xa1, 0x3d, 0x67, 0x27, 0xe0, 0x3d, 0x9a, 0x18,
		     0xf8, 0x9b, 0x9c, 0xb3, 0xee, 0x80, 0xcc, 0x2d, 0x1e,
		     0xb5, 0x33, 0xf4, 0xcc, 0xce, 0x09, 0x4f, 0xb3, 0xe2,
		     0x33, 0xde, 0xd0, 0x79, 0x10, 0x5d, 0x1f, 0xbc, 0x21,
		     0xdc, 0xa3, 0xf8, 0x72, 0x97, 0xc3, 0x7a, 0xe5, 0x37,
		     0xb8, 0x88, 0x71, 0xce, 0xdc, 0xf5, 0x65, 0x89, 0xf1,
		     0x13, 0xc9, 0xb3, 0x2a, 0xcf, 0xb5, 0xb8, 0xd4, 0x2d,
		     0xf6, 0x03, 0x27, 0x87, 0x14, 0xd8, 0xa6, 0xbe, 0x1c,
		     0x7e, 0x74, 0x27, 0x04, 0xf1, 0x27, 0x79, 0xf1, 0x08,
		     0x92, 0xca, 0xf0, 0xef, 0x40, 0x4f, 0x06, 0xa0, 0x8c,
		     0xff, 0x11, 0x02, 0x97, 0xac, 0x0d, 0x2a, 0x99, 0x93,
		     0x6c, 0x15, 0xfa, 0x30, 0xed, 0x7d, 0xee, 0x19, 0xc5,
		     0xbb, 0x5c, 0x6d, 0x55, 0x54, 0x2e, 0xa0, 0xb9, 0x4f,
		     0xed, 0xe0, 0x32, 0x73, 0xf6, 0x2a, 0x11, 0xc6, 0xd6,
		     0xe4, 0x27, 0x9c, 0x19 ),
	   GENERATOR ( 0x3a, 0xb1, 0xe0, 0x51, 0x9b, 0xcc, 0xc1, 0xa4, 0x6c,
		       0xb9, 0x12, 0x4c, 0xb1, 0x1a, 0xcf, 0x3b, 0x42, 0x69,
		       0x1a, 0x7f, 0x40, 0x83, 0x10, 0x5f, 0xd9, 0xac, 0xe0,
		       0x13, 0x54, 0xec, 0xcb, 0xed, 0x1d, 0x05, 0xb6, 0x4b,
		       0x14, 0x94, 0x91, 0xd1, 0x0a, 0x92, 0xdb, 0x1f, 0x56,
		       0xea, 0x9d, 0x9b, 0xe2, 0xba, 0x60, 0x31, 0x0c, 0x1e,
		       0xe3, 0xf6, 0xa2, 0x7c, 0x20, 0x19, 0x36, 0x32, 0xff,
		       0x4f, 0x8e, 0x0e, 0x79, 0xe8, 0xd1, 0x7d, 0x27, 0x51,
		       0xa5, 0x78, 0x08, 0x60, 0xd1, 0x52, 0x53, 0x75, 0x49,
		       0x42, 0x5c, 0x69, 0x6e, 0x66, 0x26, 0xaa, 0x5e, 0x79,
		       0x42, 0x76, 0xf1, 0x86, 0xfd, 0xcc, 0x04, 0x5b, 0x30,
		       0x55, 0xec, 0x5b, 0x40, 0xf4, 0xe3, 0xbf, 0xf9, 0x8a,
		       0x2b, 0x1a, 0x54, 0x5b, 0x80, 0x19, 0xc4, 0xa1, 0x36,
		       0x0a, 0x49, 0x0e, 0xb9, 0xc6, 0xd7, 0x7e, 0xfb, 0x24,
		       0x84, 0x26, 0x10, 0xb2, 0xfc, 0xcd, 0x9b, 0xd9, 0x53,
		       0xc9, 0xd7, 0x0f, 0x6d, 0xf4, 0xcf, 0xdc, 0xe7, 0xb0,
		       0x1a, 0x4d, 0x21, 0x4c, 0x99, 0xb1, 0x1c, 0x5d, 0x28,
		       0x09, 0x96, 0xd5, 0x3e, 0xef, 0x09, 0x3b, 0xd5, 0x29,
		       0xa5, 0x2d, 0xcc, 0xdb, 0x9c, 0xaf, 0xd9, 0x6f, 0xa7,
		       0xc9, 0x28, 0x8b, 0x14, 0xf4, 0x8e, 0x0b, 0x4e, 0x35,
		       0xdf, 0x57, 0x22, 0xb2, 0x8a, 0xa0, 0x05, 0xa8, 0x84,
		       0xb4, 0x57, 0xcb, 0x12, 0x07, 0x8f, 0x9e, 0x83, 0xb3,
		       0xbe, 0xac, 0xb0, 0x3e, 0xf2, 0x47, 0x6b, 0x22, 0x63,
		       0xd2, 0x03, 0xa3, 0x3b, 0xc5, 0x17, 0x05, 0xe7, 0xe8,
		       0xe6, 0x18, 0xd3, 0x41, 0x94, 0xc7, 0x16, 0xde, 0x38,
		       0x0c, 0xb1, 0xb7, 0xeb, 0xe5, 0x55, 0xf2, 0x2e, 0xa6,
		       0xe5, 0x61, 0x75, 0xd0, 0x7d, 0x6c, 0xbe, 0xd1, 0xc9,
		       0x4c, 0x59, 0xfb, 0x37, 0xb7, 0xdc, 0xd5, 0x6f, 0xf7,
		       0xb2, 0xf2, 0xf9, 0x97 ),
	   PARTNER ( 0x65, 0x6c, 0x5d, 0x99, 0xbc, 0xb5, 0x92, 0x1f, 0xc2,
		     0xf7, 0x86, 0x99, 0xd9, 0x8e, 0xe0, 0x46, 0xbe, 0x70,
		     0x76, 0x1a, 0x85, 0x7a, 0x9d, 0xfe, 0xad, 0x51, 0x7b,
		     0xee, 0x16, 0xe8, 0xae, 0xf9, 0x19, 0x26, 0x04, 0xa8,
		     0x29, 0x5e, 0x02, 0xef, 0xd4, 0xfe, 0xf2, 0x43, 0x64,
		     0xd0, 0x53, 0x4f, 0xa2, 0x7c, 0x41, 0xef, 0x85, 0x0d,
		     0xe2, 0xee, 0xb7, 0x4f, 0x03, 0xb4, 0x7d, 0x59, 0x0b,
		     0x8d, 0x32, 0x4c, 0x9b, 0xf9, 0x48, 0x53, 0x7b, 0xa9,
		     0xb7, 0x32, 0x1d, 0x1e, 0x51, 0xf2, 0x69, 0x66, 0x44,
		     0xf3, 0x9c, 0x78, 0xa9, 0xa7, 0x4a, 0x93, 0x90, 0xb0,
		     0xc3, 0x79, 0x36, 0xdd, 0xda, 0x92, 0xfe, 0x71, 0x60,
		     0x21, 0xf6, 0xea, 0xb6, 0xa9, 0xbe, 0x9f, 0x42, 0x12,
		     0xf7, 0x8a, 0x51, 0x2e, 0x39, 0x43, 0x50, 0xbb, 0xe4,
		     0xbf, 0x6a, 0x5f, 0x8c, 0x97, 0x97, 0x74, 0x67, 0xf6,
		     0xb1, 0xd9, 0x2a, 0xc9, 0x88, 0xc8, 0x1c, 0xe2, 0x48,
		     0x81, 0xba, 0x59, 0x5d, 0x5f, 0x62, 0x8b, 0x25, 0x24,
		     0x57, 0x9e, 0x47, 0xc9, 0xdc, 0x88, 0x6a, 0x32, 0x4a,
		     0x6c, 0x37, 0xd7, 0x73, 0x38, 0x0b, 0x6a, 0xf6, 0x45,
		     0xac, 0x38, 0x49, 0xd0, 0xc7, 0x8d, 0x1d, 0x9e, 0x3a,
		     0x4c, 0xce, 0xba, 0x0a, 0xc9, 0x54, 0x6b, 0x9d, 0x9c,
		     0xb7, 0x2b, 0xf4, 0x7a, 0x00, 0xd3, 0xd2, 0xfe, 0x79,
		     0x8d, 0xb4, 0x6e, 0xe9, 0x38, 0x95, 0xaa, 0x21, 0x2f,
		     0x71, 0xe1, 0xd4, 0xa8, 0x21, 0x4a, 0x87, 0x4f, 0xcd,
		     0xa6, 0xca, 0x60, 0x4d, 0xc1, 0xa2, 0x28, 0xcf, 0xa9,
		     0x55, 0x14, 0xca, 0x24, 0xa9, 0x1b, 0x99, 0xbe, 0x98,
		     0x4c, 0x7d, 0xff, 0x5e, 0x05, 0xf9, 0x1e, 0x56, 0x65,
		     0x00, 0x22, 0x65, 0xfb, 0x72, 0x30, 0x3f, 0xae, 0xc0,
		     0xbb, 0x67, 0xcc, 0xe2, 0x51, 0xa7, 0x06, 0x59, 0x86,
		     0x1d, 0x43, 0xbb, 0x35 ),
	   PRIVATE ( 0x83, 0x27, 0x46, 0xed, 0x42, 0x07, 0xdf, 0xff, 0x0a,
		     0x89, 0x19, 0xae, 0x6a, 0xb5, 0x29, 0x2c, 0x37, 0xcb,
		     0x8b, 0x26, 0xd4, 0x06, 0x27, 0xee, 0xb2, 0xf6, 0x36,
		     0x46, 0x08, 0xc8, 0x34, 0x54 ),
	   PUBLIC ( 0x88, 0x9d, 0xa3, 0xae, 0x17, 0x79, 0xf8, 0xb6, 0x4e,
		    0xc4, 0x56, 0xa8, 0x8d, 0x7a, 0x10, 0x5b, 0xdd, 0x7a,
		    0xaa, 0x87, 0x10, 0xa3, 0x3c, 0xcc, 0x7d, 0x80, 0x59,
		    0x43, 0x22, 0x91, 0xfc, 0xbf, 0x6d, 0x63, 0x5e, 0x13,
		    0xc2, 0x8d, 0x19, 0x94, 0x84, 0xbc, 0x56, 0x45, 0x31,
		    0x71, 0x75, 0x2c, 0x81, 0x18, 0x6e, 0x10, 0x1d, 0x2b,
		    0x3d, 0x2c, 0xf9, 0x2e, 0x93, 0x6b, 0xed, 0xfa, 0x17,
		    0xd2, 0x95, 0x30, 0xca, 0xc8, 0x26, 0x0c, 0x78, 0x8f,
		    0x75, 0x87, 0x0e, 0x30, 0x1e, 0x6c, 0x5e, 0xae, 0xc7,
		    0x99, 0x37, 0xb0, 0x02, 0x00, 0xda, 0xc1, 0x36, 0x24,
		    0x12, 0xdd, 0xe4, 0x2b, 0x8a, 0xca, 0xf9, 0x35, 0x36,
		    0x4b, 0x6d, 0xff, 0xa2, 0x51, 0xd1, 0xe6, 0xa0, 0x78,
		    0xf3, 0x0f, 0xc3, 0x9e, 0x4e, 0x1c, 0xc6, 0xf6, 0x38,
		    0x50, 0x07, 0x18, 0x42, 0x04, 0x8c, 0xe7, 0xdf, 0xeb,
		    0x2a, 0xda, 0x96, 0x28, 0xae, 0xe1, 0x7d, 0x55, 0x96,
		    0xb0, 0x5a, 0x4e, 0x71, 0xa9, 0xfa, 0x6f, 0x0e, 0x0d,
		    0xff, 0xea, 0xe0, 0xca, 0xa2, 0xf9, 0x5a, 0x68, 0x3b,
		    0x3b, 0x29, 0x7e, 0x15, 0x6b, 0x90, 0x95, 0x1c, 0x80,
		    0xf9, 0x08, 0x03, 0xa8, 0xa5, 0x19, 0x9b, 0xe9, 0xde,
		    0xb4, 0xf0, 0x18, 0xb4, 0xcd, 0x3c, 0xaf, 0x0b, 0x4d,
		    0x3c, 0xe5, 0xd6, 0x58, 0xe3, 0x0f, 0xe2, 0x72, 0xcd,
		    0xe9, 0xd2, 0xc2, 0x57, 0x9a, 0x5e, 0x3f, 0xbd, 0xe2,
		    0xaa, 0x3a, 0xd4, 0xe0, 0xae, 0x05, 0x9c, 0xfc, 0x69,
		    0xda, 0x62, 0xd0, 0x6f, 0x70, 0x4f, 0xeb, 0x77, 0x09,
		    0x1f, 0xac, 0x6f, 0xde, 0x3a, 0xe3, 0xbb, 0x6b, 0x3d,
		    0xae, 0xec, 0x9d, 0xd6, 0xbc, 0xf5, 0xb1, 0x69, 0x70,
		    0xf2, 0x6e, 0xc8, 0xde, 0x3c, 0xe7, 0x9b, 0x8f, 0x37,
		    0xe2, 0x7e, 0xe0, 0x4a, 0x28, 0x0f, 0x28, 0xac, 0xe8,
		    0x63, 0xf2, 0xd3, 0x13 ),
	   SHARED ( 0x26, 0x2d, 0x59, 0xd0, 0x4c, 0x02, 0x1b, 0x84, 0xc1,
		    0x16, 0x96, 0xc9, 0x9f, 0xbc, 0x03, 0x50, 0x5b, 0xe5,
		    0x6b, 0x76, 0x0d, 0x63, 0xf1, 0x0b, 0xa5, 0x30, 0x9c,
		    0x58, 0x60, 0x96, 0x9d, 0xf6, 0xc1, 0x3c, 0x61, 0xb3,
		    0x9a, 0x21, 0x37, 0x4c, 0xb3, 0x5e, 0x17, 0x72, 0x60,
		    0x50, 0x8a, 0xbb, 0xe0, 0x02, 0xf3, 0x7d, 0x15, 0x67,
		    0x76, 0x97, 0x39, 0x85, 0xe4, 0x66, 0x50, 0x91, 0x31,
		    0xbc, 0x45, 0xef, 0x0e, 0x32, 0xb9, 0x8b, 0xdb, 0x6c,
		    0x18, 0x72, 0x06, 0xde, 0x7b, 0xda, 0xc4, 0xaf, 0x5e,
		    0x39, 0x47, 0x85, 0x0e, 0xea, 0x89, 0x7a, 0xd1, 0xe3,
		    0xa7, 0x60, 0x73, 0x86, 0xd4, 0xbe, 0x8b, 0xae, 0x01,
		    0xf3, 0x3d, 0x28, 0xfd, 0xd7, 0x6a, 0xbb, 0x62, 0x80,
		    0xc6, 0x7b, 0xaa, 0xf3, 0x1e, 0xff, 0x52, 0xf2, 0x87,
		    0x4c, 0x8b, 0x0e, 0x2f, 0x3c, 0x38, 0x24, 0x13, 0x3a,
		    0xad, 0x4a, 0x4f, 0x87, 0x0a, 0x81, 0x8b, 0xdf, 0x69,
		    0x14, 0xd7, 0xe0, 0xbe, 0x1e, 0x61, 0x9f, 0x62, 0xa8,
		    0x47, 0x78, 0x66, 0x9b, 0xef, 0x83, 0xbf, 0x5d, 0xd9,
		    0xdf, 0x5f, 0x29, 0x23, 0x9a, 0x57, 0xc6, 0x35, 0x2e,
		    0x70, 0x9e, 0x30, 0x05, 0x83, 0x29, 0xd8, 0xdd, 0xc1,
		    0xe6, 0x9b, 0x08, 0x4b, 0x49, 0x53, 0x60, 0xde, 0xe0,
		    0xda, 0x83, 0xf4, 0x58, 0xf6, 0x53, 0x49, 0x1b, 0x9b,
		    0x77, 0x70, 0x16, 0xe4, 0xd9, 0x14, 0x72, 0xd7, 0xca,
		    0xf6, 0x84, 0xd9, 0x79, 0xdf, 0xfa, 0xaa, 0xa6, 0xeb,
		    0xa1, 0x5e, 0x63, 0x35, 0x08, 0x8b, 0xff, 0x15, 0x1b,
		    0x39, 0xde, 0x8a, 0x20, 0x3d, 0xb8, 0xff, 0x29, 0x03,
		    0x4c, 0x98, 0xcc, 0xcb, 0xcd, 0x7e, 0x0b, 0x3b, 0x77,
		    0x5d, 0xa4, 0x77, 0x1a, 0xfd, 0xb5, 0x5c, 0x5d, 0x3f,
		    0x80, 0xd0, 0xe2, 0xb3, 0x00, 0xdf, 0x14, 0x26, 0x77,
		    0x81, 0x95, 0xbf, 0x64 ) );

/* KASValidityTest_FFCEphem_NOKC_ZZOnly_resp.fax test 0 */
DHE_TEST ( kasvaliditytest_ffcephem_nokc_zzonly_resp_fb_0,
	   MODULUS ( 0x92, 0xac, 0x82, 0xb2, 0x88, 0x5b, 0x17, 0x4d, 0x24,
		     0x3d, 0x1e, 0x7d, 0x5b, 0xed, 0x4b, 0x6b, 0xac, 0x61,
		     0xbd, 0x71, 0x32, 0x30, 0x1b, 0xe5, 0xf1, 0xe6, 0x69,
		     0x43, 0x6d, 0x40, 0x92, 0x95, 0xee, 0x9c, 0x70, 0x05,
		     0xe5, 0xa1, 0x6f, 0x84, 0x78, 0xfb, 0x6f, 0xd2, 0x14,
		     0xa1, 0x96, 0xc9, 0x8e, 0xf9, 0xe5, 0x91, 0xe4, 0x5c,
		     0xd9, 0x33, 0x17, 0x70, 0x3a, 0x2a, 0xec, 0x5b, 0x10,
		     0x6e, 0xd7, 0xcc, 0x4a, 0x98, 0x7b, 0xa1, 0xae, 0xf7,
		     0xf6, 0x21, 0x34, 0x8f, 0x00, 0x2b, 0xa9, 0x87, 0x55,
		     0xe0, 0x10, 0xf0, 0x32, 0x7c, 0xb5, 0x38, 0xec, 0x9b,
		     0x1a, 0xec, 0x01, 0xdf, 0xaf, 0x26, 0xf1, 0xac, 0x33,
		     0xb9, 0x17, 0xa9, 0x89, 0xbc, 0x73, 0x98, 0xf1, 0x23,
		     0x4c, 0x25, 0x52, 0xdc, 0xda, 0xc6, 0x5e, 0x4c, 0xab,
		     0xc3, 0x7b, 0x5b, 0x24, 0x91, 0x7d, 0xc1, 0x66, 0x46,
		     0x4b, 0x7a, 0xff, 0x26, 0x89, 0x13, 0x88, 0x79, 0xcf,
		     0xfa, 0x2a, 0x9e, 0x7c, 0x11, 0xab, 0xff, 0xf1, 0x19,
		     0x17, 0x77, 0x9b, 0xad, 0x99, 0xfd, 0x31, 0x69, 0x44,
		     0x7e, 0xf5, 0x08, 0x7f, 0xb6, 0x1e, 0x1a, 0x92, 0x95,
		     0x55, 0x2d, 0xb7, 0x61, 0xe0, 0x67, 0x87, 0xfa, 0x44,
		     0xce, 0x83, 0xdf, 0xd5, 0x22, 0x9f, 0x0b, 0xd1, 0x2a,
		     0x86, 0x73, 0xd0, 0xfc, 0x90, 0x55, 0xbe, 0x05, 0x94,
		     0x89, 0x8b, 0x5b, 0x03, 0x10, 0xfd, 0xc5, 0x0b, 0xa1,
		     0x3f, 0x79, 0x00, 0xd5, 0xff, 0xf1, 0xe8, 0x1f, 0xbc,
		     0x90, 0x42, 0xbf, 0x20, 0x4d, 0x5f, 0x0a, 0xaa, 0x69,
		     0x66, 0xf7, 0x25, 0xa4, 0xb6, 0xc4, 0x84, 0x8a, 0x8d,
		     0x81, 0x68, 0xe3, 0x38, 0x9d, 0x7d, 0xfe, 0xca, 0x6a,
		     0xcd, 0x89, 0xde, 0x55, 0xf4, 0x2d, 0x91, 0x1b, 0x74,
		     0x9b, 0xb5, 0xb8, 0x2d, 0x61, 0xef, 0x46, 0x32, 0x16,
		     0xd6, 0x69, 0xac, 0x73 ),
	   GENERATOR ( 0x2e, 0xf4, 0x37, 0xfb, 0x4f, 0x8d, 0xed, 0x56, 0x1d,
		       0xb4, 0xfe, 0x58, 0x81, 0x53, 0xa3, 0xbe, 0x42, 0x58,
		       0x72, 0x70, 0xa0, 0xec, 0xf6, 0x42, 0x05, 0x01, 0x87,
		       0x59, 0xeb, 0x0b, 0xc0, 0x60, 0xf6, 0x27, 0x77, 0x29,
		       0x7b, 0x3a, 0x99, 0xf4, 0x89, 0x53, 0xfd, 0xce, 0x34,
		       0xa0, 0xe9, 0x66, 0xdc, 0x9f, 0x79, 0xbc, 0x3b, 0xee,
		       0x43, 0xbe, 0xad, 0x12, 0xd5, 0x93, 0x72, 0x91, 0x30,
		       0xef, 0xf2, 0x2a, 0xd7, 0xe8, 0x35, 0x14, 0x79, 0x8f,
		       0xbf, 0xfc, 0x9c, 0xcc, 0xb3, 0x4a, 0x87, 0x93, 0xaa,
		       0x95, 0xaf, 0xd2, 0x68, 0x62, 0x47, 0xb5, 0xc1, 0x78,
		       0x02, 0xc1, 0xd6, 0xe6, 0x46, 0x33, 0x7b, 0xad, 0x88,
		       0x11, 0x8b, 0x74, 0x07, 0x44, 0x71, 0x3a, 0xd8, 0x4b,
		       0xf4, 0x82, 0xa6, 0x50, 0x36, 0x4c, 0xf0, 0xc4, 0x4b,
		       0xe2, 0xe6, 0x42, 0x42, 0xf2, 0x98, 0x98, 0xaf, 0x72,
		       0x2f, 0xec, 0xf3, 0x9a, 0x49, 0xab, 0xa0, 0xbe, 0xfc,
		       0x3d, 0xb6, 0x8c, 0xc7, 0x89, 0xf6, 0x07, 0x6f, 0xe9,
		       0x00, 0x85, 0x6c, 0x01, 0xb1, 0xfd, 0x1e, 0x70, 0xce,
		       0x0e, 0xce, 0xae, 0xe1, 0x71, 0x7f, 0xff, 0x94, 0x8a,
		       0xd9, 0xd6, 0x6d, 0x76, 0x28, 0x0a, 0x5f, 0x3e, 0x64,
		       0xaa, 0x4a, 0x37, 0x39, 0x51, 0xe8, 0x9d, 0xd1, 0x78,
		       0x53, 0xe0, 0x81, 0xb6, 0xa8, 0x90, 0xdd, 0x79, 0xb1,
		       0x47, 0x81, 0x75, 0x34, 0xf4, 0x6f, 0xc3, 0x8e, 0x7d,
		       0xa7, 0x29, 0xc6, 0xd4, 0x5f, 0x5e, 0x25, 0x4f, 0xa1,
		       0xe1, 0x90, 0xb3, 0x85, 0xfb, 0xe9, 0xf1, 0x60, 0x65,
		       0x4a, 0x14, 0xe5, 0xb3, 0xc2, 0x83, 0x66, 0x7f, 0x5f,
		       0x2e, 0x0a, 0xd1, 0x44, 0xbf, 0x10, 0xbf, 0xd3, 0xf9,
		       0x12, 0x37, 0x0e, 0x81, 0x24, 0x35, 0x49, 0xac, 0xd4,
		       0x48, 0x3d, 0xaa, 0x43, 0xf1, 0x4a, 0xbc, 0x08, 0xfc,
		       0xaf, 0xa0, 0xbe, 0x29 ),
	   PARTNER ( 0x4d, 0x7c, 0xea, 0x09, 0x9b, 0x64, 0xdb, 0x3d, 0xdc,
		     0x8d, 0x3f, 0x8b, 0xd8, 0xb1, 0xcb, 0xea, 0xd7, 0xe0,
		     0x71, 0xe4, 0x1c, 0x58, 0xcf, 0x08, 0xda, 0x3d, 0xa1,
		     0x8c, 0x0b, 0x7c, 0xac, 0x6f, 0x76, 0xa4, 0xd8, 0xa9,
		     0x91, 0xec, 0x9e, 0x74, 0x34, 0x52, 0x02, 0x43, 0x55,
		     0xa9, 0xa1, 0xf6, 0xd5, 0xc8, 0x67, 0xad, 0x4d, 0x98,
		     0xeb, 0x8e, 0xf4, 0x72, 0x95, 0x20, 0xd4, 0x7b, 0x38,
		     0x4a, 0xa3, 0x8b, 0x21, 0x9b, 0x85, 0x4d, 0x05, 0xf1,
		     0x94, 0x89, 0xa3, 0x34, 0x9d, 0x9e, 0xaa, 0x23, 0x97,
		     0xac, 0x7e, 0xb2, 0xe8, 0x7f, 0x9e, 0x1e, 0x59, 0x5e,
		     0xe0, 0xea, 0xcf, 0xff, 0x86, 0x56, 0x80, 0x22, 0xc9,
		     0x03, 0x6c, 0x05, 0x8c, 0x19, 0x6d, 0xaa, 0x82, 0xdd,
		     0x47, 0xab, 0xbf, 0x24, 0xe6, 0xea, 0xb8, 0xc9, 0xc1,
		     0x9e, 0x64, 0xd1, 0x7b, 0xee, 0x9b, 0xed, 0x99, 0xd2,
		     0x90, 0x3a, 0x4b, 0x90, 0xc3, 0xbb, 0xf5, 0xa4, 0x49,
		     0xb5, 0xfc, 0xa8, 0x6e, 0x07, 0x9a, 0x27, 0xfb, 0x0f,
		     0x67, 0xa1, 0x14, 0x0e, 0x30, 0x80, 0x4e, 0x60, 0xdf,
		     0x01, 0x62, 0x25, 0x8f, 0xb7, 0x18, 0x14, 0x56, 0x5e,
		     0x76, 0x14, 0xe9, 0x15, 0x35, 0xdb, 0x52, 0xef, 0x34,
		     0x60, 0x22, 0x01, 0x3c, 0x75, 0x01, 0x27, 0x13, 0xe9,
		     0x0a, 0xa7, 0x67, 0x05, 0xb5, 0xec, 0x7a, 0x60, 0xdc,
		     0xb5, 0xdd, 0xdc, 0xc9, 0x9d, 0xbb, 0x10, 0x7b, 0x3d,
		     0xd9, 0x8f, 0x31, 0xff, 0x26, 0xce, 0x7c, 0x78, 0x24,
		     0xa8, 0xe7, 0xa5, 0x6d, 0xb4, 0xc2, 0x98, 0x66, 0xac,
		     0xac, 0x06, 0x4e, 0xa3, 0x85, 0x63, 0xd6, 0xf7, 0xa4,
		     0x53, 0x91, 0xe5, 0x16, 0x74, 0x14, 0xfe, 0x91, 0x61,
		     0x95, 0x1f, 0x30, 0xdb, 0xbc, 0x8e, 0xf1, 0xf3, 0x08,
		     0x8b, 0x0e, 0x8a, 0xc8, 0x6f, 0xe0, 0x5e, 0x48, 0x19,
		     0x6b, 0xe6, 0x70, 0x10 ),
	   PRIVATE ( 0x68, 0x22, 0xbd, 0xe0, 0x50, 0xfc, 0xdd, 0xf9, 0x0b,
		     0xa8, 0xc3, 0x98, 0xd6, 0xc9, 0x73, 0x55, 0x2b, 0xe0,
		     0xf8, 0xf4, 0xb4, 0x8c, 0x86, 0xdf, 0xdd, 0xa2, 0xc1,
		     0x87 ),
	   PUBLIC ( 0x1e, 0xab, 0x79, 0xb7, 0x83, 0xe8, 0x74, 0x90, 0xa8,
		    0xab, 0x90, 0x24, 0x02, 0xaa, 0xd7, 0x9d, 0x2c, 0x90,
		    0x35, 0x2a, 0xf3, 0x54, 0x26, 0x69, 0x1c, 0xac, 0xe0,
		    0x73, 0x8f, 0xc9, 0xa4, 0x18, 0xee, 0x4f, 0xfd, 0x9f,
		    0xd3, 0x0d, 0x04, 0x8b, 0xa4, 0x38, 0xa4, 0x87, 0xf6,
		    0x97, 0x5d, 0x5c, 0x20, 0x22, 0x53, 0xcb, 0xf2, 0xa8,
		    0x54, 0xff, 0x36, 0x28, 0xde, 0xf8, 0x1d, 0x92, 0x3a,
		    0x88, 0x20, 0x00, 0xa3, 0xd4, 0xdd, 0x4d, 0x8b, 0x16,
		    0x7a, 0xdc, 0xcd, 0x71, 0xe4, 0x2e, 0x61, 0xad, 0xae,
		    0x02, 0x4f, 0xa1, 0xf6, 0x69, 0x09, 0x9c, 0xd8, 0x0a,
		    0xb3, 0x13, 0xb2, 0xd8, 0x44, 0xea, 0xdf, 0xdf, 0x78,
		    0x6d, 0xfd, 0x5b, 0x6b, 0x5d, 0x94, 0xe5, 0x47, 0x79,
		    0xde, 0xc0, 0x1f, 0xf8, 0x63, 0x39, 0x4c, 0xdd, 0xd3,
		    0xe8, 0x5a, 0x91, 0xe7, 0x2d, 0xd6, 0xa4, 0x0b, 0x77,
		    0x44, 0xbc, 0xad, 0x37, 0x62, 0x5e, 0xb9, 0x88, 0x7e,
		    0x9a, 0x12, 0x5f, 0x7b, 0x61, 0x6c, 0xa7, 0x45, 0x9b,
		    0xae, 0x96, 0xd5, 0x29, 0x72, 0x53, 0xdf, 0x39, 0x8e,
		    0xd3, 0xfd, 0x0d, 0x1a, 0xf9, 0xf3, 0xbf, 0x66, 0x3a,
		    0x2c, 0xdf, 0x36, 0x03, 0xb0, 0x5d, 0x72, 0xe7, 0x27,
		    0x65, 0x8a, 0x40, 0x83, 0xfb, 0x49, 0x81, 0xea, 0xb3,
		    0x81, 0xcc, 0x99, 0x7a, 0x82, 0x19, 0xd4, 0x5e, 0xd6,
		    0xef, 0xa5, 0x5f, 0x64, 0x3d, 0x0c, 0x8e, 0xbe, 0x3a,
		    0x7c, 0x32, 0xd0, 0x5e, 0x90, 0x1e, 0xe2, 0x24, 0x40,
		    0x2d, 0xbf, 0xfc, 0x1e, 0xa1, 0xa2, 0x6b, 0xbc, 0xba,
		    0xdd, 0xc8, 0x01, 0xcf, 0x9f, 0x51, 0x19, 0xb8, 0x3d,
		    0x6e, 0xe9, 0x17, 0x74, 0x71, 0x2e, 0x56, 0x76, 0xf1,
		    0x67, 0x48, 0xe2, 0x82, 0xa8, 0x44, 0x39, 0x5b, 0x47,
		    0xe1, 0xc1, 0xee, 0xef, 0x41, 0x33, 0x1d, 0xdb, 0xcd,
		    0xe6, 0x5e, 0x47, 0x5e ),
	   SHARED ( 0x35, 0xd7, 0xbe, 0xe8, 0x45, 0x91, 0xa8, 0xaf, 0xe9,
		    0x29, 0x38, 0xaf, 0x83, 0x18, 0xe2, 0xb3, 0x50, 0xea,
		    0x55, 0x03, 0x80, 0xa7, 0x8f, 0x81, 0x70, 0xb8, 0x6d,
		    0x47, 0x0a, 0x2b, 0x69, 0x15, 0x4b, 0x6f, 0x96, 0xc9,
		    0x7e, 0x46, 0x6d, 0x68, 0x75, 0xf5, 0x8f, 0x41, 0x0f,
		    0xa2, 0x94, 0xa6, 0xa8, 0xcb, 0x55, 0x2d, 0x79, 0xf1,
		    0xa4, 0xaa, 0x01, 0xf3, 0x01, 0x9c, 0x56, 0x5e, 0xce,
		    0xf3, 0xe9, 0x2b, 0xbd, 0xa1, 0x8e, 0xe3, 0xf4, 0x76,
		    0x0e, 0x91, 0x95, 0xff, 0x75, 0x72, 0x3b, 0xe3, 0x0b,
		    0x86, 0x27, 0xe7, 0xd4, 0x46, 0x17, 0x4c, 0x13, 0x8f,
		    0x6c, 0x49, 0x1d, 0x1a, 0xeb, 0x12, 0x36, 0xe4, 0x68,
		    0xd0, 0xbe, 0xed, 0x08, 0x95, 0xad, 0x1a, 0x8d, 0xd6,
		    0x3f, 0xe9, 0xc5, 0xe5, 0xed, 0x5e, 0xed, 0xbd, 0x34,
		    0xb1, 0x67, 0x19, 0x03, 0x31, 0x9e, 0x3d, 0xbd, 0x1d,
		    0x6f, 0xac, 0x86, 0x2d, 0xff, 0x0a, 0xa5, 0x82, 0x8d,
		    0x4a, 0xb5, 0xc8, 0x28, 0x4b, 0xf7, 0x41, 0x35, 0xf4,
		    0x18, 0xd3, 0x9a, 0x92, 0x01, 0xd2, 0x48, 0x3f, 0xbb,
		    0x33, 0xb2, 0x83, 0x3f, 0xf0, 0x3e, 0x7c, 0xd1, 0xfb,
		    0x1f, 0xda, 0x78, 0x13, 0xaf, 0x84, 0x87, 0x42, 0x02,
		    0x95, 0xa8, 0xce, 0xd4, 0xc5, 0x8c, 0xd4, 0x82, 0xf3,
		    0x8d, 0xf0, 0x44, 0xcc, 0x3f, 0x88, 0xfa, 0x02, 0x32,
		    0x46, 0xd1, 0x0d, 0x22, 0x6c, 0x77, 0x33, 0x73, 0x68,
		    0xbb, 0x41, 0x0c, 0x9e, 0x66, 0x80, 0x75, 0x6d, 0x22,
		    0x47, 0x37, 0x06, 0xd2, 0xfd, 0xa1, 0xd1, 0x27, 0xcc,
		    0x47, 0x0c, 0x24, 0xa7, 0xe7, 0xb6, 0xce, 0x5b, 0x4e,
		    0x9c, 0xa6, 0x15, 0x96, 0xc3, 0xd1, 0x0a, 0xb0, 0x98,
		    0x38, 0x0a, 0xd6, 0x13, 0xc9, 0xf8, 0x1a, 0x18, 0xcf,
		    0xdf, 0x0c, 0x75, 0x78, 0xc1, 0x1c, 0x75, 0xb8, 0x86,
		    0x00, 0xb4, 0xca, 0x38 ) );

/* KASValidityTest_FFCEphem_NOKC_ZZOnly_resp.fax test 0 */
DHE_TEST ( kasvaliditytest_ffcephem_nokc_zzonly_resp_fc_0,
	   MODULUS ( 0xf5, 0x28, 0xaa, 0x27, 0x62, 0xdf, 0x76, 0xd7, 0x80,
		     0x2f, 0xea, 0x08, 0x70, 0x05, 0xd7, 0x6f, 0xeb, 0x69,
		     0xb7, 0xaf, 0xd9, 0xe7, 0xfb, 0x36, 0x37, 0x15, 0xa1,
		     0xb4, 0x4f, 0x09, 0xdb, 0xda, 0x5d, 0x06, 0xa6, 0x3e,
		     0x3d, 0x51, 0x2a, 0xbe, 0x9c, 0x68, 0x10, 0xab, 0xba,
		     0x2b, 0xfd, 0x48, 0xcf, 0x55, 0x45, 0x13, 0x71, 0x2d,
		     0xf2, 0x5f, 0x78, 0x62, 0x65, 0xa7, 0x5f, 0x4e, 0x1d,
		     0xac, 0xaa, 0xcb, 0xbe, 0x9e, 0x52, 0x8a, 0x63, 0x46,
		     0xa8, 0xbf, 0x38, 0x01, 0x5e, 0xcb, 0xc5, 0x23, 0xd4,
		     0xe7, 0x8d, 0x73, 0x8b, 0x88, 0x52, 0xbc, 0xd6, 0x66,
		     0x00, 0xdc, 0x43, 0x42, 0x86, 0xc7, 0x8f, 0x6c, 0xca,
		     0xe9, 0x92, 0x56, 0x8b, 0x25, 0x8b, 0xb0, 0xd3, 0xb5,
		     0x03, 0xe3, 0x7e, 0x7a, 0x95, 0x6e, 0xc1, 0x61, 0x69,
		     0x78, 0xc1, 0xe7, 0x98, 0x92, 0x29, 0xe3, 0x99, 0x14,
		     0xea, 0x7c, 0x1b, 0xea, 0x39, 0x02, 0xe6, 0x69, 0xdf,
		     0xc6, 0x23, 0x68, 0xda, 0x13, 0x74, 0x48, 0x35, 0x0c,
		     0x1d, 0xef, 0x54, 0xbf, 0x3e, 0xe2, 0x14, 0x2c, 0x06,
		     0x56, 0x01, 0xdf, 0xae, 0x6b, 0xca, 0x07, 0xe9, 0x3c,
		     0xc4, 0x63, 0x39, 0x3c, 0xbc, 0x48, 0x3b, 0x2c, 0x72,
		     0x72, 0x55, 0x37, 0x88, 0x2e, 0xd5, 0xb9, 0x60, 0x74,
		     0xf5, 0x19, 0xfc, 0x73, 0x8f, 0x50, 0x33, 0xb8, 0x7b,
		     0x5e, 0xdc, 0xa0, 0xec, 0x4a, 0x34, 0xdc, 0x0f, 0x84,
		     0xd2, 0xfa, 0x2c, 0x36, 0x5f, 0x69, 0x1a, 0x8f, 0xc7,
		     0xdc, 0x64, 0xbc, 0x08, 0x39, 0xa5, 0x81, 0x3f, 0xef,
		     0x9a, 0xf3, 0x85, 0xc6, 0x8e, 0x95, 0x2a, 0xf5, 0x66,
		     0xf7, 0x9b, 0x08, 0x02, 0xbe, 0x99, 0x00, 0xcf, 0x08,
		     0x38, 0x03, 0x2e, 0xd2, 0x49, 0xaf, 0xea, 0x05, 0x57,
		     0x21, 0x54, 0x0c, 0x89, 0x61, 0xe6, 0x30, 0x0a, 0xd5,
		     0xde, 0xc6, 0x51, 0xc7 ),
	   GENERATOR ( 0x08, 0x5c, 0x0a, 0x05, 0x12, 0xc2, 0x41, 0xb5, 0x83,
		       0xd4, 0x17, 0x03, 0xed, 0xfb, 0xfe, 0xa2, 0xa3, 0xe8,
		       0x63, 0xde, 0xac, 0x68, 0x85, 0x50, 0x97, 0x70, 0x79,
		       0x67, 0xe0, 0x97, 0x18, 0x6e, 0xf8, 0x9d, 0x05, 0xc6,
		       0x52, 0x27, 0xf5, 0x6b, 0x12, 0xde, 0x61, 0x23, 0xba,
		       0xc8, 0x6c, 0x3c, 0x13, 0xd6, 0x80, 0x99, 0x4b, 0xff,
		       0xaf, 0x24, 0xb2, 0xa1, 0xed, 0x7f, 0x01, 0xc1, 0x08,
		       0xb0, 0x65, 0x93, 0xf2, 0x2f, 0x74, 0xb4, 0xaf, 0x22,
		       0x2b, 0x46, 0xe5, 0xfb, 0x89, 0x48, 0x2f, 0xbd, 0x96,
		       0xf5, 0x45, 0x1c, 0x9f, 0x45, 0x39, 0x31, 0x36, 0xec,
		       0x03, 0x7a, 0xa8, 0x1a, 0x81, 0x24, 0x54, 0x59, 0xec,
		       0x01, 0x80, 0x24, 0x51, 0x83, 0x94, 0xf4, 0xd9, 0x36,
		       0x59, 0x6d, 0xc5, 0x3c, 0x3d, 0x8a, 0x9f, 0x73, 0x29,
		       0x03, 0x71, 0x97, 0x96, 0xc0, 0x45, 0xb6, 0x2f, 0xad,
		       0xea, 0x9d, 0xd1, 0xb2, 0xfa, 0xbe, 0xc1, 0x56, 0x0d,
		       0xdb, 0x3b, 0x78, 0x0d, 0x96, 0x46, 0xad, 0x0d, 0xd3,
		       0x16, 0x8c, 0x07, 0xcc, 0x99, 0x4f, 0x79, 0xee, 0x80,
		       0x4c, 0xae, 0x07, 0x57, 0x39, 0x12, 0x51, 0x1d, 0xe0,
		       0x50, 0xd0, 0x5a, 0x0d, 0x58, 0xb8, 0x19, 0xec, 0x41,
		       0xe7, 0xc1, 0x20, 0x5d, 0xc7, 0x19, 0x9f, 0xc6, 0x5a,
		       0x6a, 0x1a, 0x4f, 0xfc, 0xb4, 0xdf, 0x38, 0xd9, 0xb6,
		       0x75, 0x72, 0x69, 0x00, 0x34, 0x01, 0xd8, 0x4c, 0x73,
		       0x23, 0x85, 0xa5, 0x51, 0x74, 0xf2, 0x7d, 0x4b, 0x49,
		       0x3c, 0xb7, 0x10, 0x98, 0x0c, 0x3a, 0xf9, 0x8b, 0xe7,
		       0xbc, 0xff, 0x94, 0x67, 0xe8, 0x17, 0x92, 0xf2, 0xd2,
		       0xf9, 0xa2, 0xf8, 0xd5, 0xd0, 0xdd, 0xc9, 0xa2, 0x29,
		       0x79, 0x01, 0x92, 0xa1, 0x94, 0xda, 0x5b, 0x20, 0x32,
		       0xf2, 0xf0, 0xcc, 0x23, 0xff, 0xbc, 0xb2, 0xcc, 0x16,
		       0x6f, 0x21, 0x28, 0xee ),
	   PARTNER ( 0x81, 0x5e, 0x1c, 0x8e, 0x64, 0xbe, 0x67, 0x86, 0xb6,
		     0xc2, 0x19, 0x45, 0x06, 0xb6, 0x61, 0x19, 0x9a, 0x56,
		     0xc8, 0x07, 0xc6, 0x34, 0x0a, 0x83, 0x69, 0x7d, 0x3b,
		     0xc7, 0x53, 0x04, 0xfb, 0xeb, 0x91, 0xbb, 0x15, 0xac,
		     0x79, 0x81, 0xba, 0xbb, 0xa2, 0x60, 0xa6, 0x53, 0xe6,
		     0x66, 0xb0, 0xda, 0x80, 0x12, 0x04, 0x2a, 0x08, 0x85,
		     0x47, 0xf0, 0xfb, 0xd1, 0xef, 0x56, 0xd1, 0xd1, 0xf6,
		     0x5e, 0xd8, 0x09, 0xca, 0x85, 0x45, 0x13, 0x9f, 0x57,
		     0xd3, 0xeb, 0xda, 0x3e, 0x0f, 0x13, 0xe7, 0xef, 0x19,
		     0xbd, 0xed, 0x4a, 0x8d, 0xfa, 0x20, 0x4f, 0x24, 0x57,
		     0xa3, 0xd3, 0x92, 0x83, 0x9d, 0xd7, 0x0b, 0xec, 0x53,
		     0x8a, 0x29, 0xd7, 0x88, 0x74, 0x70, 0x9f, 0x90, 0x5e,
		     0xc2, 0x87, 0x40, 0xbe, 0x15, 0x76, 0x96, 0xaa, 0x86,
		     0x2b, 0x57, 0xab, 0x51, 0xef, 0x20, 0x03, 0x36, 0x01,
		     0xff, 0x83, 0xba, 0x3f, 0xb0, 0xe8, 0x02, 0x55, 0x67,
		     0x4e, 0x0e, 0x6e, 0x72, 0x69, 0xcd, 0x85, 0xca, 0x10,
		     0x37, 0x9d, 0x41, 0x8b, 0x73, 0x7d, 0x3c, 0xde, 0xb5,
		     0xf3, 0x1f, 0xb0, 0x6f, 0x4a, 0x14, 0x11, 0x45, 0x93,
		     0x6a, 0x9e, 0xec, 0x05, 0x51, 0xb4, 0xba, 0x59, 0xc5,
		     0x9d, 0x2b, 0x3e, 0xdd, 0xc1, 0x9e, 0xa1, 0x89, 0x78,
		     0x27, 0x9c, 0xf1, 0x1f, 0x21, 0xa6, 0x5a, 0x9d, 0x2a,
		     0x10, 0x4c, 0xfe, 0x6a, 0xf9, 0x3f, 0xa8, 0xa1, 0xcc,
		     0xd1, 0x8b, 0x03, 0x01, 0x7a, 0x8a, 0xb3, 0x19, 0x43,
		     0x99, 0x31, 0x92, 0xf7, 0x94, 0xd8, 0x7b, 0xbc, 0x27,
		     0x3e, 0xf9, 0x16, 0x0a, 0xf9, 0x54, 0x48, 0x07, 0x83,
		     0xcf, 0xb6, 0xa3, 0x7b, 0xa0, 0x85, 0x50, 0x3b, 0xa2,
		     0x58, 0x16, 0x9a, 0x8d, 0xab, 0xa7, 0x1d, 0x4c, 0x52,
		     0xde, 0x7c, 0x4e, 0x65, 0x47, 0x22, 0x23, 0xc2, 0x02,
		     0xad, 0x27, 0x7e, 0x93 ),
	   PRIVATE ( 0x56, 0x1c, 0x94, 0x8b, 0xe9, 0xd0, 0xd9, 0x68, 0xcd,
		     0xfc, 0x6d, 0x27, 0xcb, 0xfe, 0x52, 0xcd, 0xa3, 0x42,
		     0xf5, 0x44, 0xf5, 0x6d, 0x57, 0xf1, 0x18, 0x86, 0x63,
		     0x26, 0xf7, 0x6e, 0x1f, 0x70 ),
	   PUBLIC ( 0x3c, 0x24, 0x97, 0x3b, 0x32, 0x96, 0x26, 0x97, 0x59,
		    0xce, 0x38, 0xe2, 0xe5, 0x25, 0xc0, 0x09, 0x5d, 0x4b,
		    0x5c, 0x34, 0xbe, 0x5d, 0x13, 0xb4, 0x5b, 0x22, 0x2b,
		    0x8e, 0x48, 0x9c, 0x27, 0xfb, 0x44, 0x42, 0xf3, 0x2c,
		    0xc7, 0x64, 0x66, 0x5e, 0x28, 0xa0, 0x66, 0x55, 0xc7,
		    0x1f, 0xb3, 0x7f, 0xd8, 0x75, 0xa9, 0x21, 0xd1, 0x79,
		    0x55, 0x1a, 0x1e, 0x4f, 0x2a, 0x90, 0x54, 0xa7, 0x6c,
		    0xae, 0x2a, 0x61, 0xd3, 0xcb, 0xec, 0x55, 0xc3, 0xbe,
		    0x19, 0x85, 0x3a, 0x54, 0x09, 0xd9, 0xff, 0x91, 0x4b,
		    0x93, 0xbc, 0x78, 0xb8, 0xaa, 0x15, 0x25, 0xb9, 0x08,
		    0xf3, 0x24, 0x19, 0xa8, 0x8d, 0x77, 0x26, 0xea, 0xd7,
		    0x6a, 0x3f, 0x88, 0x95, 0xd6, 0x30, 0x71, 0xa9, 0xb0,
		    0xa6, 0x3f, 0xe4, 0x72, 0x8d, 0x19, 0x51, 0x8d, 0x1d,
		    0x08, 0x08, 0x81, 0x41, 0xb8, 0x26, 0x9f, 0x0b, 0x0c,
		    0xd7, 0x71, 0x12, 0xd4, 0x76, 0xaf, 0x9e, 0xfc, 0xc7,
		    0xf5, 0x90, 0xaf, 0x8f, 0xc1, 0xf9, 0xdc, 0x5e, 0x4c,
		    0x00, 0xcd, 0x5d, 0xfa, 0x64, 0xa3, 0x3b, 0x2d, 0xf4,
		    0xdb, 0x9d, 0x85, 0x94, 0xd8, 0x74, 0x89, 0xbe, 0xa6,
		    0xf6, 0xf3, 0x79, 0x58, 0xbf, 0xb5, 0x98, 0xe5, 0x69,
		    0x2b, 0x81, 0xbf, 0x11, 0x6b, 0x60, 0x22, 0x7b, 0x62,
		    0x52, 0xa6, 0x43, 0x8f, 0x04, 0x9c, 0x5c, 0x44, 0x9b,
		    0xab, 0x02, 0x77, 0x40, 0xf8, 0x55, 0x1b, 0xf1, 0xff,
		    0xe2, 0x50, 0x84, 0xf2, 0x31, 0xff, 0x64, 0x63, 0x88,
		    0xd0, 0x09, 0xba, 0x22, 0x19, 0x32, 0x62, 0x02, 0x9b,
		    0xa1, 0x9a, 0xf2, 0x64, 0x3d, 0xd6, 0x79, 0xf2, 0x83,
		    0x21, 0x2a, 0x2d, 0x26, 0xad, 0x91, 0x7e, 0xfe, 0x96,
		    0x42, 0xc7, 0x48, 0xfc, 0xeb, 0x33, 0xfb, 0x0a, 0x6c,
		    0x13, 0x2f, 0x37, 0x8d, 0xad, 0xa2, 0xcc, 0xce, 0xde,
		    0x08, 0x6d, 0x8a, 0x31 ),
	   SHARED ( 0xec, 0x23, 0x09, 0xa7, 0xd2, 0x38, 0xae, 0xb0, 0x67,
		    0x14, 0xc2, 0x7c, 0x1b, 0xbb, 0xb1, 0xb1, 0xc5, 0xaa,
		    0x3c, 0xdd, 0xdd, 0x76, 0xa4, 0x19, 0xb1, 0xf3, 0x70,
		    0x4d, 0xd3, 0x43, 0x7c, 0xd1, 0xf2, 0xc8, 0x84, 0x3f,
		    0x35, 0x0d, 0x67, 0x87, 0x2e, 0xe3, 0x25, 0x97, 0x3f,
		    0x4e, 0x5a, 0xce, 0x7d, 0x40, 0x6b, 0xe5, 0xde, 0x75,
		    0xd9, 0xa9, 0x12, 0x0a, 0xf6, 0x7e, 0x32, 0xf0, 0x29,
		    0x1e, 0x77, 0xe7, 0xa3, 0x97, 0x62, 0x49, 0x29, 0xd6,
		    0x3d, 0xc0, 0xc4, 0x2e, 0xf8, 0xf4, 0x42, 0xce, 0x89,
		    0xa3, 0x9b, 0x19, 0x2f, 0xee, 0x38, 0x6c, 0xe6, 0x83,
		    0x01, 0xc4, 0xb8, 0x28, 0xea, 0x91, 0x89, 0x79, 0x83,
		    0x46, 0xb6, 0x0f, 0x61, 0x5d, 0xd9, 0x63, 0x91, 0x05,
		    0xff, 0xea, 0x6e, 0xc6, 0x1e, 0xe5, 0xe4, 0xa7, 0xd6,
		    0x8c, 0xe7, 0x2b, 0xbf, 0x12, 0x81, 0xd6, 0x86, 0x4e,
		    0x30, 0x18, 0x1c, 0xe4, 0x19, 0x95, 0x2a, 0x3e, 0xf8,
		    0x3a, 0x9a, 0xd7, 0xb2, 0x6f, 0xc7, 0x29, 0x2a, 0xd7,
		    0x45, 0xbf, 0xb5, 0x43, 0xe5, 0xc2, 0xea, 0x31, 0x0a,
		    0x41, 0x59, 0xe9, 0xd6, 0x60, 0x27, 0x9d, 0x12, 0xc1,
		    0xe0, 0x38, 0x50, 0xe8, 0x37, 0xc0, 0x1c, 0x54, 0x2a,
		    0x0f, 0x59, 0xad, 0x61, 0xd0, 0x73, 0x10, 0x05, 0xe8,
		    0x00, 0x9a, 0x3d, 0x84, 0x06, 0x69, 0x1a, 0xbb, 0x22,
		    0xf5, 0xf2, 0xe9, 0x6a, 0xe3, 0x45, 0x78, 0x3c, 0x40,
		    0x3e, 0x59, 0xb9, 0xe9, 0x48, 0xad, 0xdb, 0xea, 0x8a,
		    0xc7, 0xd7, 0x70, 0x82, 0x10, 0x44, 0xe0, 0x3f, 0x15,
		    0xae, 0x6f, 0xc3, 0x67, 0xdd, 0xc8, 0x5b, 0xa6, 0x2a,
		    0x26, 0xb4, 0xd9, 0x4d, 0x77, 0x05, 0xf5, 0xec, 0xad,
		    0x8a, 0xa2, 0x1b, 0x61, 0x9d, 0x0e, 0x09, 0xf1, 0x24,
		    0xbe, 0xe8, 0x65, 0x8a, 0x21, 0x87, 0xf7, 0x02, 0x91,
		    0x07, 0x10, 0x5d, 0xbf ) );

/**
 * Perform Ephemeral Diffie-Hellman self-tests
 *
 */
static void dhe_test_exec ( void ) {

	dhe_key_ok ( &kasvaliditytest_ffcephem_nokc_zzonly_init_fb_0 );
	dhe_key_ok ( &kasvaliditytest_ffcephem_nokc_zzonly_init_fc_0 );
	dhe_key_ok ( &kasvaliditytest_ffcephem_nokc_zzonly_resp_fb_0 );
	dhe_key_ok ( &kasvaliditytest_ffcephem_nokc_zzonly_resp_fc_0 );
}

/** Ephemeral Diffie-Hellman self-test */
struct self_test dhe_test __self_test = {
	.name = "dhe",
	.exec = dhe_test_exec,
};
