/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: checkpc.c
 * Check out a printcap; create spool dirs, fix ownerships, etc.
 ***************************************************************************
 * checkpc: check out the entries in the printcap,
 * checking to see that the default files and directories are present
 * or absent as are indicated by the printcap entry.
 *
 * checkpc [-l] [-s] [-a] [-o] [-f] [-Pprinter] [printcapfile]
 * The default is to create all the various files;
 * the following flags are used to disable automatic creation:
 * -l - no log file
 * -s - no status file
 * -o - no output device
 * -a - no accounting file
 * -f - fix, attempt to create with the desired perms
 * -P printer this printer only
 * printcapfile: name of the printcap file to be used; default is /etc/printcap
 */

#include "lp.h"
#include "library/errormsg.h"
#include "library/readconf.h"
#include "patchlevel.h"

static int nolog;			/* no log file */
static int noout;			/* no output file */
static int nostat;			/* no status file */
static int noacct;			/* no accounting file */
static int fix;				/* fix if bad or missing */
static struct passwd *pwent;		/* password entry */
static struct group *grent;		/* group entry */
static int DU, DG;
static char *DaemonGroup;
static int canchown;			/* can do chown */

static void fixup(int, char *, int, char *, char *);

/*
 * check and fix if necessary
 */
static int all_check(int remove, int type, char *path,
	int perms, char *owner, char *group)
{
    struct stat statbuf;
    char cmd[BUFSIZ];
    int status;
    int ok;

    assert(path!=(char*)0);
    ok = 1;
    if (stat (path, &statbuf) < 0) {
	fprintf (stdout, "cannot be stated\n");
	if (remove) {
	    return 1;
	}
	if (fix) {
	    fixup (type, path, perms, owner, group);
	}
	return (0);
    }
    if (remove) {
	(void) fprintf (stdout, "removing %s\n", path);
	(void) fflush (stdout);
	(void) sprintf (cmd, "rm -f %s", path);
	status = system (cmd);
	if (status) {
	    (void) fprintf (stdout, "fix: %s failed, try by hand\n", cmd);
	    (void) fflush (stdout);
	    exit (1);
	}
    }
    if ((statbuf.st_mode & type) == 0) {
	(void) fprintf (stdout, "not a %s ",
			(type == S_IFDIR) ? "directory" : "file/device");
	(void) fflush (stdout);
	return (0);
    }
    if (strcmp (path, "/dev/null")) {
	/* don't try and fix /dev/null!! */
	if (statbuf.st_uid != DU) {
	    ok = 0;
	    (void) fprintf (stdout, "not owner '%s' ", owner);
	    (void) fflush (stdout);
	    if (fix && canchown) {
		fixup (type, path, perms, owner, group);
		return 0;
	    }
	}
	if (statbuf.st_gid != DG) {
	    ok = 0;
	    (void) fprintf (stdout, "not group '%s' ", group);
	    (void) fflush (stdout);
	    if (fix && canchown) {
		fixup (type, path, perms, owner, group);
		return 0;
	    }
	}
	/* if mode & perms == perms, it's grand for our purposes */
	if ((statbuf.st_mode & perms) != perms) {
	    ok = 0;
	    (void) fprintf (stdout, "perms are %o ", (unsigned int)
	   		 (statbuf.st_mode & 0777));
	    (void) fflush (stdout);
	    if (fix) {
		fixup (type, path, perms, owner, group);
		return 0;
	    }
	}
    }
    putchar ('\n');
    (void) fflush (stdout);
    return (ok);
}

/*
 * doprinter: 1. get the printcap entry 2. check the spool direcotry for ownership, etc.
 * 3. check the log, accounting, and lock file for existence.
 */

static void doprinter(void) {
    if (Get_pc_entry (Printer, All_pc_vars, All_pc_len) == 0) {
	(void) fprintf (stdout, "no printer %s\n", Printer);
	(void) fflush (stdout);
	return;
    }
    (void) fprintf (stdout, "checking printer %s: ", Printer);
    (void) fflush (stdout);
    /*
     * check to see if the spool directory exists
     */
    (void) fprintf (stdout, "SD %s, ", SD);
    (void) fflush (stdout);
    (void) all_check (0, S_IFDIR, SD, 0755, Daemon_user, DaemonGroup);
    if (chdir (SD) < 0) {
	logerr (XLOG_DEBUG, "cannot chdir to SD");
	return;
    }
    if (noout == 0 && LP && *LP) {
	/* ignore the LP check if it's an LP-pipe */
	if (*LP != '|') {
	    (void) fprintf (stdout, "LP %s, ", LP);
	    (void) fflush (stdout);
	    (void) all_check (noout, S_IFREG | S_IFBLK, LP, 0644, Daemon_user, DaemonGroup);
	}
    }
    if (nolog == 0 && LF && *LF) {
	(void) fprintf (stdout, "LF %s, ", LF);
	(void) fflush (stdout);
	(void) all_check (nolog, S_IFREG, LF, 0664, Daemon_user, DaemonGroup);
    }
    if (LO && *LO) {
	(void) fprintf (stdout, "LO %s, ", LO);
	(void) fflush (stdout);
	(void) all_check (0, S_IFREG, LO, 0644, Daemon_user, DaemonGroup);
    }
    if (noacct || AF == 0 || *AF == 0) {
	(void) fprintf (stdout, "no AF, ");
	(void) fflush (stdout);
    } else {
	(void) fprintf (stdout, "AF %s, ", AF);
	(void) fflush (stdout);
	(void) all_check (noacct, S_IFREG, AF, 0664, Daemon_user, DaemonGroup);
    }
    if (nostat == 0 && ST && *ST) {
	(void) fprintf (stdout, "ST %s, ", ST);
	(void) fflush (stdout);
	(void) all_check (nostat, S_IFREG, ST, 0664, Daemon_user, DaemonGroup);
    }
    if (PS && *PS) {
	(void) fprintf (stdout, "PS %s, ", PS);
	(void) fflush (stdout);
	(void) all_check (nostat, S_IFREG, PS, 0664, Daemon_user, DaemonGroup);
    }
    (void) fprintf (stdout, "\n");
    (void) fflush (stdout);
}

plp_signal_t cleanup(int signal) {;}

static void usage(void) {
    (void) fprintf (stdout,
		    "Usage: %s [-aflso][-Pprinter] [printcapfile]\n", Name);
    (void) fflush (stdout);
    exit (1);
}

int main(int argc, char *argv[], char *envp[]) {
    char **prlist;
    int option;
    char *tokpath;

    /*
     * set umask to avoid problems with user umask
     */
    (void) umask (0);

    while ((option = Getopt (argc, argv, "D:afolsP:")) != EOF) {
	switch (option) {
	case 'D':		/* turn on Debugging */
	    Parse_debug (Optarg);
	    break;
	default:
	    usage ();
	    break;
	case 'a':		/* no acct file */
	    ++noacct;
	    break;
	case 'f':		/* no acct file */
	    ++fix;
	    break;
	case 'o':		/* no output file */
	    ++noout;
	    break;
	case 'l':		/* no log file */
	    ++nolog;
	    break;
	case 's':		/* no status file */
	    ++nostat;
	    break;
	case 'P':		/* printer */
	    if (Optarg) {
		Printer = Optarg;
	    } else {
		usage ();
	    }
	    break;
	}
    }
    Std_environ (argc, argv, envp);
    Readconf ();

    if (Printcap_path == NULL || !*Printcap_path) {
	logerr_die (XLOG_INFO, "printcap-path is unset!");
    }
    if (Optind < argc) {
	tokpath = (char *) malloc (strlen (Printcap_path) + strlen (argv[Optind]) + 2);
	(void) sprintf (tokpath, "%s:%s", argv[Optind], Printcap_path);
	++Optind;
    } else {
	tokpath = strdup (Printcap_path);
    }
    if (Optind < argc) {
	usage ();
    }
    canchown = (geteuid () == 0);

    if (canchown == 0) {
	(void) fprintf (stdout, "%s: warning: not running as root, can't chown\n", Name);
	(void) fflush (stdout);
    }
    if (Daemon_user == NULL) {
	logerr_die (XLOG_DEBUG, "haven't read config file yet");
    }
    if ((pwent = getpwnam (Daemon_user)) == NULL) {
	logerr_die (XLOG_DEBUG, "getpwnam failed for %s", Daemon_user);
    }
    DU = pwent->pw_uid;
    DG = pwent->pw_gid;

    if ((grent = getgrgid (DG)) == NULL) {
	logerr_die (XLOG_DEBUG, "getgrgid failed for gid %d",
		    DG);
    }
    DaemonGroup = grent->gr_name;

    {
	char *pc;

	pc = (char *) strtok (tokpath, ":");
	do {
	    if (pc[0] != '/') {
		(void) fprintf (stdout,
				"Printcap file '%s' must be absolute path name\n", pc);

		(void) fflush (stdout);
		exit (1);
	    }
	} while (strtok (NULL, ":"));
    }
    free (tokpath);

    if (Printer) {
	doprinter ();
    } else {
	(void) fprintf (stdout, "Doing all printers\n");
	(void) fflush (stdout);
	for (prlist = All_printers (); (Printer = *prlist); ++prlist) {
	    doprinter ();
	}
    }
    return (0);
}

static void fix_system(char *cmd) {
    int status;
    static char path_decl[BUFSIZ];
    static char *cmd_with_path = NULL;

    assert(cmd!=(char*)0);
    (void) fprintf (stdout, "FIX: %s\n", cmd);
    (void) fflush (stdout);

    if (cmd_with_path == NULL) {
	/* try and ensure chown is in the path. */
	(void) strcpy (path_decl, "PATH=/etc:/usr/etc:/usr/sbin:$PATH; ");
	cmd_with_path = &(path_decl [strlen (path_decl)]);
    }

    strcpy (cmd_with_path, cmd);

    status = system (cmd_with_path);
    if (status) {
	(void) fprintf (stdout, "fix: %s failed, try by hand!\n", cmd);
    }
}

/*
 * try to create the file/directory with the appropriate permissions
 */
static void fixup(int type, char *path,
	int perms, char *owner, char *group)
{
    char cmd[BUFSIZ];
    struct stat statbuf;

    (void) fprintf (stdout, "trying to fix %s %s\n",
   	(type == S_IFDIR) ? "directory" : "file", path);
    (void) fflush (stdout);

    if (stat (path, &statbuf) < 0) {
	if (type == S_IFDIR) {
	    (void) sprintf (cmd, "mkdir %s", path);
	} else {
	    (void) sprintf (cmd, "touch %s", path);
	}
        fix_system (cmd);
    }

    if (stat (path, &statbuf) < 0) {
	(void) fprintf (stdout, "cannot stat %s, you have to try by hand\n", path);
	(void) fflush (stdout);
    }
    /*
     * fix up ownership and perms, you are root
     */
    if (canchown) {
	if (type == S_IFDIR) {
	    (void) sprintf (cmd, "cd %s; chown %s .; chgrp %s .", path, owner, group);
	} else {
	    (void) sprintf (cmd, "chown %s %s; chgrp %s %s", owner, path, group, path);
	}
    	fix_system (cmd);
    }
    (void) sprintf (cmd, "chmod %o %s", perms, path);
    fix_system (cmd);
}
