/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: lpr_canprint.c
 * Checks to see if a file is printable
 *
 * This will be a system dependent function.  If you have setreuid(),
 * you can implement this in a fairly effective manner, as the file
 * open and stat() is done as the user, not as root.
 *
 * The "is_exec" and "is_arch" functions are very system dependent.
 * If you have the source for file(1) available,  check to see how it
 * determines the file types.  This may be more bizzare than you expect.
 ***************************************************************************/

#include <assert.h>
#include "library/errormsg.h"
#include "lpr.h"
#include "lpr_canprint.h"
#include "lpr_global.h"

#define MMDF_SEPARATOR "\001\001\001\001"

static int is_mmdf_mail(char *buf) {
    assert(buf!=(char*)0);
    return (!strncmp (buf, MMDF_SEPARATOR, strlen (MMDF_SEPARATOR)));
}

/*
 * The is_exec and is_arch are system dependent functions which check if a file is an
 * executable or archive file, based on the information in the header.
 */

#ifdef HAVE_A_OUT_H
#include <a.out.h>
#endif
#ifdef HAVE_EXECHDR_H
#include <sys/exechdr.h>
#endif

#ifdef IS_NEXT
/* this causes trouble, eg. on SunOS. */
#ifdef HAVE_SYS_LOADER_H
#include <sys/loader.h>
#endif
#ifdef HAVE_NLIST_H
#include <nlist.h>
#endif
#ifdef HAVE_STAB_H
#include <stab.h>
#endif
#ifdef HAVE_RELOC_H
#include <reloc.h>
#endif
#endif

#if defined(HAVE_FILEHDR_H) && !defined(HAVE_A_OUT_H)
#include <filehdr.h>
#endif
#if defined(HAVE_AOUTHDR_H) && !defined(HAVE_A_OUT_H)
#include <aouthdr.h>
#endif
#ifdef HAVE_SGS_H
#include <sgs.h>
#endif

#ifndef XYZZQ_
#define XYZZQ_ 1		/* ugh! antediluvian BSDism, I think */
#endif

/*
 * The N_BADMAG macro isn't present on some OS'es.
 * 
 * If it isn't defined on yours, just edit your system's "system/os.h" file and #undef both
 * USE_A_OUT_H and USE_EXECHDR_H. This function will allow files through if it can't
 * find a working magic-interpretation macro.
 */
#ifndef N_BADMAG
#ifdef NMAGIC
#define N_BADMAG(x) \
	((x).a_magic!=OMAGIC && (x).a_magic!=NMAGIC && (x).a_magic!=ZMAGIC)

#else				/* no NMAGIC */

#ifdef MAG_OVERLAY		/* AIX */
#define N_BADMAG(x) (x.a_magic == MAG_OVERLAY)
#endif				/* MAG_OVERLAY */

#endif				/* NMAGIC */
#endif				/* N_BADMAG */

static int is_exec(int fd, char *buf, int n)
{
    int i = 0;

    assert(buf!=(char*)0);
    if (lseek (fd, (long) 0, 0) < 0)
	logerr (XLOG_INFO, "is_exec: cannot lseek");

#ifdef N_BADMAG		/* BSD, non-mips Ultrix */
#ifdef HAVE_STRUCT_EXEC
    if (n >= sizeof (struct exec))
	i = !(N_BADMAG ((*(struct exec *) buf)));
#else
    if (n >= sizeof (struct aouthdr))
	i = !(N_BADMAG ((*(struct aouthdr *) buf)));
#endif
#endif

#ifdef ISCOFF		/* SVR4, mips Ultrix */
    if (n >= sizeof (struct filehdr))
	i = (ISCOFF (((struct filehdr *) buf)->f_magic));
#endif

#ifdef MH_MAGIC		/* NeXT */
    if (n >= sizeof (struct mach_header))
	i = (((struct mach_header *) buf)->magic == MH_MAGIC);
#endif

#ifdef IS_DATAGEN	/* Data General (forget it! ;) */
    {
	struct header header;
	if (read (fd, (char *) &header, sizeof (header)) == sizeof (header))
	    i = ISMAGIC (header.magic_number);
    }
#endif

    return (i);
}

#include <ar.h>

static int is_arch(int fd, char *buf, int n)
{
    assert(buf!=(char*)0);
#ifdef ARMAG
    return (!strncmp (buf, ARMAG, strlen (ARMAG)));
#else
    return (0);
#endif				/* ARMAG */
}

/***************************************************************************
 * Test to see if this is a printable file.
 * Return: 0 if unprintable; 1 otherwise
 ***************************************************************************/
static char err[] = "File '%s' not printed: %s";

int Is_printable(char *file, struct stat *statb)
{
    char buf[BUFSIZ];
    int fd = -1;		/* file descriptor */
    int n, i, c;		/* Acme Integers, Inc. */
    int succ = 0;

    assert(file!=(char*)0);
    assert(statb!=(struct stat*)0);
    if ((fd = open (file, O_RDONLY, 0)) < 0) {
	Warnmsg (err, file, Errormsg (errno));
    } else if (fstat (fd, statb) < 0) {
	Warnmsg (err, file, "cannot stat it");
    } else if ((statb->st_mode & S_IFMT) == S_IFDIR) {
	Warnmsg (err, file, "it is a directory");
    } else if (statb->st_size == 0) {
	Warnmsg (err, file, "it is empty");
    } else if ((n = read (fd, buf, sizeof (buf))) <= 0) {
	Warnmsg (err, file, "cannot read it");
    } else if (Binary || Format == 'l') {
	succ = 1;
    } else if (!(strchr ("fpr", Format) || (XT && strchr (XT, Format)))) {
	/*
	 * We don't have to do the following checks, applicable to text files.
	 */
	succ = 1;

    } else if (is_exec (fd, buf, n)) {
	Warnmsg (err, file, "executable program");
    } else if (is_arch (fd, buf, n)) {
	Warnmsg (err, file, "archive file");
    } else if (is_mmdf_mail (buf)) {
	succ = 1;
    } else {
	succ = 1;
	if (Check_for_nonprintable) {
	    for (i = 0; i < BUFSIZ; ++i) {
		c = buf[i];
		if (!isascii (c)) {
		    if (c != '' && c != '\b' && !isprint (c) && !isspace (c)) {
			Warnmsg (err, file, "it may be a binary file, try -b option.");
			succ = 0;
			break;
		    }
		}
	    }
	}
    }

    if (succ == 0) {
	(void) close (fd);
	fd = -1;
    } else if (lseek (fd, 0L, SEEK_SET) < 0) {
	logerr (XLOG_INFO, "Is_printable: lseek failed %s", file);
	(void) close (fd);
	fd = -1;
    }
    return (fd);
}
