#include "lp.h"
#include "library/errormsg.h"
#include "library/printcap.h"

#ifdef NIS
#include <rpc/rpc.h>
#include <rpcsvc/ypclnt.h>
#include <rpcsvc/yp_prot.h>

extern int change_domain;

#ifdef YP_BIND_RETURNS
#define yp_domain_exists yp_bind
#else

#ifndef YP_BIND_TIMEOUT
#define YP_BIND_TIMEOUT 2
#endif

#include <setjmp.h>
static jmp_buf yp_bind_failed;

void
yp_bind_timeout (void) {
    longjmp (yp_bind_failed, 1);
}

#ifdef YP_FORK
static int parent_ready = 0;

plp_signal_t
yp_child_handler (int signal) {
    parent_ready = 1;
    (void) plp_signal (SIGINT, SIG_DFL);
}

#endif

/******************************************************************************
 * This is a gross kludge. There doesn't seem to be any way of testing if NIS
 * domains exist, so this routine does it. If your yp_bind returns
 * YPERR_DOMAIN for non-existant domains, you're lucky. Set YP_BIND_RETURNS.
 * If yp_bind is interruptable by alarm(), you're not quite so lucky.
 * Otherwise, set YP_FORK. This implements a timeout for yp_bind by forking
 * and using the child process to wait for a specified time and interrupt the
 * parent. Setting an alarm() function doesn't work, because yp_bind appears
 * to be non-interruptable. NIS is a load of codswallop. --AJCD
 *****************************************************************************/
static int
yp_domain_exists (char *where) {
    static plp_signal_t (*func) (int);	/* what SIGALRM was set to */
    static unsigned int secs;
    static int res = 0;
#ifdef YP_FORK
    pid_t cpid, pid;
    plp_status_t status;
#endif

    if (DbgPcap > 5)
	log (XLOG_DEBUG, "yp_domain_exists: testing for domain %s", where);

#ifdef YP_FORK
# ifdef HAVE_SIGPROCMASK
    {
	sigset_t mask;
	sigemptyset (&mask);
	sigprocmask (SIG_SETMASK, &mask, NULL);
    }
# else
    (void) sigsetmask (0);
# endif

    if ((func = plp_signal (SIGINT, yp_child_handler)) == -1) {
	logerr_die (XLOG_INFO, "yp_domain_exists: can't set signal handler");
    }
    if ((cpid = fork ()) == -1) {
	logerr_die (XLOG_INFO, "yp_domain_exists: fork() failed");

    } else if (cpid == 0) {	/* child process */

#ifdef HAVE_SIGPROCMASK
	sigset_t mask, omask;
	sigemptyset (&mask);
	sigaddset (&mask, SIGINT);
	sigprocmask (SIG_BLOCK, &mask, &omask);
#else
	sigblock (sigmask (SIGINT));
#endif
	while (!parent_ready)
	    (void) sigpause (0);

#ifdef HAVE_SIGPROCMASK
	sigprocmask (SIG_SETMASK, &omask, NULL);
#else
	(void) sigsetmask (0);
#endif
	(void) sleep (YP_BIND_TIMEOUT);
	kill (getppid (), SIGINT);
	exit (0);
    } else {			/* parent process */
	if (DbgPcap > 5)
	    log (XLOG_DEBUG, "yp_domain_exists: child pid is %d", cpid);
	/* set up time out function */
	if (setjmp (yp_bind_failed)) {
	    if (DbgPcap > 5)
		log (XLOG_DEBUG, "yp_domain_exists: yp_bind to %s timed out", where);
	    res = 1;		/* timed out */
	} else {
	    (void) plp_signal (SIGINT, yp_bind_timeout);
	    kill (cpid, SIGINT);/* tell child we're ready */
	    res = yp_bind (where);
	    (void) plp_signal (SIGINT, SIG_IGN);	/* ignore interrupts */
	    kill (cpid, SIGINT);/* kill child, yp_bind succeeded */
	    if (DbgPcap > 5)
		log (XLOG_DEBUG, "yp_domain_exists: yp_bind(%s) returned %d", where, res);
	}

	/* wait for child process to exit */
	while ((pid = plp_wait_kid (&status, WNOHANG)) > 0) {
	    if (DbgPcap > 3)
		log (XLOG_DEBUG, "yp_domain_exists: process %d, status %s", pid,
		     Decode_status (&status));
	}
    }
    (void) plp_signal (SIGINT, func);	/* restore SIGALRM handler */
#else
    secs = alarm ((unsigned int) 0);	/* disarm */

#ifdef HAVE_SIGPROCMASK
    {
	sigset_t mask;
	sigemptyset (&mask);
	sigaddset (&mask, SIGALRM);
	sigprocmask (SIG_SETMASK, &mask, NULL);
    }
#else
    (void) sigsetmask (0);
    if (sigblock (0) & sigmask (SIGALRM)) {
	logerr_die (XLOG_INFO, "yp_domain_exists: SIGALRM is blocked");
    }
#endif

    /* set up time out function */
    if ((func = plp_signal (SIGALRM, yp_bind_timeout)) == (plp_signal_t *) - 1) {
	logerr_die (XLOG_INFO, "yp_domain_exists: can't set signal handler");
    }
    if (setjmp (yp_bind_failed)) {
	(void) alarm ((unsigned int) 0);	/* disarm alarm */
	if (DbgPcap > 5)
	    log (XLOG_DEBUG, "yp_domain_exists: yp_bind to %s timed out", where);
	res = 1;		/* timed out */
    } else {
	if (DbgPcap > 5)
	    log (XLOG_DEBUG, "yp_domain_exists: setting up alarm(%d)", YP_BIND_TIMEOUT);
	(void) alarm ((unsigned int) YP_BIND_TIMEOUT);
	if (DbgPcap > 5)
	    log (XLOG_DEBUG, "yp_domain_exists: calling yp_bind(%s)", where);
	res = yp_bind (where);
	(void) alarm ((unsigned int) 0);	/* disarm alarm */
	if (DbgPcap > 5)
	    log (XLOG_DEBUG, "yp_domain_exists: yp_bind(%s) returned %d", where, res);
    }
    (void) plp_signal (SIGALRM, func);	/* restore SIGALRM handler */
    if (secs)
	alarm (secs);		/* restart alarm */
#endif
    return res;
}

#endif

void
All_printers_nis (char *NISDomain) {
#ifndef YP_GET_DOMAIN_BROKEN
    char *d_flag = NISDomain;
#endif
    char *outval;
    int outvallen;

    if (NISDomain == NULL)
	if (yp_get_default_domain (&NISDomain))
	    return;

    if (DbgPcap > 5)
	log (XLOG_DEBUG, "All_printers_NIS: trying domain %s", NISDomain);

    if (!NIS_printcap_byname || !*NIS_printcap_byname) {
	logerr_die (XLOG_CRIT, "nis-printcap not set!");
    }
    if (yp_match (NISDomain, NIS_printcap_byname, "printers", strlen ("printers"),
		  &outval, &outvallen) == 0) {
	char *plist, *nextp;

	outval[outvallen] = '\0';
	for (plist = outval; plist && *plist; plist = nextp) {
	    if ((nextp = strchr (plist, '\n')))
		*nextp++ = '\0';
	    if (*plist)
		add_printer_list (plist);
	}
	(void) free (outval);
    }
#ifndef YP_GET_DOMAIN_BROKEN
    if (NISDomain && d_flag == NULL)
	(void) free (NISDomain);
#endif
    return;
}

char *
First_printer_nis (char *NISDomain) {
    char *bp;
    char *outkey, *outval;
    int outkeylen, outvallen;
#ifndef YP_GET_DOMAIN_BROKEN
    char *d_flag = NISDomain;
#endif
    static char first[PRNAMELEN + 1];
    char capbuf[BUFSIZ];

    if (NISDomain == NULL)
	if (yp_get_default_domain (&NISDomain))
	    return (NULL);

    if (!NIS_printcap_byname || !*NIS_printcap_byname) {
	logerr_die (XLOG_CRIT, "nis-printcap not set!");
    }
    if (!yp_first (NISDomain, NIS_printcap_byname, &outkey, &outkeylen,
		   &outval, &outvallen)) {
	(void) strcpy (capbuf, outval);
	free (outkey);
	free (outval);
	init_pc_entry (Status_pc_vars, Status_pc_len);
	getpcvars (Status_pc_vars, Status_pc_len, DefaultPrintcap, DO_CONT);
	getpcvars (Status_pc_vars, Status_pc_len, capbuf, DO_CONT);
	if (DbgPcap > 6)
	    show_pc (Status_pc_vars, Status_pc_len);
	if (SD != NULL && *SD != NULL) {
	    if ((bp = strchr (capbuf, '|')) == NULL) {
		bp = strchr (capbuf, ':');
	    }
	    *bp = '\0';
	    (void) strcpy (first, capbuf);
#ifndef YP_GET_DOMAIN_BROKEN
	    if (NISDomain && d_flag == NULL)
		free (NISDomain);
#endif
	    return (first);
	}
    }
#ifndef YP_GET_DOMAIN_BROKEN
    if (NISDomain && d_flag == NULL)
	free (NISDomain);
#endif
    return (NULL);
}

/*
 * Set_pc_entry_nis looks for a printcap entry in the NIS maps. Note that these rules are
 * different from the Hesiod lookup rules, because NIS is not a truly distributed
 * database, and it is unlikely to have servers for all zones accessible from one zone.
 * Search order depends on whether name domain is specified (contains @), and whether
 * lookup domain is default (NULL): name, default domain:        name/host, name name,
 * domain given:          name@domain name@where, default domain:  name@where name@where,
 * domain given:    name@where Note: hostname is used without domain for local lookup.
 */

int
Set_pc_entry_nis (char *NISDomain, char *name, char capbuf[]) {
    char *outval;
    int outvallen;
    char *mp, *stddomain;	/* map, standard domain ptr */
    char where[YPMAXDOMAIN];
    char map[YPMAXMAP];

    if (yp_get_default_domain (&stddomain))
	return (0);

    strcpy (map, name);		/* put name in map */

    if ((mp = strchr (map, '@'))) {
	*mp = '\0';		/* end of map ptr */
	strcpy (where, mp + 1);
    } else {
	mp = map + strlen (map);
	if (NISDomain)
	    strcpy (where, NISDomain);
	else
	    *where = '\0';
    }

    if (strchr (map, PNAME_SEPCH)) {
	logerr_die (XLOG_INFO, "printer name contains '%c' %s",
		    PNAME_SEPCH, name);
    }
    if (*where) {		/* domain check required */
	if (DbgPcap > 5) {
	    log (XLOG_DEBUG, "Set_pc_entry_NIS: trying %s in domain %s", map, where);
	}
	if (!NIS_printcap_byname || !*NIS_printcap_byname) {
	    logerr_die (XLOG_CRIT, "nis-printcap not set!");
	}
	if (yp_domain_exists (where) == 0 &&
	    yp_match (where, NIS_printcap_byname, map, strlen (map),
		      &outval, &outvallen) == 0) {
	    set_domain (where, stddomain);
	    bcopy (outval, capbuf, outvallen + 2);
	    free (outval);
#ifndef YP_GET_DOMAIN_BROKEN
	    free (stddomain);
#endif
	    return (1);		/* successful match */
	}
    } else {
	if (change_domain)
	    out_domain = NULL;	/* local lookup */
	*mp = PNAME_SEPCH;
	strcpy (mp + 1, ShortHost);	/* tag on host name */
	if (DbgPcap > 5) {
	    log (XLOG_DEBUG, "Set_pc_entry_NIS: trying %s in domain %s", map, stddomain);
	}
	if (yp_match (stddomain, NIS_printcap_byname, map, strlen (map),
		      &outval, &outvallen) == 0) {
	    bcopy (outval, capbuf, outvallen + 2);
	    free (outval);
#ifndef YP_GET_DOMAIN_BROKEN
	    free (stddomain);
#endif
	    return (1);		/* successful match */
	}
	*mp = '\0';		/* try without hostname */

	if (DbgPcap > 5) {
	    log (XLOG_DEBUG, "Set_pc_entry_NIS: trying %s in domain %s", map, stddomain);
	}
	if (yp_match (stddomain, NIS_printcap_byname, map, strlen (map),
		      &outval, &outvallen) == 0) {
	    bcopy (outval, capbuf, outvallen + 2);
	    free (outval);
	    free (stddomain);
	    return (1);		/* successful match */
	}
    }

#ifndef YP_GET_DOMAIN_BROKEN
    free (stddomain);
#endif
    return (0);
}

#endif				/* NIS */
