/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: lpr.c
 * lpr - off line print
 **************************************************************************/
#include "library/errormsg.h"
#include "library/readconf.h"

#include "lpr.h"
#include "lpr_canprint.h"
#include "lpr_filters.h"
#include "lpr_global.h"
#include "lpr_job.h"
#include "lpr_parms.h"
#include "lpr_temp.h"
#include "patchlevel.h"

int u_option;

/***************************************************************************
 * SYNOPSIS
 *      lpr [ -PPrinter ] [ -#num ] [ -C class ] [ -J job ] [
 *      -RremoteAccount ] [ -m[mailTo] ] [ -T title ] [-i[numcols]]
 *      [ -1234 font ] [ -wnum ] [ -Zzoptions ] [ -Uuser ] [ -HHost
 *      ] [ -Ffilter ] [ -bhrs ] [ -Dn ] [ -X ] [ filename ...  ]
 * DESCRIPTION
 *      Lpr uses a spooling server to print the named files when
 *      facilities become available.  If no Names appear, the stan-
 *      dard input is assumed.
 *      -PPrinter  Output to the specified Printer
 *      -F?  Filter or format specification
 *      -p text to be printed using pr(1) to format the files.
 *      -l (literal) text with control characters to be printed
 *      -t output from troff(1)
 *      -n output from DITROFF
 *      -d output from tex(l) (DVI format from Stanford).
 *      -g standard plot data as produced by the plot(3X) routines
 *      -v a raster image for devices like the Benson Varian.
 *      -c data produced by cifplot(l).
 *      -f same as -Fr, FORTAN carriage control
 *      -m[mailTo] Send mail upon completion
 *      -s Use symbolic links.
 *      -J jobname  Specify the job name to print on the burst page
 *      -T title specify the title used by pr(1);
 *      -wwidth  specify the page width for pr.
 *      -C class or priority (A - Z)
 *      -R remoteAccount
 *      -#num number of copies of each file to be printed.
 *      -i[numcols] Cause the output to be indented
 *      -b The files are assumed to contain binary data
 *      -Z extra options
 *      -D[n] debug level
 *      -Uuser Used by root process to specify a user
 *      -HHost Used by root process to specify a Host
 ****************************************************************************
 * Implementation:
 * 	Each time lpr is invoked it creates a "job" entry in the appropriate
 * spool directory.  Each job consists of a control file and zero or more
 * data files.  The control file contains lines that specify the job
 * parameters, such as job Name, etc., and the Names of the data files.
 *      Control file format
 *      First character in the line flags kind of entry, remainder of line is
 *          the arguemnt.
 *
 *		C -- "class Name" on banner page
 *		H -- "Host Name" of machine where lpr was done
 *		I -- "indent" amount to indent output
 *		J -- "job Name" on banner page
 *		L -- "literal" user's Name to print on banner
 *		M -- "mail" to user when done printing
 *		N -- "Name" of file (used by lpq)
 *		P -- "Person" user's login Name
 *		R -- account id  for charging
 *		U -- "unlink" Name of file to remove after we print it
 *		W -- "width" page width for PR
 *		X -- "header" header title for PR
 *		Z -- xtra options to filters
 *
 *	Lower case letters are formats, and are together with the Name
 *      of the file to print
 *		f -- "file Name" Name of text file to print
 *		l -- "file Name" text file with control chars
 *		p -- "file Name" text file to print with pr(1)
 *		t -- "file Name" troff(1) file to print
 *		n -- "file Name" ditroff(1) file to print
 *		d -- "file Name" dvi file to print
 *		g -- "file Name" plot(1G) file to print
 *		v -- "file Name" plain raster file to print
 *		c -- "file Name" cifplot file to print
 ***************************************************************************/

/***************************************************************************
 * remove the temp files
 ***************************************************************************/

plp_signal_t cleanup (int signal) {
    plp_block_signals ();
    Remove_temp ();
    exit (1);
}

/***************************************************************************
 * 1. get the Host computer Name and user Name
 * 2. get the parameters and other information
 * 3. set up signals to handle abrupt termination.
 * 4. set up the control file for the job
 * 5. if we are spooling from stdin, copy stdin to a file.
 * 6. if we have -p option (use pr), run the input files through pr
 *    and save to a single file.
 * 7. create a job entry
 * 8. start the server
 ****************************************************************************/

int main(int argc, char *argv[], char *envp[])
{
    struct passwd *pw_ent;    /* user entry in /etc/passwd */
    int i, fd, job_size=0;

    u_option = 0;
    (void) umask (0);
    set_user_uid (0);
    root_to_user ();

    Get_parms (argc, argv);	/* read the switches, set debug */
    Std_environ (argc, argv, envp);
    Readconf ();

    FQDN = Host;		/* set up the From information */
    From = ShortHost;

    /* get user name */
    if ((pw_ent = getpwuid (UserUID)) == 0) {
	logerr_die (XLOG_INFO, "getpwuid failed on uid %d", UserUID);
    }
    (void) strcpy (LOGNAME, pw_ent->pw_name);
    Person = LOGNAME;

    if (!UserUID) {
	Is_root = 1;		/* we are being invoked by root */
    }

    /* set up chatty information for user */
    if (BNRNAME[0] == '\0') {
	/*
	 * Add extra check - gecos field may be empty - if so stick with username.  What
	 * we wand to avoid is L field being unset - this means no banner to BSD systems.
	 */
	if (strsame (LOGNAME, pw_ent->pw_name) && (pw_ent->pw_gecos[0])) {
	    (void) strncpy (BNRNAME, pw_ent->pw_gecos, MAXPARMLEN);
	    BNRNAME[MAXPARMLEN] = '\0';
	} else {
	    (void) strcpy (BNRNAME, LOGNAME);
	}
    }
    Setup_parms ();			/* setup parameters */

    (void) plp_signal (SIGPIPE, (plp_sigfunc_t)SIG_IGN);	/* set signals */
    (void) plp_signal (SIGHUP, cleanup);
    (void) plp_signal (SIGINT, cleanup);
    (void) plp_signal (SIGQUIT, cleanup);
    (void) plp_signal (SIGTERM, cleanup);

    /**************************************************************************
     * The following possibilities exist:
     *                         Files              No Files
     * p format,no 'f' PF      PR ->spool         input->PR -> spool
     * p format, 'f' PF        PR->PF->spool      input->PR->PF->spool
     * ? format, no PF         files->spool files input->spool
     * ? format, PF            PF(files) -> file  input->spool
     *
     * If we have a prefilter for a file format,  we have to run the files
     * through the prefilter,  and spool the output of the prefilter.
     * The prefilter will create a single output file which is printed
     *
     * If we have 'p' format,  we will create a printer process, and run
     * its output into a temporary file in the spool directory.
     * If there are no names specified, we copy the input to a temporary
     * file in the spool directory.  The Read_stdin file is the name of the
     * output file.  This is also used to hold the name of the output of
     * the PR program.
     *
     * If we have a prefilter,  we place its output in the Read_filter
     * file.
     *************************************************************************/

    if (Format == 'p') {
	/*
	 * run PR and put the output in a spool file
	 */
	if (Debug > 3)
	    log (XLOG_DEBUG, "using PR");
	if (Run_pr () == 0) {
	    Diemsg ("nothing to print");
	}
	Format = 'f';

    } else if (Parmcount == 0) {
        if (Prefilter_name[Format - 'a'] == NULL) {
	    if (Debug > 3)
		log (XLOG_DEBUG, "Copying stdin");
	    if (Copy_stdin () == 0) {
		Diemsg ("nothing to print");
	    }
	}

    } else {
	/*
	 * check to see that the input files are printable
	 */
	job_size = 0;
	for (i = 0; i < Parmcount; ++i) {
	    fd = Is_printable (Parms[i].str, &LO_statb);
	    if (fd < 0) {
		Parms[i].str = NULL;
	    } else {
		job_size = 1;
	    }
	    (void) close (fd);
	}
	if (job_size == 0) {
	    Diemsg ("nothing to print");
	}
    }

    /*
     * now we check for prefilter; if we have one, then we have to run all the files
     * through it; we invoke the prefilter with the appropriate parameters. The output of
     * the prefilter (on stdout) is spooled
     */
    if (Prefilter_name[Format - 'a']) {
	/* yes, we have prefilter */
	if (Debug > 3)
	    log (XLOG_DEBUG, "Prefilter %c '%s'", Format,
		 Prefilter_name[Format - 'a']);

        if (Parmcount == 0 || Pr_stdin) {	/* also if pr has been run */
	    if (Do_prefilter (Prefilter_name[Format - 'a'], NULL) <= 0) {
	        job_size = 0;
	    } else {
	        job_size = 1;
	    }

	} else {
	    for (i = 0; i < Parmcount; i++) {
		if ((Parms[i].str == NULL) ||
			(Do_prefilter (Prefilter_name[Format - 'a'], &(Parms[i])) <= 0))
		{
		    Parms[i].str = NULL;
		} else {
		    job_size = 1;
		}
	    }
	}
	if (job_size == 0) {
	    Diemsg ("nothing to print");
	}
    }
    /*
     * we now either have to spool a single file, produced by the PR and/or Prefilters,
     * or we have to spool all the input files
     */
    Make_job ();
    /*
     * start up the server
     */
    (void) Startserver ();
    return (0);
}
