/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: Startprinter.c
 * Printer queue job handler
 *
 * StartPrinter is responsible for handling the Printer queue.
 * 1.  Get the printcap entry.
 * 2.  Check the spool directory to see if printing is enabled
 * 3.  Check to see if there are any jobs
 * 4.  If a remote Printer, call remotePrinter()
 *     If a local Printer, call localPrinter()
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "LPD/setproctitle.h"
#include "common/setstatus.h"

extern int Printjob (), Sendjob ();
extern int Printinit (), Setup_xfer ();
extern void Printfinal (), Printerror (), Allsent (), Send_error ();

extern int Single_printer;

static void Start_server (), unspool ();
static int do_job_now ();

static int job_removed;		/* set if the current job gets lprmed */
plp_signal_t handle_remove ();	/* handle 'remove job' signals */

void
Startprinter (void) {
    pid_t pid;			/* process id */
    char *ps, *st;		/* ACME Pointers */
    FILE *afp;			/* attach file pointer */
    char afname[MAXPARMLEN];	/* attach file name */
    char spoold[MAXPARMLEN];	/* actual spool directory */

    *spoold = '\0';
    /*
     * ignore SIGCHILD,  explicitly wait for them
     */
    (void) plp_signal (SIGCHLD, (plp_sigfunc_t)SIG_IGN);
    /*
     * handle rmjob requests.
     */
    job_removed = 0;
    (void) plp_signal (SIGINT, handle_remove);

reattach:			/* restart from here if printer is attached to another */
    /*
     * get the printcap entry
     */
    if (Get_pc_entry (Printer, All_pc_vars, All_pc_len) == 0) {
	log (XLOG_INFO, "trying to start non-existent Printer");
	exit (0);
    }
    /*
     * set the Debug flag with values from the printcap
     */
    if (DB && Debug == 0) {
	Debug = DB;
    }
    if (SS && *SS) {
	/*
	 * we have a spool server that is to be started
	 */
	ps = PS;
	st = ST;
	if (Set_pc_entry (SS, All_pc_vars, All_pc_len) == 0) {
	    log (XLOG_INFO, "queue %s does not exist!", SS);
	    exit (0);
	}
	PS = ps;
	ST = st;
	LO = Printer;
	SV = NULL;
    }
    if (SD == 0 || *SD == 0) {
	/*
	 * no spooling directory, not a Printer
	 */
	if (Debug > 0)
	    log (XLOG_DEBUG, "Startprinter: not a Printer");
	exit (0);
    }
    if (strsame (SD, spoold)) {		/* we've been here before */
	fatal (XLOG_INFO, "Printer %s attached to itself", Printer);
    }
    if (RM && NW && !*spoold) {
	(void) Remote_start ();
	exit (0);
    }
    chdir_SD ();
    /*
     * check to see if attached to another printerq, alter printer if attached
     */
    if (*Attach_file && ((afp = fopen (Attach_file, "r")))) {
	if (!*spoold)		/* set actual spool directory */
	    strcpy (spoold, SD);

	if (fscanf (afp, "%s", afname) == 1) {
	    if (strsame (afname, Printer)) {
		fatal (XLOG_INFO, "Printer %s attached to itself", Printer);
	    }
	    Printer = afname;
	}
	if (Debug > 2)
	    log (XLOG_INFO, "printer reattached to %s", Printer);
	fclose (afp);
	goto reattach;
    }
    if (*spoold) {		/* printer re-attached */
	SD = spoold;		/* set spool directory */
	if (!RP || !*RP)	/* set remote printer if necessary */
	    RP = Printer;
	if (!RM || !*RM)	/* set remote machine */
	    RM = FQDN;
	chdir_SD ();
    }
    /*
     * check to see if there is a daemon running
     */
    Lfd = Getlockfile (LO, &pid, (char *) 0, 0, &LO_statb);
    if (Lfd == NULL) {
	if (Debug > 2)
	    log (XLOG_DEBUG, "Startprinter: server present, pid %u", pid);
	if (chmod (LO, (int) (LO_statb.st_mode & 0777) | FORCE_REQUE) < 0) {
	    logerr (XLOG_INFO, "cannot force requeue on lockfile %s", LO);
	}
	exit (0);
    }
    if (fseek (Lfd, 0L, 0) < 0) {
	logerr_die (XLOG_INFO, "Startprinter: T1 fseek failed");
    }
    /*
     * check for unspooling enabled
     */
    if (LO_statb.st_mode & DISABLE_PRINT) {
	if (Debug > 0)
	    log (XLOG_DEBUG, "Startprinter: unspooling disabled");
	exit (0);
    }
    /*
     * handle multiple servers
     */
    if (SV && *SV) {
	Closelockfile (LO, Lfd);
	Lfd = 0;
	Start_server ();
	/*
	 * Check on the status of the queue lock file
	 */
	Lfd = Getlockfile (LO, &pid, (char *) 0, 0, &LO_statb);
	if (Lfd == NULL) {
	    if (Debug > 2)
		log (XLOG_DEBUG, "Startprinter: server present, pid %u", pid);
	    if (chmod_daemon (LO, (int) (LO_statb.st_mode & 0777) | FORCE_REQUE) < 0) {
		logerr (XLOG_INFO, "cannot force requeue on lockfile %s", LO);
	    }
	    exit (0);
	}
    }
    /*
     * set up log file
     */
    if (fseek (Lfd, 0L, 0) < 0) {
	logerr_die (XLOG_INFO, "Startprinter: T2 fseek failed");
    }
    if (!Single_printer)	/* queue debugging mode */
	Setuplog (LF, 0);

    if (fseek (Lfd, 0L, 0) < 0) {
	logerr_die (XLOG_INFO, "Startprinter: T3 fseek failed");
    }
    if (!Single_printer) {
	/*
	 * OK, you are the server, go to it. set process group to the queue handler
	 * process
	 */
	if (plp_setsid () == -1)
	    logerr_die (XLOG_INFO, "setsid failed");
    }
    Setlockfile (LO, Lfd, getpid (), "(checking queue)");
    setstatus ("Printing since %s.", Time_str ());
    setproctitle ("Printer %s: checking queue.", Printer);

    /*
     * search the spool directory for work and sort by queue order.
     */
    setproctitle ("%s: unqueueing jobs", Printer);
    if (RM) {
	unspool (Setup_xfer, Sendjob, Allsent, Send_error);
    } else {
	unspool (Printinit, Printjob, Printfinal, Printerror);
    }
    if (Debug > 2)
	log (XLOG_DEBUG, "Startprinter: work done");
    setstatus ("Ready since %s.", Time_str ());
    /* setproctitle unneeded -- we're about to exit */
    Closelockfile (LO, Lfd);
    exit (0);
}

static int
check_limits (struct plp_queue *q) {
    long limit;
    /*
     * Check the size.  The mx field in the printcap entry specifies the maximum size of
     * job which can be queued.  Here we check on the current limit on jobs to be
     * _printed_.
     * 
     * If current_limit() returns -1 then there is no limit on the size of jobs to print.
     */
    limit = current_limit ();
    if (Debug > 2)
	log (XLOG_DEBUG, "check_limits: current value %ld for %s", limit, q->q_name);
    if ((limit != -1L) && ((q->q_size / 1024) > limit)) {
	if (Debug > 2)
	    log (XLOG_DEBUG,
		 "check_limits: not doing %s, size = %ld, limit = %ld",
		 q->q_name, (q->q_size / 1024), limit);
	return 0;
    }
    return -1;
}

/*
 * do_job_now() makes the decision about whether a job should be sent to the printer now
 * on not.  Reasons for not sending a job include 1. the job is held 2. the job is larger
 * than the current print limit 3. the job's forms do not match the forms currently set
 * on the printer
 * 
 * If the printer is remote we just pass the job through.
 * This routine returns 0 when the job should not be printed; 1 when should be printed.
 */
static int
do_job_now (struct plp_queue *q) {
    if (Debug > 3)
	log (XLOG_DEBUG, "do_job_now: %s, RM = %s; value of q_held: %d",
	     q->q_name, (RM != NULL) ? RM : "NULL", q->q_held);

    /* remote printer - let the job through */
    if ((RM != NULL) && (*RM != '\0')) {
	/* But keep held jobs in the queue */
	if (q->q_held) 
	    {
		if (Debug > 3) {
		    log (XLOG_DEBUG, "do_job_now: %s RM = %s will not be printed; reason: held.",
			 q->q_name, (RM != NULL) ? RM : "NULL");
		}
		return 0;
	    } 
	else 
	    {
		if (Debug > 3) {
		    log (XLOG_DEBUG, "do_job_now: %s RM = %s will be printed.",
			 q->q_name, (RM != NULL) ? RM : "NULL");
		}
		return 1;
	    }
    }
    /* Held job - not to be printed now */
	if (q->q_held) {
	    if (Debug > 3) {
		log (XLOG_DEBUG, "do_job_now: %s will not be printed; reason: held.",
		     q->q_name);
	    }
	    return 0;
	}

#ifdef EUCS_ZOPTIONS
    if (!check_forms (q))
	return 0;
#endif

    if (LI && *LI && (!check_limits (q)))
	{
	    if (Debug > 3) {
		log (XLOG_DEBUG, "do_job_now: %s will not be printed; reason: limits.",
		     q->q_name);
	    }
	    return 0;
	}

    /*
     * If we've got this far then we haven't found a reason not to print the job.
     */
    return 1;
}

/***********************************************************
 * unspool (init, jobhandler, closer)
 *
 * do{
 *   check queue for new jobs
 *   jobdone = 0
 *   for(n = 0;n < Jobcount; ++n)
 *      if queue needs to be resorted then
 *         set jobdone and break from "for" loop;
 *      while not successful and retry count is less than a limit do
 *         init();
 *         success = jobhandler(job n);
 *     -- note: job can be busy (no action), done successfully,
 *              done unsuccessfully and should be abandoned,
 *              or done unsuccessfully and should be repeated
 *      if action was done then set jobdone and remove job
 *          else call closer();
 * }while (jobdone)
 * call closer()
 ***********************************************************/

static void
unspool (int (*init) (), int (*dojob) (), void (*closer) (),
	void (*err) ())
{
    int n;			/* job number */
    int retries;		/* number of retries */
    int success;		/* job success status */
    struct plp_queue *q;	/* job entry  */
    int jobdone;		/* job done in this pass */
    FILE *cfp;			/* control file */
    int init_done;		/* initialization done */
    int not_nows;

    /* really big loop */
    init_done = 0;
    do {
	do {
	    /*
	     * search the spool directory for more work.
	     */
    startcheck:
	    not_nows = 0;
	    Jobcount = Getq ();
	    if (Jobcount <= not_nows) {
		break;
	    }
	    if (Debug > 4)
		log (XLOG_DEBUG, "unspool: %d jobs to unqueue", Jobcount);
	    /*
	     * reset "update queue order" flag
	     */
	    if (fstat (fileno (Lfd), &LO_statb) < 0) {
		logerr_die (XLOG_NOTICE, "unspool: cannot stat %s", LO);
	    }
	    if (Debug > 5)
		log (XLOG_DEBUG, "unspool: '%s' perms 0%o", LO,
		     LO_statb.st_mode);
	    if (LO_statb.st_mode & DISABLE_PRINT) {
		if (Debug > 0)
		    log (XLOG_DEBUG, "unspool: unspooling disabled");
		break;
	    }
#ifdef HAVE_FCHMOD
	    if (fchmod (fileno (Lfd), (int) (LO_statb.st_mode & CLEAR_REQUE)) < 0)
#else
	    if (chmod (LO, (int) (LO_statb.st_mode & CLEAR_REQUE)) < 0)
#endif
	    {
		logerr_die (XLOG_NOTICE, "unspool: cannot chmod '%s'", LO);
	    }
	    jobdone = 0;
	    /*
	     * scan the queue
	     */
	    if (Debug > 4)
		log (XLOG_DEBUG, "unspool: running through jobs");

	    for (n = 0; n < Jobcount; ++n) {
		if (fstat (fileno (Lfd), &LO_statb) < 0) {
		    logerr_die (XLOG_NOTICE, "unspool: cannot stat %s", LO);
		}
		/* check to see if you should re-examine the queue */
		if ((LO_statb.st_mode & FORCE_REQUE) ||
		    (LO_statb.st_mode & DISABLE_PRINT)) {
		    if (Debug > 5)
			log (XLOG_DEBUG, "unspool: rechecking queue");
		    goto startcheck;
		}
		q = &Queue[n];

		/*
		 * leave until other jobs have gone through, forms are fitted, etc.
		 */
		if (!do_job_now (q)) {
		    if (Debug > 5) 
			log (XLOG_DEBUG, "job %s: can't do now", q->q_name);
		    not_nows++;
		    continue;
		}
		/*
		 * lock the control file
		 */
		if ((cfp = Lockcf (q->q_name)) == NULL) {
		    if (Debug > 4) 
			log (XLOG_DEBUG, "job %s: already being processed", q->q_name);
		    continue;
		}
		Setlockfile (LO, Lfd, getpid (), q->q_name);
		/*
		 * try printing this file until successful, number of attempts
		 * exceeds limit, or abort try sending indefinately until successful
		 */
		success = JFAIL;

		/*
		 * If RT is 0 then retry forever (this is needed for pad printers).
		 *
		 * Paul Haldane says: FYI - pad printers are
		 * printers connected to X.25 terminal lines - we
		 * used to drive almost all of our printers that way -
		 * now we have only one or two - worked by making
		 * call to the printer's X.25 address - as many
		 * machines could drive the same printer it is quite
		 * normal to have many retries.
		 */
		for (retries = 0; (success == JFAIL || success == JFULL)
				      && retries <= RT; ++retries)
		    {
			if (job_removed) {
			    if (Debug > 2)
				log (XLOG_DEBUG, "job %s: removed", q->q_name);
			    break;	/* out of for loop */
			}
			    
			if (success == JFULL) {
			    /* remote fs is full -- sleep 10 mins and retry */
			    sleep (600);
			    init_done = 0;
			}
			/*
			 * STDIO file error reset, needed on some implementations
			 */
			clearerr (cfp);
			/*
			 * seek to the start of the file
			 */
		    if (fseek (cfp, 0L, 0) < 0) {
			logerr_die (XLOG_INFO, "unspool: fseek failed (%s)",
				    q->q_name);
		    }
		    /*
		     * update status information
		     */
		    if (Debug)
			log (XLOG_DEBUG, "job %s: starting, attempt %d...",
			     q->q_name, retries);
		    /*
		     * initialize
		     */
		    if (init_done == 0) {
			init_done = (*init) ();
			if (init_done != JSUCC) {
			    setstatus ("Can't initialize device: %s.", Last_errormsg);
			    return;
			}
		    }
		    setstatus ("Printing (started at %s, attempt %d).",
			       Time_str (), retries + 1);
		    /*
		     * try to print the job
		     */
		    if (Debug > 2)
			log (XLOG_DEBUG,
			     "job %s: starting unqueue", q->q_name);

		    success = (*dojob) (cfp, q);

		    if (Debug > 2)
			log (XLOG_DEBUG,
			     "job %s: retry %d (of %d): status %d",
			     q->q_name, retries + 1, RT, success);

		    if (success == JFAIL) {
			init_done = 0;

		    } else if (success == JNOCONF) {
			/* we didn't get a final confirm; job has 
			 * probably printed, but the link is closed
			 * anyway, so we'll need to re-open it.
			 */
			init_done = 0;
			success = JSUCC;
		    }
		}

		if (job_removed) {
		    job_removed = 0;

		} else {
		    if (Debug > 2)
			log (XLOG_DEBUG,
			     "job %s: out of retries: status %d",
			     q->q_name, success);

		    if ((success == JFAIL) || (success == JFULL)) {
			if (Debug > 2)
			    log (XLOG_DEBUG,
				 "job %s: failed to print", q->q_name);
			(*err) ();
			init_done = 0;
		    } else {
			if (Debug > 2)
			    log (XLOG_DEBUG,
				 "job %s: printed successfully", q->q_name);
		    }
		    /* send mail if user asks for it or if job failed */
		    if (MAILNAME[0] || (success != JSUCC)) {
			sendmail (q, success);
		    }
		}
		jobdone = 1;
		Remove_job (cfp, q);
		(void) fclose (cfp);
		Reapchild (0);		/* collect any garbage */
	    }
	} while (jobdone);

	Reapchild (0);			/* collect any garbage */
	sleep (10);

    /* we break at the top of the loop; using Jobcount here,
     * instead of Getq(), saves a lot of CPU time.
     */
    } while (Jobcount > not_nows);

    Reapchild (0);			/* you should know by now */
    /*
     * queue finished
     */
    if (Debug > 0)
	log (XLOG_DEBUG, "unspool: no more work to do");
    setstatus ("Ready since %s.", Time_str ());
    (*closer) ();
}

/***************************************************************************
 * Start_server()
 *
 * start multiple servers for a single queue. The SV printcap parameter
 * has a list of the servers. These servers use the spool queues
 * directory, and have a lock file in the server directory.
 *
 * This routine will extract the names of each server, fork a process
 * for it, and then will read the printcap information for the server
 * from the queue.
 *
 * The printer names are separated by commas.
 ***************************************************************************/

static void
Start_server (void) {
    char *ps;
    char *last;
    pid_t pid;
    char parent = 0;

/* Changed logic of this function so that process for each printer
 * is child of main daemon (which immediately exits here).  Previous
 * code lead to first n-1 processes being children of last one and if
 * this last one exits any of the others who are sleeping get a HUP
 * and exit.  I don't understand why this hasn't been a problem up
 * till now.
 *                   -- Paul Haldane (EUCS)
 */

    last = 0;
    ps = SV;
    while (ps) {
	Printer = ps;
	if ((last = (char *) strchr (ps, ','))) {
	    *last = 0;
	    ps = last + 1;
	} else {
	    ps = 0;
	    parent = 1;
	}
	if ((pid = fork ()) < 0) {
	    logerr_die (XLOG_INFO, "Start_server: fork failed");
	} else if (pid == 0) {
	    ps = 0;
	    parent = 0;
	}
    }
    if (parent) {
	exit (0);
    }
    if (Set_pc_entry (Printer, All_pc_vars, All_pc_len) == 0) {
	fatal (XLOG_INFO, "trying to start non-existent server");
    }
    LO = Printer;
    if (Debug > 3)
	log (XLOG_DEBUG, "Start_server: started");
}

#ifdef EUCS_FORMS
int 
opt_match (char *opt, char *loaded) {
    /*
     * It is possible for a printer to be capable of accepting jobs for more than one
     * forms type at a time.  The most likely example would be a printer with headed
     * paper loaded in one bin and plain in the other.  We want to be able to send both
     * plain and headed jobs to this printer.  To do this we have introduced the concept
     * of multiply loaded forms.  If the loaded forms spec is of the form form1/form2
     * then jobs of type form1 and form2 can be printed.
     * 
     * This now copes with more than two forms, eg a/b/c
     */
    char *lf, *form;

    lf = strdup (loaded);
    lf[39] = '\0';

    form = strtok (lf, "/");
    while (form != NULL) {
	if (strsame (opt, form)) {
	    free (lf);
	    return (0);
	}
	form = strtok (NULL, "/");
    }
    free (lf);
    return (1);
}

static int
check_forms (struct plp_queue *q) {
    char *current_options ();
    char *options;		/* current printer options (= forms) */

    /*
     * Print the job if
     * 
     * 1. neither the job nor the printer specify special options or 2. they both do and
     * they're the same
     */
    if (FL && *FL && (options = current_options ())) {
	if (!((q->q_options == NULL &&
	       (options == NULL || strsame (options, DEF_OPTIONS))) ||
	      (q->q_options != NULL &&
	       options != NULL && !opt_match (q->q_options, options)))) {
	    if (Debug > 2)
		log (XLOG_DEBUG,
		     "unspool: not doing %s, options = %s, current = %s",
		     q->q_name,
		     q->q_options ? q->q_options : "NULL",
		     options ? options : "NULL");
	    return 0;
	}
    }
}

#endif				/* EUCS_FORMS */

int
Job_removed (void) {
    if (job_removed && Debug) {
        log (XLOG_DEBUG, "current job is lprmed: faking successful xfer");
    }
    return (job_removed);
}

extern void Print_intr ();

/*
 * handle 'remove-job' signals. Used to signal an lpd to stop
 * processing the current job, and to skip it.
 */
plp_signal_t
handle_remove (int signal) {
    Print_intr ();	/* in case we're printing */
    job_removed = 1;
#ifdef SVR4
    plp_signal (signal, handle_remove);
#endif
}
