/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: banner.c
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "LPD/banner.h"

extern long jobsize;		/* set in Printjob() - used in longbanner() */

/*
 * Print a banner The large character fonts have been stolen from the BANNER program,
 * which stole them from an old card deck...
 * 
 * banner( fd ) will print the banner 1. form feed if needed 2. print banner 3. another form
 * feed
 */

static int linecnt;
static int fail;
extern char chartable[][HEIGHT];
static char *bline;
static unsigned blen;
static int fd;

static void shortbanner (), longbanner (), Out_line (), breakline (), do_char (),
#ifdef EUCS_BANNER
     Out_ff (), longbanner_EUCS (),
#endif
     do_char_2 (), bigprint (), bigprint_2 (), longbanner_PLP ();

int banner(int pr) {
    linecnt = 0;
    /*
     * print the banner
     */

    fd = pr;
    fail = 0;
    if (bline == 0) {
	blen = BUFSIZ;
	if ((bline = (char *) malloc (blen)) == 0) {
	    logerr_die (XLOG_INFO, Malloc_failed_msg);
	}
    }
    if (blen < PW + 1) {
	blen = PW + 2;
	if ((bline = (char *) realloc (bline, blen)) == 0) {
	    logerr_die (XLOG_INFO, Malloc_failed_msg);
	}
    }
    if (SB) {
	shortbanner ();
	if (DbgLocal > 4)
	    log (XLOG_DEBUG, "banner: try to print shortbanner");
    } else {
	longbanner ();
	if (DbgLocal > 4)
	    log (XLOG_DEBUG, "banner: try to print longbanner");
    }
    if (DbgLocal > 4)
	log (XLOG_DEBUG, "banner: printed, fail %d", fail);
    if (fail) {
	return (JFAIL);
    } else {
	return (JSUCC);
    }
}

/*
 * shortbanner single line, has format:
 *
 * adobe:shore  Job: test.data  Date: Tue Sep 18 16:22:33 1984 ^Y^A
 */
static void
shortbanner (void) {
#ifdef EUCS_BANNER
    if (EUCS_banner_format) {
	/* make short banner look like Sun etc short banner */
	if (strstr (CLASSNAME, MAGIC_STRING) != NULL)
	    (void) sprintf (bline, "%s:%s Job: %s  Date: %s", CLASSNAME, LOGNAME,
			    JOBNAME, Time_str ());
	else
	    (void) sprintf (bline, "%s:%s Job: %s  Date: %s\031\001",
			    FROMHOST, LOGNAME, JOBNAME, Time_str ());
	Out_ff ();
    } else
#endif
    {
	(void) sprintf (bline, "%s:%s Job: %s Date: %s\031\001", LOGNAME,
			FROMHOST, JOBNAME, Time_str());
	Out_line ();
    }
}

/*
 * userinfo just print the information
 */
static void
userinfo (void) {
    (void) sprintf (bline, "User: %s@%s", LOGNAME, FROMHOST);
    Out_line ();
    (void) sprintf (bline, "Name: %s", BNRNAME);
    Out_line ();
    (void) sprintf (bline, "Date: %s", Time_str ());
    Out_line ();
    (void) sprintf (bline, "Job: %s", JOBNAME);
    Out_line ();
    (void) sprintf (bline, "Class: %s", CLASSNAME);
    Out_line ();
}

/*
 * long banner get fancy and print separators, etc.
 */
#ifdef EUCS_BANNER
static void
longbanner_EUCS (void) {
    char files[FILES_LENGTH + 1];
    char username[USER_INFO_LENGTH + 1];
    char delivery[DELIVERY_LENGTH + 1];
    char options[OPTIONS_LENGTH + 1];
    char erccreg[ERCCREG_LENGTH + 1];
    char account[ACCOUNT_LENGTH + 1];

    interpret_job (JOBNAME, files, username, delivery, options, erccreg, account);
    if (strstr (options, "a4") != NULL)
	PW = 80;

    breakline ('*');
    (void) sprintf (bline, "*%*c*", PW - 2, ' ');
    Out_line ();

    if (PW <= 80) {
	int remaining = PW - 34;
	(void) sprintf (bline, "* %-14.14s %-*.*s %-*.*s %-14.14s *",
			FROMHOST, remaining / 2, remaining / 2, username,
			remaining - (remaining / 2 + 1), remaining - (remaining / 2 + 1),
			LOGNAME, FROMHOST);

	Out_line ();
	(void) sprintf (bline, "* %-14.14s %-*.*s %-14.14s *",
			FROMHOST, remaining, remaining, Time_str (), FROMHOST);
	Out_line ();
	(void) sprintf (bline, "* %-14.14s DELIVER TO - %-*.*s %-14.14s *",
			FROMHOST, PW - 47, PW - 47, delivery, FROMHOST);
	Out_line ();
	(void) sprintf (bline, "* %-14.14s %-*.*s %-14.14s *",
			FROMHOST, remaining, remaining, files, FROMHOST);
	Out_line ();
	(void) sprintf (bline, "* %-14.14s %9dk%*s %-14.14s *",
			FROMHOST, jobsize, remaining - 10, "", FROMHOST);
	Out_line ();
    } else {
	(void) sprintf (bline,
		 "** %-15.15s %-15.15s %-20.20s %-11.11s %-29.29s %-15.15s %-15.15s **",
		FROMHOST, FROMHOST, username, LOGNAME, Time_str (), FROMHOST, FROMHOST);
	Out_line ();
	(void) sprintf (bline,
		      "** %-15.15s %-15.15s DELIVER TO - %-49.49s %-15.15s %-15.15s **",
			FROMHOST, FROMHOST, delivery, FROMHOST, FROMHOST);
	Out_line ();
	(void) sprintf (bline, "** %-15.15s %-15.15s %-62.62s %-15.15s %-15.15s **",
			FROMHOST, FROMHOST, files, FROMHOST, FROMHOST);
	Out_line ();
	(void) sprintf (bline, "** %-15.15s %-15.15s %10dk%51s %-15.15s %-15.15s **",
			FROMHOST, FROMHOST, jobsize, "", FROMHOST, FROMHOST);
	Out_line ();
    }

    (void) sprintf (bline, "*%*c*", PW - 2, ' ');
    Out_line ();
    breakline ('*');
    (void) strcpy (bline, " ");
    Out_ff ();
}

#endif

static void
longbanner_PLP (void) {
    int i;			/* ACME integers, INC */

    for (i = 0; i < BLAST; ++i) {
	breakline ('*');
    }
    for (i = 0; i < SEP; ++i) {
	breakline (0);
    }
    /*
     * print the Name, Host and Jobname in BIG letters
     */
    bigprint (LOGNAME);
    bigprint (FROMHOST);
    bigprint_2 (JOBNAME); /* Comment this line if you don't like the jobname here */
    userinfo ();          /* And comment out then this for loop! */
    /*    for (i = 0; i < SEP; ++i) {
	breakline (0);
    } */

    if (PL > 0) {
	/* separator from banner */
	while ((linecnt + BLAST) % PL) {
	    breakline (0);
	}
	/* page bottom marker */
	while (linecnt % PL) {
	    breakline ('*');
	}
    }
}

static void
longbanner (void) {
#ifdef EUCS_BANNER
    if (EUCS_banner_format) {
	longbanner_EUCS ();
    } else
#endif
    {
	longbanner_PLP ();
    }
}

static void
breakline (int c) {
    int i;			/* ACME Integers, Inc. */

    i = 0;
    if (c) {
	for (i = 0; i < PW; ++i) {
	    bline[i] = c;
	}
    }
    bline[i] = 0;
    Out_line ();
}

/***************************************************************************
 * bigprint( char * line )
 * print the line in big characters
 * for i = topline to bottomline do
 *  for each character in the line do
 *    get the scan line representation for the character
 *    foreach bit in the string do
 *       if the bit is set, print X, else print ' ';
 *	  endfor
 *  endfor
 * endfor
 ***************************************************************************/

static void
bigprint (char *line) {
    int i, j;			/* ACME Integers, Inc. */
    int cnt;			/* number of characters */

    if (DbgLocal > 5)
	log (XLOG_DEBUG, "bigprint: '%s'", line);
    cnt = strlen (line);
    i = PW / WIDTH;
    if (cnt > i) {
	cnt = i;
    }
    for (i = 0; i < HEIGHT + DROP; ++i) {
	for (j = 0; j < cnt; ++j) {
	    do_char (line[j], i, bline + (j * WIDTH));
	}
	bline[j * WIDTH] = 0;
	Out_line ();
    }
}

/* Second version */
static void
bigprint_2 (char *line) {
    int i, j;			/* ACME Integers, Inc. */
    int cnt;			/* number of characters */

    if (DbgLocal > 5)
	log (XLOG_DEBUG, "bigprint: '%s'", line);
    cnt = strlen (line);
    i = PW / WIDTH;
    if (cnt > i) {
	cnt = i;
    }
    for (i = 0; i < HEIGHT + DROP; ++i) {
	for (j = 0; j < cnt; ++j) {
	    do_char_2 (line[j], i, bline + (j * WIDTH));
	}
	bline[j * WIDTH] = 0;
	Out_line ();
    }
}

/***************************************************************************
 * do_char( c, line, buf )
 *  print out the raster line corresponding to character c, line l
 *  If a character is not a descender, you can use the direct mapping.
 *  If it is a descender,  you have to shift down by the descender amount.
 *  1.  for lines above the "top" of a descender line
 *      if the character is a descender, print a space
 *      otherwise print normally
 *  2.  for lines at and below the top of a descender
 *      if the character is a descender, print line
 *      otherwise print normally
 *  3.  for lines below the baseline
 *      if a descender, print line
 *      otherwise print blank
 ***************************************************************************/
static int
is_descender (char c) {
    switch (c) {
    case '_':
    case ';':
    case ',':
    case 'g':
    case 'j':
    case 'p':
    case 'q':
    case 'y':
	return (1);
    default:
	break;
    }
    return (0);
}

static void
do_char (int c, int line, char *buf) {
    int i;			/* ACME Integer, Inc. */
    int position;		/* index into chartable */
    int rep;			/* representation of character scan line */

    if (!isprint (c)) {
	c = ' ';
    }
    if (DbgLocal > 5)
	log (XLOG_DEBUG, "do_char: %c, line %d", c, line);
    position = c - ' ';
    if (is_descender (c)) {
	if (line >= DROP) {
	    line = line - DROP;
	} else {
	    position = 0;
	    line = 0;
	}
    }
    if (line >= HEIGHT) {
	position = 0;
	line = 0;
    }
    rep = chartable[position][line];
    if (DbgLocal > 5)
	log (XLOG_DEBUG, "do_char: position %d, line %d, rep 0x%x",
	     position, line, rep);
    for (i = 6; i >= 0; --i) {
	if (rep & (1 << i)) {
	    buf[6 - i] = 'o';
	} else {
	    buf[6 - i] = ' ';
	}
    }
    for (i = 7; i < WIDTH; ++i) {
	buf[i] = ' ';
    }
}

static void
do_char_2 (int c, int line, char *buf) {
    int i;			/* ACME Integer, Inc. */
    int position;		/* index into chartable */
    int rep;			/* representation of character scan line */

    if (!isprint (c)) {
	c = ' ';
    }
    if (DbgLocal > 5)
	log (XLOG_DEBUG, "do_char: %c, line %d", c, line);
    position = c - ' ';
    if (is_descender (c)) {
	if (line >= DROP) {
	    line = line - DROP;
	} else {
	    position = 0;
	    line = 0;
	}
    }
    if (line >= HEIGHT) {
	position = 0;
	line = 0;
    }
    rep = chartable[position][line];
    if (DbgLocal > 5)
	log (XLOG_DEBUG, "do_char: position %d, line %d, rep 0x%x",
	     position, line, rep);
    for (i = 6; i >= 0; --i) {
	if (rep & (1 << i)) {
	    buf[6 - i] = '#';
	} else {
	    buf[6 - i] = ' ';
	}
    }
    for (i = 7; i < WIDTH; ++i) {
	buf[i] = ' ';
    }
}


/***************************************************************************
 * write the buffer out to the file descriptor.
 * don;t do if fail is invalid.
 ***************************************************************************/
static void
Out_line (void) {
    int i;
    i = strlen (bline);

    if (fail == 0) {
	if ((i > 0 && write (fd, bline, i) != i) || write (fd, "\n", 1) != 1) {
	    logerr (XLOG_INFO, "Out_line: write failed");
	    fail = 1;
	}
    }
    ++linecnt;
}

#ifdef EUCS_BANNER
/***************************************************************************
 * write the buffer out to the file descriptor.
 * don;t do if fail is invalid.
 ***************************************************************************/
static void
Out_ff (void) {
    int i;
    i = strlen (bline);

    if (fail == 0) {
	if ((i > 0 && write (fd, bline, i) != i)
	    || (write (fd, "\n", 1) != 1)
	    || (write (fd, "\f", 1) != 1)) {
	    logerr (XLOG_INFO, "Out_ff: write failed");
	    fail = 1;
	}
    }
    linecnt = 0;
}

#endif
