r"""
Multiplicative Abelian Groups

This module lets you compute with finitely generated Abelian groups of the form

.. MATH::

    G = \ZZ^r \oplus \ZZ_{k_1} \oplus \cdots \oplus \ZZ_{k_t}

It is customary to denote the infinite cyclic group `\ZZ` as having
order `0`, so the data defining the Abelian group can be written as an
integer vector

.. MATH::

    \vec{k} = (0, \dots, 0, k_1, \dots, k_t)

where there are `r` zeroes and `t` nonzero values. To construct this
Abelian group in Sage, you can either specify all entries of `\vec{k}`
or only the nonzero entries together with the total number of
generators::

    sage: AbelianGroup([0,0,0,2,3])
    Multiplicative Abelian group isomorphic to Z x Z x Z x C2 x C3
    sage: AbelianGroup(5, [2,3])
    Multiplicative Abelian group isomorphic to Z x Z x Z x C2 x C3

It is also legal to specify `1` as the order. The corresponding
generator will be the neutral element, but it will still take up an
index in the labelling of the generators::

    sage: G = AbelianGroup([2,1,3], names='g')
    sage: G.gens()
    (g0, 1, g2)

Note that this presentation is not unique, for example
`\ZZ_6 \cong \ZZ_2 \times \ZZ_3`. The orders of the generators
`\vec{k}=(0,\dots,0,k_1,\dots, k_t)` has previously been called
invariants in Sage, even though they are not necessarily the (unique)
invariant factors of the group. You should now use
:meth:`~AbelianGroup_class.gens_orders` instead::

    sage: J = AbelianGroup([2,0,3,2,4]);  J
    Multiplicative Abelian group isomorphic to C2 x Z x C3 x C2 x C4
    sage: J.gens_orders()            # use this instead
    (2, 0, 3, 2, 4)
    sage: J.invariants()             # deprecated
    (2, 0, 3, 2, 4)
    sage: J.elementary_divisors()    # these are the "invariant factors"
    (2, 2, 12, 0)
    sage: for i in range(J.ngens()):
    ....:     print((i, J.gen(i), J.gen(i).order()))     # or use this form
    (0, f0, 2)
    (1, f1, +Infinity)
    (2, f2, 3)
    (3, f3, 2)
    (4, f4, 4)

Background on invariant factors and the Smith normal form
(according to section 4.1 of [Cohen1]_): An abelian group is a
group `A` for which there exists an exact sequence
`\ZZ^k \rightarrow \ZZ^\ell \rightarrow A \rightarrow 1`,
for some positive integers
`k,\ell` with `k\leq \ell`. For example, a finite abelian group has a
decomposition

.. MATH::

    A = \langle a_1\rangle \times \dots \times  \langle a_\ell\rangle ,

where `\mathrm{ord}(a_i)=p_i^{c_i}`, for some primes `p_i` and some
positive integers `c_i`, `i=1,...,\ell`. GAP calls the
list (ordered by size) of the `p_i^{c_i}` the *abelian invariants*.
In Sage they will be called *invariants*.
In this situation, `k=\ell` and `\phi:  \ZZ^\ell \rightarrow A` is the map
`\phi(x_1,...,x_\ell) = a_1^{x_1}...a_\ell^{x_\ell}`,
for `(x_1,...,x_\ell)\in \ZZ^\ell`. The matrix of relations
`M:\ZZ^k \rightarrow \ZZ^\ell` is the matrix
whose rows generate the kernel of `\phi` as a `\ZZ`-module.
In other words, `M=(M_{ij})` is a `\ell\times \ell`
diagonal matrix with `M_{ii}=p_i^{c_i}`. Consider now the
subgroup `B\subset A` generated by
`b_1 = a_1^{f_{1,1}}...a_\ell^{f_{\ell,1}}`, ...,
`b_m = a_1^{f_{1,m}}...a_\ell^{f_{\ell,m}}`.
The kernel of the map `\phi_B:  \ZZ^m \rightarrow B` defined by
`\phi_B(y_1,...,y_m) = b_1^{y_1}...b_m^{y_m}`,
for `(y_1,...,y_m)\in \ZZ^m`, is the kernel of the matrix

.. MATH::

    F=
    \left(
    \begin{array}{cccc}
    f_{11} & f_{12} & \dots & f_{1m}\\
    f_{21} & f_{22} & \dots & f_{2m}\\
    \vdots &        & \ddots & \vdots \\
    f_{\ell,1} & f_{\ell,2} & \dots & f_{\ell,m}
    \end{array}
    \right),

regarded as a map
`\ZZ^m\rightarrow (\ZZ/p_1^{c_1}\ZZ)\times ...\times (\ZZ/p_\ell^{c_\ell}\ZZ)`.
In particular, `B\cong \ZZ^m/\ker(F)`. If `B=A` then the
Smith normal form (SNF) of a generator matrix of
`\ker(F)` and the SNF of `M` are the same. The diagonal entries `s_i` of the
SNF `S = \mathrm{diag}[s_1,s_2,s_3, ... s_r,0,0,...0]`,
are called *determinantal divisors* of `F`.
where `r` is the rank. The *invariant factors* of `A` are:

.. MATH::

    s_1, s_2/s_1, s_3/s_2, ... s_r/s_{r-1}.

Sage supports multiplicative abelian groups on any prescribed finite
number `n \geq 0` of generators.  Use the :func:`AbelianGroup`
function to create an abelian group, and the
:meth:`~AbelianGroup_class.gen` and :meth:`~AbelianGroup_class.gens`
methods to obtain the corresponding generators.  You can print the
generators as arbitrary strings using the optional ``names`` argument
to the :func:`AbelianGroup` function.

EXAMPLE 1:

We create an abelian group in zero or more variables; the syntax ``T(1)``
creates the identity element even in the rank zero case::

    sage: T = AbelianGroup(0, [])
    sage: T
    Trivial Abelian group
    sage: T.gens()
    ()
    sage: T(1)
    1

EXAMPLE 2:

An Abelian group uses a multiplicative representation of elements, but
the underlying representation is lists of integer exponents::

    sage: F = AbelianGroup(5, [3,4,5,5,7], names = list("abcde"))
    sage: F
    Multiplicative Abelian group isomorphic to C3 x C4 x C5 x C5 x C7
    sage: (a,b,c,d,e) = F.gens()
    sage: a*b^2*e*d
    a*b^2*d*e
    sage: x = b^2*e*d*a^7
    sage: x
    a*b^2*d*e
    sage: x.list()
    [1, 2, 0, 1, 1]

REFERENCES:

.. [Cohen1] \H. Cohen, Advanced topics in computational number theory,
  Springer, 2000.

.. [Cohen2] \H. Cohen, A course in computational algebraic number theory,
  Springer, 1996.

.. [Rotman] \J. Rotman, An introduction to the theory of
  groups, 4th ed, Springer, 1995.

.. WARNING::

   Many basic properties for infinite abelian groups are not
   implemented.

AUTHORS:

- William Stein, David Joyner (2008-12): added (user requested) is_cyclic,
  fixed elementary_divisors.

- David Joyner (2006-03): (based on free abelian monoids by David
  Kohel)

- David Joyner (2006-05) several significant bug fixes

- David Joyner (2006-08) trivial changes to docs, added random, fixed
  bug in how invariants are recorded

- David Joyner (2006-10) added dual_group method

- David Joyner (2008-02) fixed serious bug in word_problem

- David Joyner (2008-03) fixed bug in trivial group case

- David Loeffler (2009-05) added subgroups method

- Volker Braun (2012-11) port to new Parent base. Use tuples for
  immutables. Rename invariants to gens_orders.
"""

# ****************************************************************************
#       Copyright (C) 2006 William Stein <wstein@gmail.com>
#       Copyright (C) 2006 David Joyner <wdjoyner@gmail.com>
#       Copyright (C) 2012 Volker Braun <vbraun.name@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#                  https://www.gnu.org/licenses/
# ****************************************************************************
from itertools import product

from sage.arith.functions import lcm
from sage.arith.misc import divisors, gcd
from sage.categories.groups import Groups
from sage.groups.abelian_gps.abelian_group_element import AbelianGroupElement
from sage.groups.group import AbelianGroup as AbelianGroupBase
from sage.matrix.constructor import matrix
from sage.matrix.special import diagonal_matrix
from sage.misc.cachefunc import cached_method
from sage.misc.misc_c import prod
from sage.misc.mrange import mrange
from sage.modules.free_module_element import vector
from sage.rings.infinity import infinity
from sage.rings.integer import Integer
from sage.rings.integer_ring import ZZ
from sage.structure.category_object import normalize_names
from sage.structure.unique_representation import UniqueRepresentation


# .. TODO::

#     this uses perm groups - the AbelianGroupElement instance method
#     uses a different implementation.
def word_problem(words, g, verbose=False):
    r"""
    `G` and `H` are abelian, `g` in `G`, `H` is a subgroup of `G` generated by
    a list (words) of elements of `G`. If `g` is in `H`, return the expression
    for `g` as a word in the elements of (words).

    The 'word problem' for a finite abelian group `G` boils down to the
    following matrix-vector analog of the Chinese remainder theorem.

    Problem: Fix integers `1<n_1\leq n_2\leq ...\leq n_k`
    (indeed, these `n_i` will all be prime powers), fix a
    generating set `g_i=(a_{i1},...,a_{ik})` (with
    `a_{ij}\in \mathrm{Z}/n_j\mathrm{Z}`), for `1\leq i\leq \ell`,
    for the group `G`, and let `d = (d_1,...,d_k)` be
    an element of the direct product
    `\mathrm{Z}/n_1\mathrm{Z} \times ...\times \mathrm{Z}/n_k\mathrm{Z}`. Find, if they
    exist, integers `c_1,...,c_\ell` such that
    `c_1g_1+...+c_\ell g_\ell = d`. In other words, solve
    the equation `cA=d` for `c\in \mathrm{Z}^\ell`, where
    `A` is the matrix whose rows are the `g_i`'s. Of
    course, it suffices to restrict the `c_i`'s to the range
    `0\leq c_i\leq N-1`, where `N` denotes the least
    common multiple of the integers `n_1,...,n_k`.

    This function does not solve this directly, as perhaps it should.
    Rather (for both speed and as a model for a similar function valid
    for more general groups), it pushes it over to GAP, which has optimized
    (non-deterministic) algorithms for the word problem. Essentially,
    this function is a wrapper for the GAP function 'Factorization'.

    EXAMPLES::

        sage: # needs sage.libs.gap
        sage: G.<a,b,c> = AbelianGroup(3, [2,3,4]); G
        Multiplicative Abelian group isomorphic to C2 x C3 x C4
        sage: w = word_problem([a*b,a*c], b*c); w       # random
        [[a*b, 1], [a*c, 1]]
        sage: prod([x^i for x,i in w]) == b*c
        True
        sage: w = word_problem([a*c,c], a); w           # random
        [[a*c, 1], [c, -1]]
        sage: prod([x^i for x,i in w]) == a
        True
        sage: word_problem([a*c,c], a, verbose=True)    # random
        a = (a*c)^1*(c)^-1
        [[a*c, 1], [c, -1]]

    ::

        sage: # needs sage.libs.gap
        sage: A.<a,b,c,d,e> = AbelianGroup(5, [4, 5, 5, 7, 8])
        sage: b1 = a^3*b*c*d^2*e^5
        sage: b2 = a^2*b*c^2*d^3*e^3
        sage: b3 = a^7*b^3*c^5*d^4*e^4
        sage: b4 = a^3*b^2*c^2*d^3*e^5
        sage: b5 = a^2*b^4*c^2*d^4*e^5
        sage: w = word_problem([b1,b2,b3,b4,b5], e); w  # random
        [[a^3*b*c*d^2*e^5, 1], [a^2*b*c^2*d^3*e^3, 1],
         [a^3*b^3*d^4*e^4, 3], [a^2*b^4*c^2*d^4*e^5, 1]]
        sage: prod([x^i for x,i in w]) == e
        True
        sage: word_problem([a,b,c,d,e], e)
        [[e, 1]]
        sage: word_problem([a,b,c,d,e], b)
        [[b, 1]]

    .. WARNING::

       1. Might have unpleasant effect when the word problem
          cannot be solved.

       2. Uses permutation groups, so may be slow when group is large.
          The instance method word_problem of the class
          AbelianGroupElement is implemented differently (wrapping
          GAP's 'EpimorphismFromFreeGroup' and
          'PreImagesRepresentative') and may be faster.
    """
    from sage.libs.gap.libgap import libgap
    A = libgap.AbelianGroup(g.parent().gens_orders())
    gens = libgap.GeneratorsOfGroup(A)

    gap_g = libgap.Product([gi**Li for gi, Li in zip(gens, g.list())])

    gensH = [libgap.Product([gi**Li for gi, Li in zip(gens, w.list())])
             for w in words]
    H = libgap.Group(gensH)

    x = libgap.Factorization(H, gap_g)

    resu = libgap.ExtRepOfObj(x).sage()  # (indice, power, indice, power, etc)
    indices = resu[0::2]
    powers = resu[1::2]
    if verbose:
        v = '*'.join('(%s)^%s' % (words[indi - 1], powi)
                     for indi, powi in zip(indices, powers))
        print('%s = %s' % (g, v))
    return [[words[indi - 1], powi] for indi, powi in zip(indices, powers)]


def _normalize(n, gens_orders=None, names='f'):
    """
    Helper function for :func:`AbelianGroup`. Beat some sense into the
    arguments.

    This function is also used by some descendents of
    :class:`AbelianGroup_class`.

    INPUT:

    See :func:`AbelianGroup`.

    OUTPUT:

    Unique data for defining a :class:`AbelianGroup_class`. Raises an
    exception if the input is invalid.

    EXAMPLES::

        sage: from sage.groups.abelian_gps.abelian_group import _normalize
        sage: _normalize(5, [2,1,0], names='abc')
        ((0, 0, 2, 1, 0), 'abc')
        sage: _normalize(5, (2.0, 1.0, 0/1), names=list('abc'))
        ((0, 0, 2, 1, 0), ('a', 'b', 'c'))
        sage: _normalize([0,2,1,0], names='a')
        ((0, 2, 1, 0), 'a')

    TESTS::

        sage: _normalize(5, (2.0, 1.5, 0/1), names=list('abc'))
        Traceback (most recent call last):
        ...
        TypeError: Attempt to coerce non-integral RealNumber to Integer
        sage: _normalize('1', '[2]', names='a')
        Traceback (most recent call last):
        ...
        TypeError: unable to convert '[' to an integer
        sage: _normalize(3, 'str', names='a')
        Traceback (most recent call last):
        ...
        TypeError: unable to convert 's' to an integer

    Verify that :issue:`38967` is fixed::

        sage: AbelianGroup([-4])
        Traceback (most recent call last):
        ...
        ValueError: orders of generators cannot be negative but they are (-4,)
    """
    if gens_orders is None:
        if isinstance(n, (list, tuple)):
            gens_orders = n
            n = len(n)
        else:
            gens_orders = []
    n = ZZ(n)
    if len(gens_orders) < n:
        gens_orders = [0] * (n - len(gens_orders)) + list(gens_orders)
    gens_orders = tuple(ZZ(i) for i in gens_orders)
    if any(i < 0 for i in gens_orders):
        raise ValueError(f'orders of generators cannot be negative but they are {gens_orders}')
    if len(gens_orders) > n:
        raise ValueError('gens_orders (='+str(gens_orders)+') must have length n (='+str(n)+')')
    if isinstance(names, list):
        names = tuple(names)
    return (gens_orders, names)


def AbelianGroup(n, gens_orders=None, names='f'):
    r"""
    Create the multiplicative abelian group in `n` generators
    with given orders of generators (which need not be prime powers).

    INPUT:

    - ``n`` -- integer (optional); if not specified, will be derived
      from ``gens_orders``

    - ``gens_orders`` -- list of nonnegative integers in the form
       `[a_0, a_1, \dots, a_{n-1}]`, typically written in increasing
       order. This list is padded with zeros if it has length less
       than `n`. The orders of the commuting generators, with `0`
       denoting an infinite cyclic factor.

    - ``names`` -- (optional) names of generators

    Alternatively, you can also give input in the form
    ``AbelianGroup(gens_orders, names='f')``, where the names keyword
    argument must be explicitly named.

    OUTPUT:

    Abelian group with generators and invariant type. The default name
    for generator ``A.i`` is ``fi``, as in GAP.

    EXAMPLES::

        sage: F = AbelianGroup(5, [5,5,7,8,9], names='abcde')
        sage: F(1)
        1
        sage: (a, b, c, d, e) = F.gens()
        sage: mul([ a, b, a, c, b, d, c, d ], F(1))
        a^2*b^2*c^2*d^2
        sage: d * b**2 * c**3
        b^2*c^3*d
        sage: F = AbelianGroup(3, [2]*3); F
        Multiplicative Abelian group isomorphic to C2 x C2 x C2
        sage: H = AbelianGroup([2,3], names='xy'); H
        Multiplicative Abelian group isomorphic to C2 x C3
        sage: AbelianGroup(5)
        Multiplicative Abelian group isomorphic to Z x Z x Z x Z x Z
        sage: AbelianGroup(5).order()
        +Infinity

    Notice that `0`'s are prepended if necessary::

        sage: G = AbelianGroup(5, [2,3,4]);  G
        Multiplicative Abelian group isomorphic to Z x Z x C2 x C3 x C4
        sage: G.gens_orders()
        (0, 0, 2, 3, 4)

    The invariant list must not be longer than the number of generators::

        sage: AbelianGroup(2, [2,3,4])
        Traceback (most recent call last):
        ...
        ValueError: gens_orders (=(2, 3, 4)) must have length n (=2)
    """
    gens_orders, names = _normalize(n, gens_orders, names)
    M = AbelianGroup_class(gens_orders, names)
    return M


def is_AbelianGroup(x):
    """
    Return ``True`` if ``x`` is an Abelian group.

    EXAMPLES::

        sage: from sage.groups.abelian_gps.abelian_group import is_AbelianGroup
        sage: F = AbelianGroup(5,[5,5,7,8,9], names=list("abcde")); F
        Multiplicative Abelian group isomorphic to C5 x C5 x C7 x C8 x C9
        sage: is_AbelianGroup(F)
        doctest:warning...
        DeprecationWarning: the function is_AbelianGroup is deprecated;
        use 'isinstance(..., AbelianGroup_class)' instead
        See https://github.com/sagemath/sage/issues/37898 for details.
        True
        sage: is_AbelianGroup(AbelianGroup(7, [3]*7))
        True
    """
    from sage.misc.superseded import deprecation
    deprecation(37898, "the function is_AbelianGroup is deprecated; use 'isinstance(..., AbelianGroup_class)' instead")
    return isinstance(x, AbelianGroup_class)


class AbelianGroup_class(UniqueRepresentation, AbelianGroupBase):
    """
    The parent for Abelian groups with chosen generator orders.

    .. WARNING::

        You should use :func:`AbelianGroup` to construct Abelian
        groups and not instantiate this class directly.

    INPUT:

    - ``generator_orders`` -- list of integers. The orders of the
      (commuting) generators. Zero denotes an infinite cyclic
      generator.

    - ``names`` -- names of the group generators (optional)

    EXAMPLES::

        sage: Z2xZ3 = AbelianGroup([2,3])
        sage: Z6 = AbelianGroup([6])
        sage: Z2xZ3 is Z2xZ3, Z6 is Z6
        (True, True)
        sage: Z2xZ3 is Z6
        False
        sage: Z2xZ3 == Z6
        False
        sage: Z2xZ3.is_isomorphic(Z6)
        True

        sage: F = AbelianGroup(5,[5,5,7,8,9], names=list("abcde")); F
        Multiplicative Abelian group isomorphic to C5 x C5 x C7 x C8 x C9
        sage: F = AbelianGroup(5,[2, 4, 12, 24, 120], names=list("abcde")); F
        Multiplicative Abelian group isomorphic to C2 x C4 x C12 x C24 x C120
        sage: F.elementary_divisors()
        (2, 4, 12, 24, 120)

        sage: F.category()
        Category of finite enumerated commutative groups

    TESTS::

        sage: AbelianGroup([]).gens_orders()
        ()
        sage: AbelianGroup([1]).gens_orders()
        (1,)
        sage: AbelianGroup([1,1]).gens_orders()
        (1, 1)
        sage: AbelianGroup(0).gens_orders()
        ()
    """
    Element = AbelianGroupElement

    def __init__(self, generator_orders, names, category=None):
        """
        The Python constructor.

        TESTS::

            sage: G = AbelianGroup([0,5,0,7], names=list("abcd")); G
            Multiplicative Abelian group isomorphic to Z x C5 x Z x C7
            sage: TestSuite(G).run()

        We check that :issue:`15140` is fixed::

            sage: A = AbelianGroup([3,3])
            sage: A.category()
            Category of finite enumerated commutative groups
            sage: A = AbelianGroup([3,0,7])
            sage: A.category()
            Category of infinite commutative groups
        """
        assert isinstance(names, (str, tuple))
        assert isinstance(generator_orders, tuple)
        assert all(isinstance(order, Integer) for order in generator_orders)
        self._gens_orders = generator_orders
        n = len(generator_orders)
        names = normalize_names(n, names)
        self._assign_names(names)
        if category is None:
            category = Groups().Commutative()
        if all(order > 0 for order in generator_orders):
            category = category.Finite().Enumerated()
        else:
            category = category.Infinite()
        AbelianGroupBase.__init__(self, category=category)

    def is_isomorphic(left, right):
        """
        Check whether ``left`` and ``right`` are isomorphic.

        INPUT:

        - ``right`` -- anything

        OUTPUT: boolean; whether ``left`` and ``right`` are isomorphic as abelian groups

        EXAMPLES::

            sage: G1 = AbelianGroup([2,3,4,5])
            sage: G2 = AbelianGroup([2,3,4,5,1])
            sage: G1.is_isomorphic(G2)
            True
        """
        if not isinstance(right, AbelianGroup_class):
            return False
        return left.elementary_divisors() == right.elementary_divisors()

    def is_subgroup(left, right):
        """
        Test whether ``left`` is a subgroup of ``right``.

        EXAMPLES::

            sage: G = AbelianGroup([2,3,4,5])
            sage: G.is_subgroup(G)
            True

            sage: H = G.subgroup([G.1])                                                 # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: H.is_subgroup(G)                                                      # needs sage.libs.gap  # optional - gap_package_polycyclic
            True

            sage: G.<a, b> = AbelianGroup(2)
            sage: H.<c> = AbelianGroup(1)
            sage: H < G
            False
        """
        return all(l in right for l in left.gens())

    __le__ = is_subgroup

    def __ge__(left, right):
        """
        Test whether ``right`` is a subgroup of ``left``.

        EXAMPLES::

            sage: G.<a, b> = AbelianGroup(2)
            sage: H.<c> = AbelianGroup(1)
            sage: G >= H
            False
        """
        return right <= left

    def __lt__(left, right):
        """
        Test whether ``left`` is a strict subgroup of ``right``.

        EXAMPLES::

            sage: G.<a, b> = AbelianGroup(2)
            sage: H.<c> = AbelianGroup(1)
            sage: H < G
            False
        """
        return left <= right and left != right

    def __gt__(left, right):
        """
        Test whether ``right`` is a strict subgroup of ``left``.

        EXAMPLES::

            sage: G.<a, b> = AbelianGroup(2)
            sage: H.<c> = AbelianGroup(1)
            sage: G > H
            False
        """
        return left >= right and left != right

    def is_trivial(self):
        """
        Return whether the group is trivial.

        A group is trivial if it has precisely one element.

        EXAMPLES::

            sage: AbelianGroup([2, 3]).is_trivial()
            False
            sage: AbelianGroup([1, 1]).is_trivial()
            True
        """
        return self.elementary_divisors() == ()

    def __bool__(self) -> bool:
        """
        Return ``True`` if this group is nontrivial.

        EXAMPLES::

            sage: T = AbelianGroup([2, 3])
            sage: bool(T)  # indirect doctest
            True
            sage: bool(AbelianGroup([]))
            False
            sage: bool(AbelianGroup([1,1,1]))
            False
        """
        return not self.is_trivial()

    @cached_method
    def dual_group(self, names='X', base_ring=None):
        """
        Return the dual group.

        INPUT:

        - ``names`` -- string or list of strings. The generator names
          for the dual group

        - ``base_ring`` -- the base ring; if ``None`` (default), then
          a suitable cyclotomic field is picked automatically

        OUTPUT: the :class:`dual abelian group <sage.groups.abelian_gps.dual_abelian_group.DualAbelianGroup_class>`

        EXAMPLES::

            sage: G = AbelianGroup([2])
            sage: G.dual_group()                                                        # needs sage.rings.number_field
            Dual of Abelian Group isomorphic to Z/2Z over Cyclotomic Field of order 2 and degree 1
            sage: G.dual_group().gens()                                                 # needs sage.rings.number_field
            (X,)
            sage: G.dual_group(names='Z').gens()                                        # needs sage.rings.number_field
            (Z,)

            sage: G.dual_group(base_ring=QQ)
            Dual of Abelian Group isomorphic to Z/2Z over Rational Field


        TESTS::

            sage: H = AbelianGroup(1)
            sage: H.dual_group()
            Traceback (most recent call last):
            ...
            ValueError: group must be finite
        """
        from sage.groups.abelian_gps.dual_abelian_group import DualAbelianGroup_class
        if not self.is_finite():
            raise ValueError('group must be finite')
        if base_ring is None:
            from sage.rings.number_field.number_field import CyclotomicField
            base_ring = CyclotomicField(lcm(self.gens_orders()))
        return DualAbelianGroup_class(self, names=names, base_ring=base_ring)

    @cached_method
    def elementary_divisors(self):
        r"""
        Return the elementary divisors of the group, using Pari.

        .. NOTE::

            Here is another algorithm for computing the elementary divisors
            `d_1, d_2, d_3, \ldots`, of a finite abelian group (where `d_1 | d_2 | d_3 | \ldots`
            are composed of prime powers dividing the invariants of the group
            in a way described below). Just factor the invariants `a_i` that
            define the abelian group.  Then the biggest `d_i` is the product
            of the maximum prime powers dividing some `a_j`. In other words, the
            largest `d_i` is the product of `p^v`, where
            `v = \max(\mathrm{ord}_p(a_j) \text{ for all } j`).
            Now divide out all those `p^v`'s into the list of invariants `a_i`,
            and get a new list of "smaller invariants". Repeat the above procedure
            on these "smaller invariants" to compute `d_{i-1}`, and so on.
            (Thanks to Robert Miller for communicating this algorithm.)

        OUTPUT: tuple of integers

        EXAMPLES::

            sage: G = AbelianGroup(2, [2,3])
            sage: G.elementary_divisors()
            (6,)
            sage: G = AbelianGroup(1, [6])
            sage: G.elementary_divisors()
            (6,)
            sage: G = AbelianGroup(2, [2,6])
            sage: G
            Multiplicative Abelian group isomorphic to C2 x C6
            sage: G.gens_orders()
            (2, 6)
            sage: G.elementary_divisors()
            (2, 6)
            sage: J = AbelianGroup([1,3,5,12])
            sage: J.elementary_divisors()
            (3, 60)
            sage: G = AbelianGroup(2, [0,6])
            sage: G.elementary_divisors()
            (6, 0)
            sage: AbelianGroup([3,4,5]).elementary_divisors()
            (60,)
        """
        from sage.matrix.constructor import diagonal_matrix
        ed = diagonal_matrix(ZZ, self.gens_orders()).elementary_divisors()
        return tuple(d for d in ed if d != 1)

    @cached_method
    def exponent(self):
        """
        Return the exponent of this abelian group.

        EXAMPLES::

            sage: G = AbelianGroup([2,3,7]); G
            Multiplicative Abelian group isomorphic to C2 x C3 x C7
            sage: G.exponent()
            42
            sage: G = AbelianGroup([2,4,6]); G
            Multiplicative Abelian group isomorphic to C2 x C4 x C6
            sage: G.exponent()
            12
        """
        return lcm(self.gens_orders())

    def identity(self):
        r"""
        Return the identity element of this group.

        EXAMPLES::

            sage: G = AbelianGroup([2,2])
            sage: e = G.identity()
            sage: e
            1
            sage: g = G.gen(0)
            sage: g*e
            f0
            sage: e*g
            f0
        """
        return self(1)

    def _group_notation(self, eldv):
        """
        Return abstract group notation for generator orders ``eldv``.

        INPUT:

        - ``eldv`` -- iterable of integers

        OUTPUT: string

        EXAMPLES::

            sage: G = AbelianGroup([2,2])
            sage: G._group_notation([0,1,2,3])
            'Z x C1 x C2 x C3'
        """
        v = []
        for x in eldv:
            if x:
                v.append("C%s" % x)
            else:
                v.append("Z")
        return ' x '.join(v)

    def _latex_(self):
        r"""
        Return latex representation of this group.

        EXAMPLES::

            sage: F = AbelianGroup(10, [2]*10)
            sage: F._latex_()
            '$\\mathrm{AbelianGroup}( 10, (2, 2, 2, 2, 2, 2, 2, 2, 2, 2) )$'
        """
        return r"$\mathrm{AbelianGroup}( %s, %s )$" % (self.ngens(),
                                                       self.gens_orders())

    @cached_method
    def _libgap_(self):
        r"""
        Return a GAP group corresponding to this group.

        EXAMPLES::

            sage: G = AbelianGroup([2,3,9])
            sage: libgap(G)                                                             # needs sage.libs.gap
            <pc group of size 54 with 3 generators>

        The result is cached::

            sage: libgap(G) is libgap(G)                                                # needs sage.libs.gap
            True

        Requires the optional ``gap_packages`` for infinite groups::

            sage: G = AbelianGroup(3, [0,3,4], names='abc')
            sage: libgap(G)   # optional - gap_package_polycyclic
            Pcp-group with orders [ 0, 3, 4 ]
        """
        from sage.libs.gap.libgap import libgap

        if self.is_finite():
            return libgap.AbelianGroup(self.gens_orders())

        # Make sure to LoadPackage("Polycyclic") in gap
        from sage.features.gap import GapPackage
        GapPackage("polycyclic", spkg='gap_packages').require()
        return libgap.AbelianPcpGroup(self.gens_orders())

    def _gap_init_(self) -> str:
        r"""
        Return string that defines corresponding abelian group in GAP.

        EXAMPLES::

            sage: G = AbelianGroup([2,3,9])
            sage: G._gap_init_()
            'AbelianGroup([2, 3, 9])'
            sage: gap(G)                                                                # needs sage.libs.gap
            Group( [ f1, f2, f3 ] )

        Requires the optional ``gap_packages`` for infinite groups::

            sage: G = AbelianGroup(3,[0,3,4], names='abc'); G
            Multiplicative Abelian group isomorphic to Z x C3 x C4
            sage: G._gap_init_()   # optional - gap_package_polycyclic
            'AbelianPcpGroup([0, 3, 4])'
        """
        if self.is_finite():
            return 'AbelianGroup(%s)' % list(self.gens_orders())

        from sage.features.gap import GapPackage
        # Make sure to LoadPackage("Polycyclic") in gap
        GapPackage("polycyclic", spkg='gap_packages').require()
        return 'AbelianPcpGroup(%s)' % list(self.gens_orders())

    def gen(self, i=0):
        """
        The `i`-th generator of the abelian group.

        EXAMPLES::

            sage: F = AbelianGroup(5,[],names='a')
            sage: F.0
            a0
            sage: F.2
            a2
            sage: F.gens_orders()
            (0, 0, 0, 0, 0)

            sage: G = AbelianGroup([2,1,3])
            sage: G.gens()
            (f0, 1, f2)
        """
        n = self.ngens()
        if i < 0 or i >= n:
            raise IndexError("Argument i (= %s) must be between 0 and %s." % (i, n-1))
        x = [0]*n
        if self._gens_orders[i] != 1:
            x[i] = 1
        return self.element_class(self, x)

    def gens(self) -> tuple:
        """
        Return the generators of the group.

        OUTPUT:

        A tuple of group elements. The generators according to the
        chosen :meth:`gens_orders`.

        EXAMPLES::

            sage: F = AbelianGroup(5, [3,2], names='abcde')
            sage: F.gens()
            (a, b, c, d, e)
            sage: [g.order() for g in F.gens()]
            [+Infinity, +Infinity, +Infinity, 3, 2]
        """
        return tuple(self.gen(i) for i in range(self.ngens()))

    def gens_orders(self):
        """
        Return the orders of the cyclic factors that this group has
        been defined with.

        Use :meth:`elementary_divisors` if you are looking for an
        invariant of the group.

        OUTPUT: tuple of integers

        EXAMPLES::

            sage: Z2xZ3 = AbelianGroup([2,3])
            sage: Z2xZ3.gens_orders()
            (2, 3)
            sage: Z2xZ3.elementary_divisors()
            (6,)

            sage: Z6 = AbelianGroup([6])
            sage: Z6.gens_orders()
            (6,)
            sage: Z6.elementary_divisors()
            (6,)

            sage: Z2xZ3.is_isomorphic(Z6)
            True
            sage: Z2xZ3 is Z6
            False

        TESTS::

            sage: F = AbelianGroup(3, [2], names='abc')
            sage: list(map(type, F.gens_orders()))
            [<class 'sage.rings.integer.Integer'>,
             <class 'sage.rings.integer.Integer'>,
             <class 'sage.rings.integer.Integer'>]
        """
        return self._gens_orders

    def invariants(self):
        """
        Return the orders of the cyclic factors that this group has
        been defined with.

        For historical reasons this has been called invariants in
        Sage, even though they are not necessarily the invariant
        factors of the group. Use :meth:`gens_orders` instead::

            sage: J = AbelianGroup([2,0,3,2,4]);  J
            Multiplicative Abelian group isomorphic to C2 x Z x C3 x C2 x C4
            sage: J.invariants()    # deprecated
            (2, 0, 3, 2, 4)
            sage: J.gens_orders()   # use this instead
            (2, 0, 3, 2, 4)
            sage: for i in range(J.ngens()):
            ....:     print((i, J.gen(i), J.gen(i).order()))     # or this
            (0, f0, 2)
            (1, f1, +Infinity)
            (2, f2, 3)
            (3, f3, 2)
            (4, f4, 4)

        Use :meth:`elementary_divisors` if you are looking for an
        invariant of the group.

        OUTPUT: tuple of integers; zero means infinite cyclic factor

        EXAMPLES::

            sage: J = AbelianGroup([2,3])
            sage: J.invariants()
            (2, 3)
            sage: J.elementary_divisors()
            (6,)

        TESTS::

            sage: F = AbelianGroup(3, [2], names='abc')
            sage: list(map(type, F.gens_orders()))
            [<class 'sage.rings.integer.Integer'>,
             <class 'sage.rings.integer.Integer'>,
             <class 'sage.rings.integer.Integer'>]
        """
        # TODO: deprecate
        return self.gens_orders()

    def is_cyclic(self):
        """
        Return ``True`` if the group is a cyclic group.

        EXAMPLES::

            sage: J = AbelianGroup([2,3])
            sage: J.gens_orders()
            (2, 3)
            sage: J.elementary_divisors()
            (6,)
            sage: J.is_cyclic()
            True
            sage: G = AbelianGroup([6])
            sage: G.gens_orders()
            (6,)
            sage: G.is_cyclic()
            True
            sage: H = AbelianGroup([2,2])
            sage: H.gens_orders()
            (2, 2)
            sage: H.is_cyclic()
            False
            sage: H = AbelianGroup([2,4])
            sage: H.elementary_divisors()
            (2, 4)
            sage: H.is_cyclic()
            False
            sage: H.permutation_group().is_cyclic()                                     # needs sage.groups
            False
            sage: T = AbelianGroup([])
            sage: T.is_cyclic()
            True
            sage: T = AbelianGroup(1, [0]); T
            Multiplicative Abelian group isomorphic to Z
            sage: T.is_cyclic()
            True
            sage: B = AbelianGroup([3,4,5])
            sage: B.is_cyclic()
            True
        """
        return len(self.elementary_divisors()) <= 1

    @cached_method
    def ngens(self):
        """
        The number of free generators of the abelian group.

        EXAMPLES::

            sage: F = AbelianGroup(10000)
            sage: F.ngens()
            10000
        """
        return len(self.gens_orders())

    @cached_method
    def order(self):
        """
        Return the order of this group.

        EXAMPLES::

            sage: G = AbelianGroup(2, [2,3])
            sage: G.order()
            6
            sage: G = AbelianGroup(3, [2,3,0])
            sage: G.order()
            +Infinity
        """
        length = prod(self.gens_orders())
        if length == 0:
            return infinity
        return length

    cardinality = order

    @cached_method
    def permutation_group(self):
        r"""
        Return the permutation group isomorphic to this abelian group.

        If the invariants are `q_1, \ldots, q_n` then the
        generators of the permutation will be of order
        `q_1, \ldots, q_n`, respectively.

        EXAMPLES::

            sage: G = AbelianGroup(2,[2,3]); G
            Multiplicative Abelian group isomorphic to C2 x C3
            sage: G.permutation_group()                                                 # needs sage.groups
            Permutation Group with generators [(1,2), (3,4,5)]

        TESTS:

        Check that :issue:`25692` is fixed::

            sage: G = AbelianGroup([0])
            sage: G.permutation_group()
            Traceback (most recent call last):
            ...
            TypeError: Abelian group must be finite
        """
        # GAP does not support infinite permutation groups
        if not self.is_finite():
            raise TypeError('Abelian group must be finite')
        from sage.groups.perm_gps.permgroup import PermutationGroup
        s = 'Image(IsomorphismPermGroup(%s))' % self._gap_init_()
        return PermutationGroup(gap_group=s)

    def is_commutative(self):
        """
        Return ``True`` since this group is commutative.

        EXAMPLES::

            sage: G = AbelianGroup([2,3,9, 0])
            sage: G.is_commutative()
            True
            sage: G.is_abelian()
            True
        """
        return True

    def random_element(self):
        """
        Return a random element of this group.

        EXAMPLES::

            sage: G = AbelianGroup([2,3,9])
            sage: G.random_element().parent() is G
            True
        """
        from sage.misc.prandom import randint
        result = self.one()
        for g in self.gens():
            order = g.order()
            if order is infinity:
                order = 42  # infinite order; randomly chosen maximum
            result *= g ** randint(0, order-1)
        return result

    def _repr_(self) -> str:
        """
        Return a string representation of ``self``.

        EXAMPLES::

            sage: G = AbelianGroup([2,3,9])
            sage: G._repr_()
            'Multiplicative Abelian group isomorphic to C2 x C3 x C9'
        """
        eldv = self.gens_orders()
        if len(eldv) == 0:
            return "Trivial Abelian group"
        g = self._group_notation(eldv)
        return "Multiplicative Abelian group isomorphic to " + g

    def subgroup(self, gensH, names='f'):
        """
        Create a subgroup of this group.

        The "big" group must be defined using "named" generators.

        INPUT:

        - ``gensH`` -- list of elements which are products of the
          generators of the ambient abelian group `G` = ``self``

        EXAMPLES::

            sage: # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: G.<a,b,c> = AbelianGroup(3, [2,3,4]); G
            Multiplicative Abelian group isomorphic to C2 x C3 x C4
            sage: H = G.subgroup([a*b,a]); H
            Multiplicative Abelian subgroup isomorphic to C2 x C3 generated by {a*b, a}
            sage: H < G
            True
            sage: F = G.subgroup([a,b^2])
            sage: F
            Multiplicative Abelian subgroup isomorphic to C2 x C3 generated by {a, b^2}
            sage: F.gens()
            (a, b^2)
            sage: F = AbelianGroup(5, [30,64,729], names=list("abcde"))
            sage: a,b,c,d,e = F.gens()
            sage: F.subgroup([a,b])
            Multiplicative Abelian subgroup isomorphic to Z x Z generated by {a, b}
            sage: F.subgroup([c,e])
            Multiplicative Abelian subgroup isomorphic to C2 x C3 x C5 x C729
            generated by {c, e}
        """
        G = self
        gensH = tuple(gensH)
        if isinstance(names, list):
            names = tuple(names)
        for h in gensH:
            if h not in G:
                raise TypeError('Subgroup generators must belong to the given group.')
        return self.Subgroup(self, gensH, names)

    def list(self):
        """
        Return tuple of all elements of this group.

        EXAMPLES::

            sage: G = AbelianGroup([2,3], names = "ab")
            sage: G.list()
            (1, b, b^2, a, a*b, a*b^2)

        ::

            sage: G = AbelianGroup([]); G
            Trivial Abelian group
            sage: G.list()
            (1,)
        """
        if not self.is_finite():
            raise NotImplementedError("group must be finite")
        return tuple(iter(self))

    def __len__(self):
        """
        Return the length of ``self``.

        EXAMPLES::

            sage: G = AbelianGroup(2,[2,3])
            sage: len(G)
            6
            sage: G = AbelianGroup(3,[2,3,0])
            sage: len(G)
            Traceback (most recent call last):
            ...
            NotImplementedError: group must be finite
        """
        length = prod(self.gens_orders())
        if length == 0:
            raise NotImplementedError("group must be finite")
        return int(length)

    def __iter__(self):
        """
        Return an iterator over the elements of this group.

        EXAMPLES::

            sage: G = AbelianGroup([2,3], names = "ab")
            sage: [a for a in G]
            [1, b, b^2, a, a*b, a*b^2]
            sage: L = list(G); L
            [1, b, b^2, a, a*b, a*b^2]

        The returned list is a reference; mutating it does not allow the
        user to (accidentally?) alter the computed generators::

            sage: L[0] = 0
            sage: list(G)
            [1, b, b^2, a, a*b, a*b^2]
            sage: G = AbelianGroup([1], names='a')
            sage: list(G)
            [1]
            sage: G = AbelianGroup([])
            sage: G.list()
            (1,)
            sage: list(G)
            [1]

        We can also iterate over infinite groups::

            sage: A = AbelianGroup([3,0,5,0])
            sage: for a in A:
            ....:   if a^2 == A([1, 2, 3, 4]):
            ....:       print(a, a^2)
            ....:       break
            f0^2*f1*f2^4*f3^2 f0*f1^2*f2^3*f3^4
        """
        invs = self.gens_orders()
        if 0 not in invs:
            # The group is finite
            yield from map(self, mrange(invs))
        else:
            # A similar approach works for infinite groups.
            # (This would also work for finite groups, but is more complicated.)
            from sage.misc.mrange import cantor_product
            yield from map(self, cantor_product(*[range(n) if n
                                                  else ZZ for n in invs]))

    def number_of_subgroups(self, order=None):
        r"""
        Return the number of subgroups of this group, possibly only of a
        specific order.

        INPUT:

        - ``order`` -- (default: ``None``) find the number of subgroups of this
          order; if ``None``, this defaults to counting all subgroups

        ALGORITHM:

        An infinite group has infinitely many subgroups. All finite subgroups of
        any group are contained in the torsion subgroup, which for finitely
        generated abelian group is itself finite. Hence, we can assume the
        group is finite. A finite abelian group is isomorphic to a direct
        product of its Sylow subgroups, and so we can reduce the problem
        further to counting subgroups of finite abelian `p`-groups.

        Assume a Sylow subgroup is a `p`-group of type `\lambda`, and using
        :func:`~sage.combinat.q_analogues.q_subgroups_of_abelian_group`
        sum the number of subgroups of type `\mu` in an abelian `p`-group of
        type `\lambda` for all `\mu` contained in `\lambda`.

        EXAMPLES::

            sage: AbelianGroup([2,0,0,3,0]).number_of_subgroups()
            +Infinity

            sage: # needs sage.combinat
            sage: AbelianGroup([2,3]).number_of_subgroups()
            4
            sage: AbelianGroup([2,4,8]).number_of_subgroups()
            81
            sage: AbelianGroup([2,4,8]).number_of_subgroups(order=4)
            19
            sage: AbelianGroup([10,15,25,12]).number_of_subgroups()
            5760
            sage: AbelianGroup([10,15,25,12]).number_of_subgroups(order=45000)
            1
            sage: AbelianGroup([10,15,25,12]).number_of_subgroups(order=14)
            0

        TESTS::

            sage: # needs sage.combinat
            sage: AbelianGroup([]).number_of_subgroups()
            1
            sage: AbelianGroup([1,3,1]).number_of_subgroups()
            2
            sage: AbelianGroup([]).number_of_subgroups(order=2)
            0
            sage: AbelianGroup([1,3,1]).number_of_subgroups(order=2)
            0
            sage: AbelianGroup([1,3,0,1]).number_of_subgroups(order=3)                  # needs sage.libs.gap  # optional - gap_package_polycyclic
            1
            sage: AbelianGroup([1,3,1]).number_of_subgroups(order=-2)
            Traceback (most recent call last):
            ...
            ValueError: subgroups order must be positive or None
        """
        if not self.is_finite():
            if order is None:
                return infinity
            else:
                # Finite order subgroups are all contained in the torsion subgroup
                return self.torsion_subgroup().number_of_subgroups(order=order)

        from itertools import chain
        from collections import defaultdict
        from sage.arith.misc import factor
        from sage.combinat.q_analogues import q_subgroups_of_abelian_group
        from sage.combinat.integer_lists import IntegerListsLex

        # The group order is prod(p^e for (p,e) in primary_factors)
        primary_factors = list(chain.from_iterable(
            factor(ed) for ed in self.elementary_divisors()))
        sylow_types = defaultdict(list)
        for p, e in primary_factors:
            sylow_types[p].append(e)
        subgroups_orders_kwds = {}

        if order is None:
            for p, p_exps in sylow_types.items():
                subgroups_orders_kwds[p] = {'max_sum': sum(p_exps)}
        else:
            order = Integer(order)
            if order < 1:
                raise ValueError('subgroups order must be positive or None')
            if not order.divides(self.order()):
                return Integer(0)
            order_exps = dict(factor(order))

            for p in (set(sylow_types) - set(order_exps)):
                del sylow_types[p]
            for p in sylow_types:
                subgroups_orders_kwds[p] = {'n': order_exps[p]}

        result = Integer(1)
        for p, p_exps in sylow_types.items():
            p_exps.sort(reverse=True)
            # The sum is over all partitions mu contained in p_exps whose size
            # is determined by subgroups_orders_kwds.
            result *= sum(q_subgroups_of_abelian_group(p_exps, mu, q=p)
                          for mu in IntegerListsLex(max_slope=0,
                                                    min_part=1,
                                                    max_length=len(p_exps),
                                                    ceiling=p_exps,
                                                    element_constructor=list,
                                                    **subgroups_orders_kwds[p]))
        return result

    def subgroups(self, check=False):
        r"""
        Compute all the subgroups of this abelian group (which must be finite).

        INPUT:

        - ``check`` -- boolean; if ``True``, performs the same computation in
          GAP and checks that the number of subgroups generated is the
          same. (I don't know how to convert GAP's output back into
          Sage, so we don't actually compare the subgroups).

        ALGORITHM:

        If the group is cyclic, the problem is easy. Otherwise, write it as
        a direct product A x B, where B is cyclic. Compute the subgroups of
        A (by recursion).

        Now, for every subgroup C of A x B, let G be its *projection onto*
        A and H its *intersection with* B. Then there is a well-defined
        homomorphism f: G -> B/H that sends a in G to the class mod H of b,
        where (a,b) is any element of C lifting a; and every subgroup C
        arises from a unique triple (G, H, f).

        .. TODO::

            This is *many orders of magnitude* slower than Magma.  Consider
            using the much faster method
            :meth:`~AbelianGroup_class.number_of_subgroups` in case you only
            need the number of subgroups, possibly of a specific order.

        EXAMPLES::

            sage: AbelianGroup([2,3]).subgroups()                                       # needs sage.libs.gap  # optional - gap_package_polycyclic
            [Multiplicative Abelian subgroup isomorphic to C2 x C3 generated by {f0*f1^2},
             Multiplicative Abelian subgroup isomorphic to C2 generated by {f0},
             Multiplicative Abelian subgroup isomorphic to C3 generated by {f1},
             Trivial Abelian subgroup]
            sage: len(AbelianGroup([2,4,8]).subgroups())                                # needs sage.libs.gap  # optional - gap_package_polycyclic
            81

        TESTS::

            sage: AbelianGroup([]).subgroups()
            [Trivial Abelian group]

        Check that :issue:`14196` is fixed::

            sage: B = AbelianGroup([1,2])
            sage: B.subgroups()                                                         # needs sage.libs.gap  # optional - gap_package_polycyclic
            [Multiplicative Abelian subgroup isomorphic to C2 generated by {f1},
             Trivial Abelian subgroup]
            sage: B.subgroups(check=True)                                               # needs sage.libs.gap  # optional - gap_package_polycyclic
            [Multiplicative Abelian subgroup isomorphic to C2 generated by {f1},
             Trivial Abelian subgroup]
        """
        if not self.is_finite():
            raise ValueError("group must be finite")
        from sage.misc.verbose import verbose

        if self.is_trivial():
            return [self]
        if self.ngens() == 1:
            n = self.gen(0).order()
            return [self.subgroup([self.gen(0)**i]) for i in divisors(n)]

        v = self.gens_orders()
        A = AbelianGroup(v[:-1])
        x = v[-1]

        Wsubs = A.subgroups()

        subgps = []
        for G in Wsubs:
            ngens = len(G.gens())
            verbose("G = subgp generated by %s" % list(G.gens()))
            verbose("invariants are: %s" % [t.order() for t in G.gens()])
            for H in divisors(x):
                # H = the subgroup of *index* H.
                its = [range(0, H, H // gcd(H, G.gen(i).order()))
                       for i in range(ngens)]
                for f in product(*its):
                    verbose("using hom from G to C_%s sending gens to %s" % (H, f))
                    new_sub = []
                    for a in range(ngens):
                        val = G.gen(a).list() + [f[a]]
                        if any(l != 0 for l in val):
                            new_sub.append(val)
                    if H != x:
                        new_sub.append([0] * A.ngens() + [H])
                    subgps.append(self.subgroup_reduced(new_sub))

        if check:
            verbose("Running Gap cross-check")
            from sage.libs.gap.libgap import libgap
            t = libgap(v).AbelianGroup().SubgroupsSolvableGroup().Size().sage()
            if t != len(subgps):
                raise ArithmeticError("For %s Gap finds %s subgroups, I found %s" % (v, t, len(subgps)))
            verbose("Gap check OK for %s: %s" % (v, t))
        return subgps

    def subgroup_reduced(self, elts, verbose=False):
        r"""
        Given a list of lists of integers (corresponding to elements of
        ``self``), find a set of independent generators for the subgroup
        generated by these elements, and return the subgroup with these as
        generators, forgetting the original generators.

        This is used by the ``subgroups`` routine.

        An error will be raised if the elements given are not linearly
        independent over `\QQ`.

        EXAMPLES::

            sage: G = AbelianGroup([4,4])
            sage: G.subgroup( [ G([1,0]), G([1,2]) ])                                   # needs sage.libs.gap  # optional - gap_package_polycyclic
            Multiplicative Abelian subgroup isomorphic to C2 x C4
            generated by {f0, f0*f1^2}
            sage: AbelianGroup([4,4]).subgroup_reduced( [ [1,0], [1,2] ])               # needs sage.libs.gap  # optional - gap_package_polycyclic
            Multiplicative Abelian subgroup isomorphic to C2 x C4
            generated by {f0^2*f1^2, f0^3}
        """
        from sage.matrix.constructor import matrix
        d = self.ngens()
        X = ZZ**d
        try:
            elt_lattice = X.submodule_with_basis(elts)
        except ValueError as e:
            # can't happen?
            print("Vectors not LI: {}".format(elts))
            raise e
        rel_lattice = X.span([X.gen(i) * self.gens_orders()[i] for i in range(d)])
        isect = elt_lattice.intersection(rel_lattice)
        mat = matrix([elt_lattice.coordinate_vector(x) for x in isect.gens()]).change_ring(ZZ)
        D, U, V = mat.smith_form()
        new_basis = [(elt_lattice.linear_combination_of_basis((~V).row(i)).list(), D[i, i]) for i in range(U.ncols())]
        return self.subgroup([self([x[0][i] % self.gens_orders()[i]
                                    for i in range(d)]) for x in new_basis if x[1] != 1])

    def torsion_subgroup(self, n=None):
        """
        Return the `n`-torsion subgroup of this abelian group when `n`
        is given, and the torsion subgroup otherwise.

        The [`n`-]torsion subgroup consists of all elements whose order
        is finite [and divides `n`].

        EXAMPLES::

            sage: # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: G = AbelianGroup([2, 3])
            sage: G.torsion_subgroup()
            Multiplicative Abelian subgroup isomorphic to C2 x C3 generated
            by {f0, f1}
            sage: G = AbelianGroup([2, 0, 0, 3, 0])
            sage: G.torsion_subgroup()
            Multiplicative Abelian subgroup isomorphic to C2 x C3 generated
            by {f0, f3}
            sage: G = AbelianGroup([])
            sage: G.torsion_subgroup()
            Trivial Abelian subgroup
            sage: G = AbelianGroup([0, 0])
            sage: G.torsion_subgroup()
            Trivial Abelian subgroup

        ::

            sage: G = AbelianGroup([2, 2*3, 2*3*5, 0, 2*3*5*7, 2*3*5*7*11])
            sage: G.torsion_subgroup(5)                                                 # needs sage.libs.gap  # optional - gap_package_polycyclic
            Multiplicative Abelian subgroup isomorphic to C5 x C5 x C5
             generated by {f2^6, f4^42, f5^462}
        """
        if n is None:
            torsion_generators = [g for g in self.gens() if g.order() != infinity]
        else:
            n = ZZ(n)
            if n <= 0:
                raise ValueError('n must be a positive integer')
            torsion_generators = []
            for g in self.gens():
                o = g.order()
                if o == infinity:
                    continue
                d = n.gcd(o)
                torsion_generators.append(g**(o//d))
        return self.subgroup(torsion_generators)


class AbelianGroup_subgroup(AbelianGroup_class):
    """
    Subgroup subclass of ``AbelianGroup_class``, so instance methods are
    inherited.

    .. TODO::

        There should be a way to coerce an element of a subgroup
        into the ambient group.
    """
    def __init__(self, ambient, gens, names='f', category=None):
        """
        EXAMPLES::

            sage: # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: F = AbelianGroup(5, [30,64,729], names=list("abcde"))
            sage: a,b,c,d,e = F.gens()
            sage: F.subgroup([a^3,b])
            Multiplicative Abelian subgroup isomorphic to Z x Z generated by {a^3, b}
            sage: F.subgroup([c])
            Multiplicative Abelian subgroup isomorphic to C2 x C3 x C5 generated by {c}
            sage: F.subgroup([a, c])
            Multiplicative Abelian subgroup isomorphic to C2 x C3 x C5 x Z generated by {a, c}
            sage: F.subgroup([a, b*c])
            Multiplicative Abelian subgroup isomorphic to Z x Z generated by {a, b*c}
            sage: F.subgroup([b*c, d])
            Multiplicative Abelian subgroup isomorphic to C64 x Z generated by {b*c, d}
            sage: F.subgroup([a*b, c^6, d], names=list("xyz"))
            Multiplicative Abelian subgroup isomorphic to C5 x C64 x Z generated by {a*b, c^6, d}
            sage: H.<x,y,z> = F.subgroup([a*b, c^6, d]); H
            Multiplicative Abelian subgroup isomorphic to C5 x C64 x Z generated by {a*b, c^6, d}
            sage: G = F.subgroup([a*b, c^6, d], names=list("xyz")); G
            Multiplicative Abelian subgroup isomorphic to C5 x C64 x Z generated by {a*b, c^6, d}
            sage: x,y,z = G.gens()
            sage: x.order()
            +Infinity
            sage: y.order()
            5
            sage: z.order()
            64
            sage: A = AbelianGroup(5, [3, 5, 5, 7, 8], names='abcde')
            sage: a,b,c,d,e = A.gens()
            sage: A.subgroup([a,b])
            Multiplicative Abelian subgroup isomorphic to C3 x C5 generated by {a, b}
            sage: A.subgroup([a,b,c,d^2,e])
            Multiplicative Abelian subgroup isomorphic to C3 x C5 x C5 x C7 x C8 generated by {a, b, c, d^2, e}
            sage: A.subgroup([a, b, c, d^2, e^2])
            Multiplicative Abelian subgroup isomorphic to C3 x C4 x C5 x C5 x C7 generated by {a, b, c, d^2, e^2}
            sage: B = A.subgroup([a^3, b, c, d, e^2]); B
            Multiplicative Abelian subgroup isomorphic to C4 x C5 x C5 x C7 generated by {b, c, d, e^2}
            sage: B.gens_orders()
            (4, 5, 5, 7)
            sage: A = AbelianGroup(4,[1009, 2003, 3001, 4001], names='abcd')
            sage: a,b,c,d = A.gens()
            sage: B = A.subgroup([a^3,b,c,d])
            sage: B.gens_orders()
            (1009, 2003, 3001, 4001)
            sage: A.order()
            24266473210027
            sage: B.order()
            24266473210027
            sage: A = AbelianGroup(4, [1008, 2003, 3001, 4001], names='abcd')
            sage: a,b,c,d = A.gens()
            sage: B = A.subgroup([a^3,b,c,d]); B
            Multiplicative Abelian subgroup isomorphic
            to C3 x C7 x C16 x C2003 x C3001 x C4001 generated by {a^3, b, c, d}

        Infinite groups can also be handled::

            sage: # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: G = AbelianGroup([3,4,0], names='abc')
            sage: a,b,c = G.gens()
            sage: F = G.subgroup([a, b^2, c]); F
            Multiplicative Abelian subgroup isomorphic to C2 x C3 x Z
             generated by {a, b^2, c}
            sage: F.gens_orders()
            (2, 3, 0)
            sage: F.gens()
            (a, b^2, c)
            sage: F.order()
            +Infinity

        Testing issue :issue:`18863`::

            sage: G = AbelianGroup(5,[2])
            sage: G.subgroup([prod(g^k for g,k in zip(G.gens(),[1,-2,3,-4,5]))])        # needs sage.libs.gap  # optional - gap_package_polycyclic
            Multiplicative Abelian subgroup isomorphic to Z
             generated by {f0*f1^-2*f2^3*f3^-4*f4}
        """
        from sage.libs.gap.libgap import libgap
        if not isinstance(ambient, AbelianGroup_class):
            raise TypeError("ambient (=%s) must be an abelian group" % ambient)
        if not isinstance(gens, tuple):
            raise TypeError("gens (=%s) must be a tuple" % gens)

        self._ambient_group = ambient
        H_gens = tuple(x for x in gens if x != ambient.one())  # clean entry
        self._gens = H_gens

        H = libgap(ambient).Subgroup(H_gens)

        invs = H.TorsionSubgroup().AbelianInvariants().sage()
        rank = len([1 for g in H.GeneratorsOfGroup()
                    if g.Order().sage() is infinity])
        invs += [0] * rank

        self._abinvs = invs
        invs = tuple(ZZ(i) for i in invs)

        if category is None:
            category = Groups().Commutative().Subobjects()
        AbelianGroup_class.__init__(self, invs, names, category=category)

    def __contains__(self, x):
        """
        Test whether ``x`` is an element of this subgroup.

        EXAMPLES::

            sage: # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: G.<a,b> = AbelianGroup(2)
            sage: A = G.subgroup([a])
            sage: a in G
            True
            sage: a in A
            True

        TESTS:

        Check that :issue:`32910` is fixed::

            sage: # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: G.<a,b> = AbelianGroup(2, [4, 576])
            sage: Hgens = [a^2, a*b^2]
            sage: H = G.subgroup(Hgens)
            sage: [g in H for g in (a^3, b^2, b^3, a^3*b^2, "junk")]
            [False, False, False, True, False]

        Check that :issue:`31507` is fixed::

            sage: # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: G = AbelianGroup(2, gens_orders=[16, 16])
            sage: f0, f1 = G.gens()
            sage: H = G.subgroup([f0*f1^3])
            sage: [g in H for g in (f0, f0*f1^2, f0*f1^3, f0*f1^4)]
            [False, False, True, False]

            sage: G.<a,b> = AbelianGroup(2)
            sage: Hgens =  [a*b, a*b^-1]
            sage: H = G.subgroup(Hgens)  # optional - gap_package_polycyclic
            sage: b^2 in H  # optional - gap_package_polycyclic
            True
        """
        if not isinstance(x, AbelianGroupElement):
            return False
        if x.parent() is self:
            return True
        if x in self.ambient_group():
            amb_inv = self.ambient_group().gens_orders()
            inv_basis = diagonal_matrix(ZZ, amb_inv)
            gens_basis = matrix(
                ZZ, len(self._gens), len(amb_inv),
                [g.list() for g in self._gens]
            )
            return (vector(ZZ, x.list())
                    in inv_basis.stack(gens_basis).row_module())
        return False

    def ambient_group(self):
        """
        Return the ambient group related to ``self``.

        OUTPUT: a multiplicative Abelian group

        EXAMPLES::

            sage: # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: G.<a,b,c> = AbelianGroup([2,3,4])
            sage: H = G.subgroup([a, b^2])
            sage: H.ambient_group() is G
            True
        """
        return self._ambient_group

    def equals(left, right):
        """
        Check whether ``left`` and ``right`` are the same (sub)group.

        INPUT:

        - ``right`` -- anything

        OUTPUT:

        boolean; if ``right`` is a subgroup, test whether ``left`` and
        ``right`` are the same subset of the ambient group. If
        ``right`` is not a subgroup, test whether they are isomorphic
        groups, see :meth:`~AbelianGroup_class.is_isomorphic`.

        EXAMPLES::

            sage: # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: G = AbelianGroup(3, [2,3,4], names='abc'); G
            Multiplicative Abelian group isomorphic to C2 x C3 x C4
            sage: a,b,c = G.gens()
            sage: F = G.subgroup([a,b^2]); F
            Multiplicative Abelian subgroup isomorphic to C2 x C3 generated by {a, b^2}
            sage: F<G
            True
            sage: A = AbelianGroup(1, [6])
            sage: A.subgroup(list(A.gens())) == A
            True
            sage: G.<a,b> = AbelianGroup(2)
            sage: A = G.subgroup([a])
            sage: B = G.subgroup([b])
            sage: A.equals(B)
            False
            sage: A == B        # sames as A.equals(B)
            False
            sage: A.is_isomorphic(B)
            True
        """
        left_ambient = left.ambient_group()
        try:
            right_ambient = right.ambient_group()
        except AttributeError:
            # right is not a subgroup
            return left.is_isomorphic(right)
        if left_ambient is not right_ambient:
            return False
        return left <= right <= left

    __eq__ = equals

    def _repr_(self):
        """
        Return a string representation.

        EXAMPLES::

            sage: G.<a,b> = AbelianGroup(2)
            sage: G._repr_()
            'Multiplicative Abelian group isomorphic to Z x Z'
            sage: A = G.subgroup([a])                                                   # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: A._repr_()                                                            # needs sage.libs.gap  # optional - gap_package_polycyclic
            'Multiplicative Abelian subgroup isomorphic to Z generated by {a}'
        """
        eldv = self._abinvs
        if self.is_trivial():
            return "Trivial Abelian subgroup"
        s = 'Multiplicative Abelian subgroup isomorphic to '
        s += self._group_notation(eldv)
        s += ' generated by '
        s += '{' + ', '.join(map(str, self.gens())) + '}'
        return s

    def gens(self) -> tuple:
        """
        Return the generators for this subgroup.

        OUTPUT: tuple of group elements generating the subgroup

        EXAMPLES::

            sage: # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: G.<a,b> = AbelianGroup(2)
            sage: A = G.subgroup([a])
            sage: G.gens()
            (a, b)
            sage: A.gens()
            (a,)
        """
        return self._gens

    def gen(self, n):
        """
        Return the `n`-th generator of this subgroup.

        EXAMPLES::

            sage: # needs sage.libs.gap  # optional - gap_package_polycyclic
            sage: G.<a,b> = AbelianGroup(2)
            sage: A = G.subgroup([a])
            sage: A.gen(0)
            a
        """
        return self._gens[n]


# We allow subclasses to override this, analogous to Element
AbelianGroup_class.Subgroup = AbelianGroup_subgroup
