/*
   This file is part of TALER
   Copyright (C) 2024, 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file exchangedb/pg_select_aml_statistics.c
 * @brief Implementation of the select_aml_statistics function for Postgres
 * @author Christian Grothoff
 */
#include "taler/platform.h"
#include "taler/taler_error_codes.h"
#include "taler/taler_dbevents.h"
#include "taler/taler_pq_lib.h"
#include "pg_select_aml_statistics.h"
#include "pg_helper.h"


/**
 * Closure for #get_statistics_cb().
 */
struct GetStatisticsContext
{
  /**
   * Function to call per result.
   */
  TALER_EXCHANGEDB_AmlStatisticsCallback cb;

  /**
   * Closure for @e cb.
   */
  void *cb_cls;

  /**
   * Plugin context.
   */
  struct PostgresClosure *pg;

  /**
   * Flag set to #GNUNET_OK as long as everything is fine.
   */
  enum GNUNET_GenericReturnValue status;

};


/**
 * Invoke the callback for each result.
 *
 * @param cls a `struct GetStatisticsContext *`
 * @param result SQL result
 * @param num_results number of rows in @a result
 */
static void
get_statistics_cb (void *cls,
                   PGresult *result,
                   unsigned int num_results)
{
  struct GetStatisticsContext *ctx = cls;

  for (unsigned int i = 0; i < num_results; i++)
  {
    uint64_t val;
    char *name;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_string ("name",
                                    &name),
      GNUNET_PQ_result_spec_uint64 ("value",
                                    &val),
      GNUNET_PQ_result_spec_end
    };

    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      GNUNET_break (0);
      ctx->status = GNUNET_SYSERR;
      return;
    }
    ctx->cb (ctx->cb_cls,
             name,
             val);
    GNUNET_PQ_cleanup_result (rs);
  }
}


enum GNUNET_DB_QueryStatus
TEH_PG_select_aml_statistics (
  void *cls,
  size_t num_names,
  const char *names[static num_names],
  struct GNUNET_TIME_Timestamp start_date,
  struct GNUNET_TIME_Timestamp end_date,
  TALER_EXCHANGEDB_AmlStatisticsCallback cb,
  void *cb_cls)
{
  struct PostgresClosure *pg = cls;
  struct GetStatisticsContext ctx = {
    .cb = cb,
    .cb_cls = cb_cls,
    .pg = pg,
    .status = GNUNET_OK
  };
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_array_ptrs_string (num_names,
                                             names,
                                             pg->conn),
    GNUNET_PQ_query_param_timestamp (&start_date),
    GNUNET_PQ_query_param_timestamp (&end_date),
    GNUNET_PQ_query_param_end
  };
  enum GNUNET_DB_QueryStatus qs;

  PREPARE (pg,
           "select_aml_statistics",
           "SELECT "
           " event_type AS name"
           ",COUNT(*) AS value"
           " FROM kyc_events"
           " WHERE event_type = ANY ($1)"
           "   AND event_timestamp >= $2"
           "   AND event_timestamp < $3"
           " GROUP BY event_type;");
  qs = GNUNET_PQ_eval_prepared_multi_select (pg->conn,
                                             "select_aml_statistics",
                                             params,
                                             &get_statistics_cb,
                                             &ctx);
  GNUNET_PQ_cleanup_query_params_closures (params);
  if (GNUNET_OK != ctx.status)
    return GNUNET_DB_STATUS_HARD_ERROR;
  return qs;
}
