//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// 19.04.24 V. Grichine, first version
//

#include "G4XTRGaussRadModel.hh"
#include "G4PhysicalConstants.hh"

#include "G4Material.hh"
#include "G4Element.hh"
#include "G4NistManager.hh"

using namespace std;
using namespace CLHEP;

////////////////////////////////////////////////////////////////////////////
// Constructor, destructor

G4XTRGaussRadModel::G4XTRGaussRadModel(
  G4LogicalVolume* anEnvelope,   G4double alphaPlate, G4double alphaGas, G4Material* foilMat, G4Material* gasMat,
  G4double aa, G4double b, G4int n, const G4String& processName)
  : G4VXTRenergyLoss(anEnvelope, foilMat, gasMat, aa, b, n, processName)
{
  if( verboseLevel > 0 )
    G4cout << "G4XTRGaussRadModel EM process is called"
           << G4endl;

  fAlphaPlate = alphaPlate; // ~40
  fAlphaGas   = alphaGas;   // ~10 
  fExitFlux   = true;       // XTR photons are moved to the end of radiator
}

///////////////////////////////////////////////////////////////////////////

G4XTRGaussRadModel::~G4XTRGaussRadModel() = default;

///////////////////////////////////////////////////////////////////////////

void G4XTRGaussRadModel::ProcessDescription(std::ostream& out) const
{
  out << "Simulation of forward X-ray transition radiation generated by\n"
         "relativistic charged particles crossing the interface between\n"
         "two materials.\n";
}

///////////////////////////////////////////////////////////////////////////

G4double G4XTRGaussRadModel::SpectralXTRdEdx(G4double energy)
{
  static constexpr G4double cofPHC = 4. * pi * hbarc;
  G4double result, sum = 0., tmp, cof1, cof2, cofMin, theta2, theta2k;
  G4double aMa, bMb, sigma, dump;
  G4int k, kMax, kMin;

  aMa   = fPlateThick * GetPlateLinearPhotoAbs(energy);
  bMb   = fGasThick * GetGasLinearPhotoAbs(energy);
  sigma = 0.5 * (aMa + bMb);
  dump  = std::exp(-fPlateNumber * sigma);
  if(verboseLevel > 2)
    G4cout << " dump = " << dump << G4endl;
  tmp  = (fSigma1 - fSigma2) / cofPHC / energy;
  cof1 = fPlateThick * tmp;
  cof2 = fGasThick * tmp;

  cofMin = energy * (fPlateThick + fGasThick) / fGamma / fGamma;
  cofMin += (fPlateThick * fSigma1 + fGasThick * fSigma2) / energy;
  cofMin /= cofPHC;

  theta2 = cofPHC / (energy * (fPlateThick + fGasThick));

  kMin = G4int(cofMin);
  if(cofMin > kMin)
    kMin++;

  kMax = kMin + 200; // 99; // 49; //

  if(verboseLevel > 2)
  {
    G4cout << cof1 << "     " << cof2 << "        " << cofMin << G4endl;
    G4cout << "kMin = " << kMin << ";    kMax = " << kMax << G4endl;
  }
  for(k = kMin; k <= kMax; ++k)
  {
    tmp    = pi * fPlateThick * (k + cof2) / (fPlateThick + fGasThick);
    result = (k - cof1) * (k - cof1) * (k + cof2) * (k + cof2);
    if(k == kMin && kMin == G4int(cofMin))
    {
      sum +=
        0.5 * std::sin(tmp) * std::sin(tmp) * std::abs(k - cofMin) / result;
    }
    else
    {
      sum += std::sin(tmp) * std::sin(tmp) * std::abs(k - cofMin) / result;
    }
    theta2k = std::sqrt(theta2 * std::abs(k - cofMin));

    if(verboseLevel > 2)
    {
      G4cout << k << "   " << theta2k << "     "
             << std::sin(tmp) * std::sin(tmp) * std::abs(k - cofMin) / result
             << "      " << sum << G4endl;
    }
  }
  result = 2 * (cof1 + cof2) * (cof1 + cof2) * sum / energy;
  result *= dump * (-1 + dump + 2 * fPlateNumber);
  return result;
}

///////////////////////////////////////////////////////////////////////////
//
// Approximation for radiator interference factor for the case of
// Gauss-distributed regular radiator. The plate and gas gap thicknesses
// are Gauss distributed with RMS
// sa and sb for plate and gas, respectively.
// The mean values of the plate and gas gap thicknesses
// are supposed to be about XTR formation zones.
// The XTR photons are moved to the end of radiator


G4double G4XTRGaussRadModel::GetStackFactor(G4double energy,
                                                    G4double gamma,
                                                    G4double varAngle)
{
  G4double result(0.); 

  G4double Ma, Mb, aMa, bMb, sigma;

  G4double sa = fPlateThick/fAlphaPlate;
  G4double sb = fGasThick/fAlphaGas;

  Ma =  GetPlateLinearPhotoAbs(energy);  
  aMa = fPlateThick * Ma;
  Mb = GetGasLinearPhotoAbs(energy);  
  bMb   = fGasThick * Mb; 
  sigma = aMa + bMb; // m1*t1+m2*t2 dimensionless
  G4double nn = G4double( fPlateNumber );

  // Gauss fluctuation of foil and gas gaps according to
  // RMS = sa = a/fAlphaPlate and RMS = sb = b/fAlphaGas

  G4complex med(0.,1.);
  G4complex Z1   = GetPlateComplexFZ( energy, gamma, varAngle);
  G4complex por1 = -0.5*med*fPlateThick/Z1 - 0.125*sa*sa/Z1/Z1;
  
  G4complex Z2   = GetGasComplexFZ( energy, gamma, varAngle);
  G4complex por2 = -0.5*med*fGasThick/Z2 - 0.125*sb*sb/Z2/Z2;
  
  G4complex npor = (por1+por2)*nn;
  
  G4complex Ha = exp(por1); 
  G4complex Hb = exp(por2);
  
  G4complex H  = Ha*Hb;
  G4complex Hn = exp(npor); 

  G4complex A  = ( 1.0 - Ha ) * ( 1.0 - Hb ) / ( 1. - H );
  G4complex B1 = ( 1.0 - Ha ) * ( 1.0 - Ha ) * Hb / ( 1. - H );
  
  G4complex R  = B1*( exp(-sigma*nn) - Hn )/( exp(-sigma) - H );

  R += A * ( 1 - exp(-sigma*nn) ) / ( 1. - exp(-sigma) ); // -> A*nn
  
  R *= OneInterfaceXTRdEdx(energy, gamma, varAngle);
  
  result      = 2.0 * std::real(R);

  return result;
}
