//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
/// \file ProcessesCount.cc
/// \brief Implementation of the ProcessesCount class

#include "ProcessesCount.hh"

#include <iomanip>

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

ProcessesCount::ProcessesCount(const G4String& name) : G4VAccumulable(name) {}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void ProcessesCount::Count(const G4String& procName)
{
  if (const auto& [it, inserted] = fProcCounter.try_emplace(procName, 1); !inserted) {
    it->second++;  // Exists already
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void ProcessesCount::Print(G4PrintOptions /*options*/) const
{
  G4int index = 0;
  std::map<G4String, G4int>::const_iterator it;
  for (it = fProcCounter.cbegin(); it != fProcCounter.cend(); it++) {
    G4String procName = it->first;
    G4int count = it->second;
    G4cout << " " << std::setw(20) << procName << "=" << std::setw(7) << count
           << (++index % 3 == 0 ? "\n" : " ");
  }
  G4cout << G4endl;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void ProcessesCount::Merge(const G4VAccumulable& other)
{
  auto worker = dynamic_cast<const ProcessesCount&>(other);

  for (const auto& proc : worker.fProcCounter) {
    const auto& [it, inserted] = fProcCounter.emplace(proc);
    if (!inserted) {
      it->second += proc.second;  // proc exists already in master
    }
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
