/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

package org.mozilla.fenix.settings.trustpanel.ui

import android.graphics.Bitmap
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.tooling.preview.PreviewLightDark
import androidx.compose.ui.unit.dp
import mozilla.components.support.ktx.kotlin.tryGetHostFromUrl
import org.mozilla.fenix.compose.Favicon
import org.mozilla.fenix.settings.trustpanel.store.WebsiteInfoState
import org.mozilla.fenix.theme.FirefoxTheme
import org.mozilla.fenix.theme.Theme

private val ICON_SIZE = 16.dp
private val ICON_PADDING = 8.dp
private val OUTER_ICON_SHAPE = RoundedCornerShape(4.dp)
private val INNER_ICON_SHAPE = RoundedCornerShape(0.dp)

@Composable
internal fun ProtectionPanelHeader(
    websiteInfoState: WebsiteInfoState,
    icon: Bitmap?,
    modifier: Modifier = Modifier,
) {
    Row(
        modifier = modifier
            .fillMaxWidth()
            .padding(horizontal = 16.dp, vertical = 6.dp),
        verticalAlignment = Alignment.CenterVertically,
    ) {
        ProtectionPanelIcon(url = websiteInfoState.websiteUrl, icon = icon)

        Spacer(modifier = Modifier.width(16.dp))

        Column(
            modifier = Modifier.weight(1f),
        ) {
            Text(
                text = websiteInfoState.websiteTitle.ifEmpty {
                    websiteInfoState.websiteUrl.tryGetHostFromUrl()
                },
                color = MaterialTheme.colorScheme.onSurface,
                maxLines = 1,
                style = FirefoxTheme.typography.headline7,
                overflow = TextOverflow.Ellipsis,
            )

            if (websiteInfoState.websiteTitle.isNotEmpty()) {
                Text(
                    text = websiteInfoState.websiteUrl.tryGetHostFromUrl(),
                    color = MaterialTheme.colorScheme.onSurfaceVariant,
                    style = FirefoxTheme.typography.body2,
                )
            }
        }
    }
}

@Composable
private fun ProtectionPanelIcon(
    url: String,
    icon: Bitmap?,
) {
    if (icon != null && !icon.isRecycled) {
        Image(
            bitmap = icon.asImageBitmap(),
            contentDescription = null,
            modifier = Modifier
                .background(
                    color = MaterialTheme.colorScheme.surfaceContainerLowest,
                    shape = OUTER_ICON_SHAPE,
                )
                .padding(all = ICON_PADDING)
                .size(ICON_SIZE),
        )
    } else {
        Favicon(
            url = url,
            modifier = Modifier
                .background(
                    color = MaterialTheme.colorScheme.surfaceContainerLowest,
                    shape = OUTER_ICON_SHAPE,
                )
                .padding(all = ICON_PADDING),
            size = ICON_SIZE,
            shape = INNER_ICON_SHAPE,
        )
    }
}

@PreviewLightDark
@Composable
private fun ProtectionPanelHeaderPreview() {
    FirefoxTheme {
        ProtectionPanelHeader(
            websiteInfoState = WebsiteInfoState(
                isSecured = true,
                websiteUrl = "https://www.mozilla.org",
                websiteTitle = "Mozilla",
                certificateName = "",
            ),
            icon = null,
            modifier = Modifier.background(color = MaterialTheme.colorScheme.surface),
        )
    }
}

@PreviewLightDark
@Composable
private fun ProtectionPanelHeaderUrlAsTitlePreview() {
    FirefoxTheme {
        ProtectionPanelHeader(
            websiteInfoState = WebsiteInfoState(
                isSecured = true,
                websiteUrl = "https://www.mozilla.org",
                websiteTitle = "",
                certificateName = "",
            ),
            icon = null,
            modifier = Modifier.background(color = MaterialTheme.colorScheme.surface),
        )
    }
}

@Preview
@Composable
private fun ProtectionPanelHeaderPrivatePreview() {
    FirefoxTheme(theme = Theme.Private) {
        ProtectionPanelHeader(
            websiteInfoState = WebsiteInfoState(
                isSecured = false,
                websiteUrl = "https://www.mozilla.org",
                websiteTitle = "Mozilla",
                certificateName = "",
            ),
            icon = null,
            modifier = Modifier.background(color = MaterialTheme.colorScheme.surface),
        )
    }
}
