from abc import ABC, abstractmethod

from typing import Callable, Optional

from markdown_it.token import Token
from markdown_it.utils import EnvType, OptionsDict

from iotas.note import Note


class HtmlGenerator(ABC):
    """HTML generator base."""

    RESOURCE_CSS_PATH = "/media/css/web/markdown.css"
    RESOURCE_KATEX_CSS_PATH = "/media/css/web/katex.min.css"
    RESOURCE_KATEX_JS_PATH = "/media/js/katex.min.js"

    @abstractmethod
    def generate(
        self,
        note: Note,
        tokens: list[Token],
        render_func: Callable[[list[Token], OptionsDict, EnvType], str],
        parser_options: OptionsDict,
        searching: bool,
        export_format: Optional[str],
        scroll_position: Optional[float] = None,
    ) -> str:
        """Generator HTML for note.

        :param Note note: Note to render
        :param list[Token] tokens: Parser tokens
        :param Callable[[list[Token], OptionsDict, EnvType], str] render_func: Render function
        :param OptionsDict parser_options: Parser options
        :param bool searching: Whether search CSS should be included
        :param Optional[str] export_format: Export format, if using
        :param Optional[float] scroll_position: Position to scroll to
        :return: Generated HTML
        :rtype: str
        """
        raise NotImplementedError()

    @abstractmethod
    def generate_user_stylesheet(self, searching: bool) -> str:
        """Generate part of stylesheet based on state (configuration etc).

        :param bool searching: Whether searching
        :return: stylesheet
        :rtype: str
        """
        raise NotImplementedError()

    @abstractmethod
    def update_font_family(self, family: str) -> None:
        """Update the font family.

        :param str family: New font family
        """
        raise NotImplementedError()
