/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.fineract.infrastructure.core.persistence;

import jakarta.persistence.EntityManager;
import jakarta.persistence.FlushModeType;
import jakarta.persistence.PersistenceContext;
import java.util.function.Supplier;
import org.springframework.stereotype.Component;

@Component
public class FlushModeHandler {

    @PersistenceContext
    private EntityManager entityManager;

    public void withFlushMode(FlushModeType flushMode, Runnable runnable) {
        withFlushMode(flushMode, () -> {
            runnable.run();
            return null;
        });
    }

    /**
     * Executes the provided supplier with the specified flush mode, then restores the original flush mode.
     *
     * @param flushMode
     *            the flush mode to set
     * @param supplier
     *            the code to execute
     * @param <T>
     *            the type of the result
     * @return the result of the supplier
     */
    public <T> T withFlushMode(FlushModeType flushMode, Supplier<T> supplier) {
        FlushModeType original = entityManager.getFlushMode();
        try {
            entityManager.setFlushMode(flushMode);
            return supplier.get();
        } finally {
            entityManager.setFlushMode(original);
        }
    }
}
