% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.R
\name{plot_SV_several_approaches}
\alias{plot_SV_several_approaches}
\title{Shapley Value Bar Plots for Several Explanation Objects}
\usage{
plot_SV_several_approaches(
  explanation_list,
  index_explicands = NULL,
  index_explicands_sort = FALSE,
  only_these_features = NULL,
  plot_phi0 = FALSE,
  digits = 4,
  print_ggplot = TRUE,
  add_zero_line = FALSE,
  axis_labels_n_dodge = NULL,
  axis_labels_rotate_angle = NULL,
  horizontal_bars = TRUE,
  facet_scales = "free",
  facet_ncol = 2,
  geom_col_width = 0.85,
  brewer_palette = NULL,
  include_group_feature_means = FALSE
)
}
\arguments{
\item{explanation_list}{A list of \code{\link[=explain]{explain()}} objects applied to the same data and model.
If the entries in the list are named, then the function use these names. Otherwise, they default to
the approach names (with integer suffix for duplicates) for the explanation objects in \code{explanation_list}.}

\item{index_explicands}{Integer vector. Which of the explicands (test observations) to plot.
E.g. if you have explained 10 observations using \code{\link[=explain]{explain()}}, you can generate a plot for the
first 5 observations/explicands and the 10th by setting \code{index_x_explain = c(1:5, 10)}.
The argument \code{index_explicands_sort} must be \code{FALSE} to plot the explicand
in the order specified in \code{index_x_explain}.}

\item{index_explicands_sort}{Boolean. If \code{FALSE} (default), then \code{shapr} plots the explicands in the order
specified in \code{index_explicands}. If \code{TRUE}, then \code{shapr} sort the indices in increasing order based on their id.}

\item{only_these_features}{String vector. Containing the names of the features which
are to be included in the bar plots.}

\item{plot_phi0}{Boolean. If we are to include the \eqn{\phi_0} in the bar plots or not.}

\item{digits}{Integer.
Number of significant digits to use in the feature description.
Applicable for \code{plot_type} \code{"bar"} and \code{"waterfall"}}

\item{print_ggplot}{Logical.
Whether to print the created \code{ggplot} object once it is returned.
The default is \code{TRUE} which ensures the plot is always displayed also in loops, functions, when sourcing a script,
and when assigning the output to a variable like \code{p <- plot.shapr(...)}.
See \code{\link[ggplot2:print.ggplot]{ggplot2::print.ggplot()}} for more details.
If you wish to further modify the returned \code{ggplot} object outside of \code{plot.shapr},
we recommend setting \code{print_ggplot = FALSE} to avoid force printing.
See the examples for a practical use case.}

\item{add_zero_line}{Boolean. If we are to add a black line for a feature contribution of 0.}

\item{axis_labels_n_dodge}{Integer. The number of rows that
should be used to render the labels. This is useful for displaying labels that would otherwise overlap.}

\item{axis_labels_rotate_angle}{Numeric. The angle of the axis label, where 0 means horizontal, 45 means tilted,
and 90 means vertical. Compared to setting the angle in \code{\link[ggplot2:theme]{ggplot2::theme()}} / \code{\link[ggplot2:element]{ggplot2::element_text()}}, this also
uses some heuristics to automatically pick the \code{hjust} and \code{vjust} that you probably want.}

\item{horizontal_bars}{Boolean. Flip Cartesian coordinates so that horizontal becomes vertical,
and vertical, horizontal. This is primarily useful for converting geoms and statistics which display
y conditional on x, to x conditional on y. See \code{\link[ggplot2:coord_flip]{ggplot2::coord_flip()}}.}

\item{facet_scales}{Should scales be free ("\code{free}", the default), fixed ("\code{fixed}"), or free in one dimension
("\code{free_x}", "\code{free_y}")? The user has to change the latter manually depending on the value of \code{horizontal_bars}.}

\item{facet_ncol}{Integer. The number of columns in the facet grid. Default is \code{facet_ncol = 2}.}

\item{geom_col_width}{Numeric. Bar width. By default, set to 85\% of the \code{\link[ggplot2:resolution]{ggplot2::resolution()}} of the data.}

\item{brewer_palette}{String. Name of one of the color palettes from \code{\link[RColorBrewer:ColorBrewer]{RColorBrewer::RColorBrewer()}}.
If \code{NULL}, then the function uses the default \code{\link[ggplot2:ggplot]{ggplot2::ggplot()}} color scheme.
The following palettes are available for use with these scales:
\describe{
\item{Diverging}{BrBG, PiYG, PRGn, PuOr, RdBu, RdGy, RdYlBu, RdYlGn, Spectral}
\item{Qualitative}{Accent, Dark2, Paired, Pastel1, Pastel2, Set1, Set2, Set3}
\item{Sequential}{Blues, BuGn, BuPu, GnBu, Greens, Greys, Oranges,
OrRd, PuBu, PuBuGn, PuRd, Purples, RdPu, Reds, YlGn, YlGnBu, YlOrBr, YlOrRd}
}}

\item{include_group_feature_means}{Logical. Whether to include the average feature value in a group on the
y-axis or not. If \code{FALSE} (default), then no value is shown for the groups. If \code{TRUE}, then \code{shapr} includes
the mean of the features in each group.}
}
\value{
A \code{\link[ggplot2:ggplot]{ggplot2::ggplot()}} object.
}
\description{
Make plots to visualize and compare the estimated Shapley values for a list of
\code{\link[=explain]{explain()}} objects applied to the same data and model. For group-wise Shapley values,
the features values plotted are the mean feature values for all features in each group.
}
\examples{
\dontrun{
if (requireNamespace("xgboost", quietly = TRUE) && requireNamespace("ggplot2", quietly = TRUE)) {
  # Get the data
  data("airquality")
  data <- data.table::as.data.table(airquality)
  data <- data[complete.cases(data), ]

  # Define the features and the response
  x_var <- c("Solar.R", "Wind", "Temp", "Month")
  y_var <- "Ozone"

  # Split data into test and training data set
  ind_x_explain <- 1:12
  x_train <- data[-ind_x_explain, ..x_var]
  y_train <- data[-ind_x_explain, get(y_var)]
  x_explain <- data[ind_x_explain, ..x_var]

  # Fitting a basic xgboost model to the training data
  model <- xgboost::xgboost(
    data = as.matrix(x_train),
    label = y_train,
    nround = 20,
    verbose = FALSE
  )

  # Specifying the phi_0, i.e. the expected prediction without any features
  phi0 <- mean(y_train)

  # Independence approach
  explanation_independence <- explain(
    model = model,
    x_explain = x_explain,
    x_train = x_train,
    approach = "independence",
    phi0 = phi0,
    n_MC_samples = 1e2
  )

  # Empirical approach
  explanation_empirical <- explain(
    model = model,
    x_explain = x_explain,
    x_train = x_train,
    approach = "empirical",
    phi0 = phi0,
    n_MC_samples = 1e2
  )

  # Gaussian 1e1 approach
  explanation_gaussian_1e1 <- explain(
    model = model,
    x_explain = x_explain,
    x_train = x_train,
    approach = "gaussian",
    phi0 = phi0,
    n_MC_samples = 1e1
  )

  # Gaussian 1e2 approach
  explanation_gaussian_1e2 <- explain(
    model = model,
    x_explain = x_explain,
    x_train = x_train,
    approach = "gaussian",
    phi0 = phi0,
    n_MC_samples = 1e2
  )

  # Combined approach
  explanation_combined <- explain(
    model = model,
    x_explain = x_explain,
    x_train = x_train,
    approach = c("gaussian", "ctree", "empirical"),
    phi0 = phi0,
    n_MC_samples = 1e2
  )

  # Create a list of explanations with names
  explanation_list <- list(
    "Ind." = explanation_independence,
    "Emp." = explanation_empirical,
    "Gaus. 1e1" = explanation_gaussian_1e1,
    "Gaus. 1e2" = explanation_gaussian_1e2,
    "Combined" = explanation_combined
  )

  # The function uses the provided names.
  plot_SV_several_approaches(explanation_list)

  # We can change the number of columns in the grid of plots and add other visual alterations
  # Set `print_ggplot = FALSE` to avoid force displaying the ggplot object before the modifications
  # outside plot_SV_several_approaches()

  plot_SV_several_approaches(explanation_list,
    facet_ncol = 3,
    facet_scales = "free_y",
    add_zero_line = TRUE,
    digits = 2,
    brewer_palette = "Paired",
    geom_col_width = 0.6,
    print_ggplot = FALSE
  ) +
    ggplot2::theme_minimal() +
    ggplot2::theme(legend.position = "bottom", plot.title = ggplot2::element_text(size = 10))


  # We can specify which explicands to plot to get less chaotic plots and make the bars vertical
  plot_SV_several_approaches(explanation_list,
    index_explicands = c(1:2, 5, 10),
    horizontal_bars = FALSE,
    axis_labels_rotate_angle = 45
  )

  # We can change the order of the features by specifying the
  # order using the `only_these_features` parameter.
  plot_SV_several_approaches(explanation_list,
    index_explicands = c(1:2, 5, 10),
    only_these_features = c("Temp", "Solar.R", "Month", "Wind")
  )

  # We can also remove certain features if we are not interested in them
  # or want to focus on, e.g., two features. The function will give a
  # message to if the user specifies non-valid feature names.
  plot_SV_several_approaches(explanation_list,
    index_explicands = c(1:2, 5, 10),
    only_these_features = c("Temp", "Solar.R"),
    plot_phi0 = TRUE
  )

}
}

}
\author{
Lars Henry Berge Olsen
}
