#' Scan alignment to detect reversion mutations
#'
#' @param roi.aln.df A data frame of the detection window alignment
#' @param pathog.mut.obj A pathogenic mutation object
#' @param reference.genome A BSgenome or DNAStringSet object of reference sequences
#' @param ensdb An EnsDb object of Ensembl-based annotation database
#' @param check.wildtype.reads A logical value indicating whether wild type reads to be processed as revertant-to-wildtype reads
#' 
#' @return A list of reversion details in a read including flags of read type, a flag of pathogenic mutation in read, a data frame of reversion mutations
#' 
#' @importFrom GenomeInfoDb seqlevels
#' @importFrom GenomeInfoDb genome
#' @importFrom BiocGenerics strand
#' @importFrom BiocGenerics start
#' @importFrom BiocGenerics end
#' @importFrom BiocGenerics do.call
#' @importFrom BSgenome getSeq
#' @importFrom IRanges IRanges
#' @importFrom IRanges ranges
#' @importFrom IRanges overlapsRanges
#' @importFrom Biostrings DNAString
#' @importFrom Biostrings reverseComplement
#' @importFrom Biostrings subseq
#' @importFrom Biostrings translate
#'
#' @examples
#' \dontrun{
#'     scanAlnForRev()
#' }
#'
#' @noRd
scanAlnForRev <- function(roi.aln.df, pathog.mut.obj, reference.genome, ensdb, check.wildtype.reads){
    
    roi.aln.df0 <- roi.aln.df
    ref.start <- min(roi.aln.df$ref_pos, na.rm=TRUE)
    ref.end <- max(roi.aln.df$ref_pos, na.rm=TRUE)
    
    chr.id <- pathog.mut.obj$chr.id
    chr.name <- pathog.mut.obj$chr.name
    pathog.mut.type <- pathog.mut.obj$mut.type
    pathog.mut.start <- pathog.mut.obj$mut.start
    pathog.mut.end <- pathog.mut.obj$mut.end
    pathog.mut.ref <- pathog.mut.obj$mut.ref
    pathog.mut.alt <- pathog.mut.obj$mut.alt
    pathog.mut.cds.location <- pathog.mut.obj$mut.cds.location
    pathog.mut.start.to.cds <- pathog.mut.obj$mut.start.to.cds
    pathog.mut.end.to.cds <- pathog.mut.obj$mut.end.to.cds
    minus.strand <- ifelse( pathog.mut.obj$strand=="-", TRUE, FALSE )
    tx.id <- pathog.mut.obj$tx.id
    ref.genome.class <- class(reference.genome)
    
    is.roi.intronic <- FALSE
    is.roi.outOfCds <- FALSE
	is.reads.wt <- FALSE
	is.reads.mt <- FALSE
	is.reads.rpm <- FALSE
	is.rpm.del <- FALSE
	is.rpm.ins <- FALSE
	is.rpm.snv <- FALSE
	is.inframe <- FALSE
	is.ORF <- FALSE
	is.reversion <- FALSE
	pathog.retained <- NA
	rev.ref.df <- NA
	
	if(pathog.mut.type=="INS"){
	    nrow.ins.start <- which(roi.aln.df$ref_pos==pathog.mut.start)
	    if( roi.aln.df$cigar[nrow.ins.start]=="D" & roi.aln.df$cigar[nrow.ins.start+1]=="I" ){
	        nrow.delins.all <- which(roi.aln.df$cigar=="D" & !is.na(roi.aln.df$del) & !is.na(roi.aln.df$ins))
	        nrow.delins.start <- max(nrow.delins.all[nrow.delins.all<=nrow.ins.start])
	        delins.ins.seq <- roi.aln.df$ins[nrow.delins.start]
	        roi.aln.df$ins[nrow.delins.start] <- NA
	        roi.aln.df$ins[nrow.ins.start] <- delins.ins.seq
	    }
	}
	
	roi.aln.df$pathog_mut <- rep(0,nrow(roi.aln.df))
	roi.aln.df$ref_cds <- rep(0,nrow(roi.aln.df))
	roi.aln.df$reads_frame <- rep(0,nrow(roi.aln.df))
	roi.aln.df$cryptic_snv <- rep(0,nrow(roi.aln.df))
	
	roi.aln.df$pathog_mut[ which(roi.aln.df$ref_pos==pathog.mut.start):which(roi.aln.df$ref_pos==pathog.mut.end) ] <- 1

	if( ref.genome.class=="BSgenome" ){
	    
    	cds.noncds <- getCdsAndNoncds(ensdb, tx.id)
    	
    	cds.gr.fwd <- cds.noncds$cds
    	GenomeInfoDb::seqlevels(cds.gr.fwd) <- chr.name
    	GenomeInfoDb::genome(cds.gr.fwd) <- NA
    	BiocGenerics::strand(cds.gr.fwd) <- "+"
    	tx.cds <- BiocGenerics::do.call(c, BSgenome::getSeq(reference.genome, cds.gr.fwd) )
    	if(minus.strand){
    	    tx.cds <- Biostrings::reverseComplement(tx.cds)
    	}
    	
    	cds.ir <- IRanges::ranges(cds.noncds$cds)
    	noncds.ir <- IRanges::ranges(cds.noncds$noncds)
    	ref.ir <- IRanges::IRanges(start=ref.start, end=ref.end)
    	ovlp.ref.cds <- IRanges::overlapsRanges(ref.ir, cds.ir)
    	
    	if( length(ovlp.ref.cds)>0 ){
    	    
    	    for(i in 1:length(ovlp.ref.cds)){
    	        range.i <- ovlp.ref.cds[i]
    	        range.i.start <- BiocGenerics::start(ovlp.ref.cds[i])
    	        range.i.end <- BiocGenerics::end(ovlp.ref.cds[i])
    	        
    	        roi.aln.df$ref_cds[ roi.aln.df$ref_pos%in%(range.i.start:range.i.end) & !is.na(roi.aln.df$ref_pos)] <- 1
    	        
    	        p1 <- which( roi.aln.df$ref_pos==range.i.start & !is.na(roi.aln.df$ref_pos) )
    	        p2 <- which( roi.aln.df$ref_pos==range.i.end & !is.na(roi.aln.df$ref_pos) )
    	        
    	        if( roi.aln.df$cigar[p1]=="D" & is.na(roi.aln.df$del[p1]) ){
    	            p1 <- max( which( roi.aln.df$cigar[1:p1]=="D" & !is.na(roi.aln.df$del[1:p1]) ) )
    	        }
    	        if( roi.aln.df$cigar[p2]=="D" ){
    	            if( is.na(roi.aln.df$del[p2]) ){
    	                p2 <- max( which( roi.aln.df$cigar[1:p2]=="D" & !is.na(roi.aln.df$del[1:p2]) ) )
    	            }
    	            if( is.na(roi.aln.df$ins[p2]) ){
    	                ins.ext <- 0
    	            }else{
    	                ins.ext <- nchar(roi.aln.df$ins[p2])
    	            }
    	            p2 <- p2 + roi.aln.df$del[p2] - 1 + ins.ext
    	        }
    	        if( roi.aln.df$cigar[p2]=="M" & !is.na(roi.aln.df$ins[p2]) ){
    	            roi.aln.df$ins[p2] <- NA
    	        }
    	        roi.aln.df$reads_frame[p1:p2] <- 1
    	    }
    	    
    	    roi.ref.start <- min( roi.aln.df$ref_pos[roi.aln.df$ref_cds==1], na.rm=TRUE )
    	    roi.ref.end <- max( roi.aln.df$ref_pos[roi.aln.df$ref_cds==1], na.rm=TRUE )
    	    roi.ref.start.cds <- genomeToCds(roi.ref.start, tx.id, ensdb)
    	    roi.ref.end.cds <- genomeToCds(roi.ref.end, tx.id, ensdb)
    	    
    	}else{
    	    is.roi.intronic <- TRUE
    	}
    	
	}else if( ref.genome.class=="DNAStringSet" ){
	    
	    tx.cds <- unlist(BSgenome::getSeq(reference.genome, chr.name))
	    
	    roi.aln.df$reads_frame <- rep(1,nrow(roi.aln.df))
	    roi.aln.df$ref_cds <- rep(0,nrow(roi.aln.df))
	    roi.aln.df$ref_cds[ !is.na(roi.aln.df$ref_pos) ] <- 1
	    
	    roi.ref.start <- min( roi.aln.df$ref_pos[roi.aln.df$ref_cds==1], na.rm=TRUE )
	    roi.ref.end <- max( roi.aln.df$ref_pos[roi.aln.df$ref_cds==1], na.rm=TRUE )
	    roi.ref.start.cds <- roi.ref.start
	    roi.ref.end.cds <- roi.ref.end
	}
	
	roi.ref.df <- roi.aln.df[ (roi.aln.df$reads_frame==1 | roi.aln.df$pathog_mut==1) & !is.na(roi.aln.df$ref_index), ]
	roi.ref.df <- roi.ref.df[ order(roi.ref.df$ref_index), ]
	
	if( pathog.mut.type == "SNV" ){
	    snv.genotype <- paste(pathog.mut.ref, pathog.mut.alt, sep=">")
	    true.genotype <- roi.ref.df$snv[roi.ref.df$ref_pos==pathog.mut.start]
	    if( !is.na(true.genotype) & true.genotype==snv.genotype ){
	        is.reads.mt <- TRUE
	    }else{
	        rpm.snv <- true.genotype
	        rpm.del <- roi.ref.df$cigar[ roi.ref.df$ref_pos %in% c( pathog.mut.start:pathog.mut.end ) ]
	        if( !is.na(rpm.snv) ){
	            is.rpm.snv <- TRUE
	        }
	        if( sum(rpm.del=="D") > 0 ){
	            is.rpm.del <- TRUE
	        }
	        if( is.rpm.snv | is.rpm.del ){
	            is.reads.rpm <- TRUE
	        }else{
	            is.reads.wt <- TRUE
	        }
	    }
	}
	
	if( pathog.mut.type == "DEL" ){
	    del.seq.len <- pathog.mut.ref
	    true.del.len <- ifelse( is.na(roi.ref.df$del[roi.ref.df$ref_pos==pathog.mut.start]), 0, roi.ref.df$del[roi.ref.df$ref_pos==pathog.mut.start])
	    
	    if( del.seq.len==true.del.len ){
	        is.reads.mt <- TRUE
	    }else{
	        rpm.snv <- roi.ref.df$snv[ roi.ref.df$ref_pos %in% c( pathog.mut.start:pathog.mut.end ) ]
	        rpm.ins <- roi.ref.df$ins[ roi.ref.df$ref_pos %in% setdiff( pathog.mut.start:pathog.mut.end, pathog.mut.end ) ] 
	        rpm.del <- roi.ref.df$cigar[ roi.ref.df$ref_pos %in% c( pathog.mut.start:pathog.mut.end ) ]
	        if( sum(!is.na(rpm.snv)) > 0 ){
	            is.rpm.snv <- TRUE
	        }
	        if( sum(!is.na(rpm.ins)) > 0 ){
	            is.rpm.ins <- TRUE
	        }
	        if( sum(rpm.del=="D") > 0 ){
	            is.rpm.del <- TRUE
	        }
	        if( is.rpm.snv | is.rpm.ins | is.rpm.del ){
	            is.reads.rpm <- TRUE
	        }else{
	            is.reads.wt <- TRUE
	        }
	    }
	}
	
	if( pathog.mut.type == "INS" ){
	    ins.seq <- pathog.mut.alt
	    true.ins <- roi.ref.df$ins[ roi.ref.df$ref_pos==pathog.mut.start ] 
	    if( !is.na(true.ins) & true.ins==ins.seq ){
	        is.reads.mt <- TRUE
	    }else{
	        rpm.ins <- true.ins
	        rpm.del <- roi.ref.df$cigar[ roi.ref.df$ref_pos %in% c( pathog.mut.start, pathog.mut.end ) ]
	        if( !is.na(rpm.ins) ){
	            is.rpm.ins <- TRUE
	        }
	        if( sum(rpm.del!="D")==0 & is.na(rpm.ins) ){
	            is.rpm.del <- TRUE
	        }
	        if( is.rpm.ins | is.rpm.del ){
	            is.reads.rpm <- TRUE
	        }else{
	            is.reads.wt <- TRUE
	        }
	    }
	}
	
	if( pathog.mut.type == "DELINS" ){
	    del.seq.len <- pathog.mut.ref
	    ins.seq <- pathog.mut.alt
	    true.del.len <- ifelse( is.na(roi.ref.df$del[roi.ref.df$ref_pos==pathog.mut.start]), 0, roi.ref.df$del[roi.ref.df$ref_pos==pathog.mut.start])
	    true.ins <- roi.ref.df$ins[ roi.ref.df$ref_pos==pathog.mut.start ]
	    if( del.seq.len==true.del.len & !is.na(true.ins) & true.ins==ins.seq ){
	        is.reads.mt <- TRUE
	    }else{
	        rpm.snv <- roi.ref.df$snv[ roi.ref.df$ref_pos %in% setdiff( pathog.mut.start:pathog.mut.end, pathog.mut.end ) ]
	        rpm.ins <- roi.ref.df$ins[ roi.ref.df$ref_pos %in% setdiff( pathog.mut.start:pathog.mut.end, pathog.mut.end ) ] 
	        rpm.del <- roi.ref.df$cigar[ roi.ref.df$ref_pos %in% setdiff( pathog.mut.start:pathog.mut.end, pathog.mut.end ) ]
	        if( sum(!is.na(rpm.snv)) > 0 ){
	            is.rpm.snv <- TRUE
	        }
	        if( sum(!is.na(rpm.ins)) > 0 ){
	            is.rpm.ins <- TRUE
	        }
	        if( sum(rpm.del=="D") > 0 ){
	            is.rpm.del <- TRUE
	        }
	        if( is.rpm.snv | is.rpm.ins | is.rpm.del ){
	            is.reads.rpm <- TRUE
	        }else{
	            is.reads.wt <- TRUE
	        }
	    }
	}
	
	if( pathog.mut.type=="SNV" & pathog.mut.cds.location=="exonic" & (is.reads.mt|is.rpm.snv) ){
	    
	    cryptic.snv <- checkExonicSNV(roi.aln.df, minus.strand, pathog.mut.alt, tx.id, ensdb)
	    if(!is.null(cryptic.snv)){
	        roi.aln.df$cryptic_snv[ roi.aln.df$ref_pos%in%cryptic.snv ] <- 1
	    }
	    
	}
	
	if( pathog.mut.type=="SNV" & pathog.mut.cds.location=="exonic.stop" & (is.reads.mt|is.rpm.snv) ){
	    
	    cryptic.snv <- checkExonicSNV(roi.aln.df, minus.strand, pathog.mut.alt, tx.id, ensdb)
	    
	    if(!is.null(cryptic.snv)){
	        
	        roi.aln.df$cryptic_snv[ roi.aln.df$ref_pos%in%cryptic.snv ] <- 1
	        
	    }else{
	        
	        stop.codon <- checkStopCodonSNV(roi.aln.df, minus.strand, pathog.mut.start.to.cds)
	        real.stop.codon <- stop.codon$real.stop.codon
	        cryptic.stop.codon <- stop.codon$cryptic.stop.codon
	        
	        if(!is.null(cryptic.stop.codon)){
	            
	            roi.aln.df$cryptic_snv[ roi.aln.df$ref_pos%in%cryptic.stop.codon & !is.na(roi.aln.df$snv) ] <- 1
	            
                if( !minus.strand ){
                    if( cryptic.stop.codon[3] < real.stop.codon[3] ){
                        roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% (cryptic.stop.codon[3]+1):real.stop.codon[3] ] <- 0
                    }else{
                        roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% (real.stop.codon[3]+1):cryptic.stop.codon[3] ] <- 1
                    }
                }else{
                    if( cryptic.stop.codon[1] > real.stop.codon[1] ){
                        roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% real.stop.codon[1]:(cryptic.stop.codon[1]-1) ] <- 0
                    }else{
                        roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% cryptic.stop.codon[1]:(real.stop.codon[1]-1) ] <- 1
                    }
                }
	            
	        }
	    }
	}
	
	if( pathog.mut.type=="SNV" & pathog.mut.cds.location=="exonic.start" & (is.reads.mt|is.rpm.snv) ){
	    
	    start.codon <- checkStartCodonSNV(roi.aln.df, minus.strand)
	    real.start.codon <- start.codon$real.start.codon
	    cryptic.start.codon <- start.codon$cryptic.start.codon
	    
	    if(!is.null(cryptic.start.codon)){
	        
	        roi.aln.df$cryptic_snv[ roi.aln.df$ref_pos%in%cryptic.start.codon & !is.na(roi.aln.df$snv) ] <- 1
	        
	        if( !minus.strand ){
	            if( cryptic.start.codon[1] > real.start.codon[1] ){
	                roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% real.start.codon[1]:(cryptic.start.codon[1]-1) ] <- 0
	            }else{
	                roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% cryptic.start.codon[1]:(real.start.codon[1]-1) ] <- 1
	            }
	        }else{
	            if( cryptic.start.codon[3] < real.start.codon[3] ){
	                roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% (cryptic.start.codon[3]+1):real.start.codon[3] ] <- 0
	            }else{
	                roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% (real.start.codon[3]+1):cryptic.start.codon[3] ] <- 1
	            }
	        }
	        
	    }
	    
	}
	
	if( pathog.mut.type=="SNV" & pathog.mut.cds.location=="splice.acceptor" & (is.reads.mt|is.rpm.snv) ){
	    
	    acceptors <- checkSpliceAcceptorSNV(roi.aln.df, minus.strand)
	    real.acceptor <- acceptors$real.acceptor
	    cryptic.acceptor <- acceptors$cryptic.acceptor
	    
	    if(!is.null(cryptic.acceptor)){
	        
	        roi.aln.df$cryptic_snv[ roi.aln.df$ref_pos%in%cryptic.acceptor & !is.na(roi.aln.df$snv) ] <- 1
	        
	        if( !minus.strand ){
	            if( cryptic.acceptor[2] > real.acceptor[2] ){
	                roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% (real.acceptor[2]+1):cryptic.acceptor[2] ] <- 0
	            }else{
	                roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% (cryptic.acceptor[2]+1):real.acceptor[2] ] <- 1
	            }
	        }else{
	            if( cryptic.acceptor[1] < real.acceptor[1] ){
	                roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% cryptic.acceptor[1]:(real.acceptor[1]-1) ] <- 0
	            }else{
	                roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% real.acceptor[1]:(cryptic.acceptor[1]-1) ] <- 1
	            }
	        }
	        
	    }
	    
	}
	
	if( pathog.mut.type=="SNV" & pathog.mut.cds.location=="splice.donor" & (is.reads.mt|is.rpm.snv) ){
	    
	    donors <- checkSpliceDonorSNV(roi.aln.df, minus.strand)
	    real.donor <- donors$real.donor
	    cryptic.donor <- donors$cryptic.donor
	    
	    if(!is.null(cryptic.donor)){
	        
	        roi.aln.df$cryptic_snv[ roi.aln.df$ref_pos%in%cryptic.donor & !is.na(roi.aln.df$snv) ] <- 1 
	        
	        if( !minus.strand ){
	            if( cryptic.donor[1] < real.donor[1] ){
	                roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% cryptic.donor[1]:(real.donor[1]-1) ] <- 0 
	            }else{
	                roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% real.donor[1]:(cryptic.donor[1]-1) ] <- 1
	            }
	        }else{
	            if( cryptic.donor[2] > real.donor[2] ){
	                roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% (real.donor[2]+1):cryptic.donor[2] ] <- 0
	            }else{
	                roi.aln.df$reads_frame[ roi.aln.df$ref_pos %in% (cryptic.donor[2]+1):real.donor[2] ] <- 1
	            }
	        }
	        
	    }
	    
	}
	
	roi.ref <- gsub("-", "", paste(roi.aln.df$ref_seq[roi.aln.df$ref_cds==1], collapse=""))
	roi.reads <- gsub("-", "", paste(roi.aln.df$reads_seq[roi.aln.df$reads_frame==1], collapse=""))
	
	if(!is.roi.intronic){
    	is.inframe <- (abs( nchar(roi.reads)-nchar(roi.ref) )%%3 == 0)
    	if(!minus.strand){
    	    rev.cds <- c(
    	        Biostrings::subseq(tx.cds, start=1, end=roi.ref.start.cds-1), 
    	        Biostrings::DNAString(roi.reads),
    	        Biostrings::subseq(tx.cds, start=roi.ref.end.cds+1, end=length(tx.cds)) 
    	    )
    	}else{
    	    rev.cds <- c(
    	        Biostrings::subseq(tx.cds, start=1, end=roi.ref.end.cds-1), 
    	        Biostrings::reverseComplement(Biostrings::DNAString(roi.reads)),
    	        Biostrings::subseq(tx.cds, start=roi.ref.start.cds+1, end=length(tx.cds)) 
    	    )
    	}
    	
    	if(length(rev.cds)==0){
    	    is.roi.outOfCds <- TRUE
    	}else{
    	    rev.aa <- suppressWarnings( Biostrings::translate(rev.cds, if.fuzzy.codon="X") )
    	    if( ref.genome.class=="BSgenome" ){
    	        is.ORF <- grepl("^M[ACDEFGHIKLMNPQRSTVWY]+\\*$", rev.aa)
    	    }else if( ref.genome.class=="DNAStringSet" ){
    	        is.ORF <- grepl("^[ACDEFGHIKLMNPQRSTVWY]+(\\*)?$", rev.aa)
    	    }    
    	}
    	
	}
	
	rev.ref.df <- roi.aln.df[ (roi.aln.df$reads_frame==1 | roi.aln.df$pathog_mut==1 | roi.aln.df$cryptic_snv==1) & !is.na(roi.aln.df$ref_index), ]
	rev.ref.df <- rev.ref.df[ order(rev.ref.df$ref_index), ]
	
	if( is.reads.mt ){
		if( pathog.mut.type=="SNV" ){
			rev.ref.df$snv[ rev.ref.df$cryptic_snv==0  ] <- NA
		}else{
			rev.ref.df$snv <- NA
		}
		if( pathog.mut.type=="DEL" ){
			rev.ref.df$del[ rev.ref.df$ref_pos==pathog.mut.start ] <- NA
		}
		if( pathog.mut.type=="INS" ){
			rev.ref.df$ins[ rev.ref.df$ref_pos==pathog.mut.start ] <- NA
		}
		if( pathog.mut.type=="DELINS" ){
		    rev.ref.df$del[ rev.ref.df$ref_pos==pathog.mut.start ] <- NA
		    rev.ref.df$ins[ rev.ref.df$ref_pos==pathog.mut.start ] <- NA
		}
		rev.ref.df <- rev.ref.df[ !is.na(rev.ref.df$snv) | !is.na(rev.ref.df$ins) | !is.na(rev.ref.df$del) ,]
		if( nrow(rev.ref.df)>0 & is.inframe & is.ORF ){
			is.reversion <- TRUE
			pathog.retained <- "Yes"
		}
	}
	
	if( is.reads.rpm ){
		if( pathog.mut.type=="SNV" ){
			rev.ref.df$snv[ rev.ref.df$cryptic_snv==0 ] <- NA
		}
	    if(pathog.mut.type=="DEL"){
	        rev.ref.df$snv[ !(rev.ref.df$ref_pos %in% c(pathog.mut.start:pathog.mut.end)) ] <- NA
	    }
		if( pathog.mut.type=="INS" ){
			rev.ref.df$snv <- NA
		}
		if( pathog.mut.type=="DELINS" ){
		    rev.ref.df$snv[ !(rev.ref.df$ref_pos %in% setdiff( pathog.mut.start:pathog.mut.end, pathog.mut.end )) ] <- NA
		}
		rev.ref.df <- rev.ref.df[ !is.na(rev.ref.df$snv) | !is.na(rev.ref.df$ins) | !is.na(rev.ref.df$del) ,]
		if( nrow(rev.ref.df)>0 & is.inframe & is.ORF ){
			is.reversion <- TRUE
			pathog.retained <- "No"
		}
	}
	
	if( is.reads.wt & check.wildtype.reads ){
		rev.ref.df$snv <- NA
		rev.ref.df$snv[rev.ref.df$ref_pos==pathog.mut.start] <- "MT>WT"
		rev.ref.df <- rev.ref.df[ !is.na(rev.ref.df$snv) | !is.na(rev.ref.df$ins) | !is.na(rev.ref.df$del),]
		if( nrow(rev.ref.df)>0 & is.inframe & is.ORF ){
			is.reversion <- TRUE
			pathog.retained <- "WT"
		}
	}
	
	roi.rev <- list(
		is.reads.wt = is.reads.wt,
		is.reads.mt = is.reads.mt,
		is.reads.rpm = is.reads.rpm,
		is.reversion = is.reversion,
		pathog.retained = pathog.retained,
		reversions = rev.ref.df
	)
	
	return(roi.rev)
}


