#' Detect reversions from a single read
#'
#' @param reference.genome A BSgenome or DNAStringSet object of reference sequences
#' @param ensdb An EnsDb object of Ensembl-based annotation database
#' @param reads.obj A read object
#' @param pathog.mut.obj A pathogenic mutation object
#' @param roi.start The start position of reversion detection window
#' @param roi.end The end position of reversion detection window
#' @param check.soft.clipping A logical value indicating whether soft-clipped reads to be realigned
#' @param softClippedReads.realign.match A non-negative integer specifying the scoring for a nucleotide match for realigning soft-clipped reads
#' @param softClippedReads.realign.mismatch A non-negative integer specifying the scoring for a nucleotide mismatch for realigning soft-clipped reads
#' @param softClippedReads.realign.gapOpening A non-negative integer specifying the cost for opening a gap in the realignment of soft-clipped reads
#' @param softClippedReads.realign.gapExtension A non-negative integer specifying the incremental cost incurred along the length of the gap in the realignment of soft-clipped reads
#' @param softClippedReads.realign.window.left A non-negative integer specifying the length of flanking region to be added to the left end of the original alignment region for realigning soft-clipped reads
#' @param softClippedReads.realign.window.right A non-negative integer specifying the length of flanking region to be added to the right end of the original alignment region for realigning soft-clipped reads
#' @param check.wildtype.reads A logical value indicating whether wild type reads to be processed as revertant-to-wildtype reads
#' 
#' @return A list of reversion call for a read including realigned read object, flags for read types, reversion event, reversion information table
#'
#' @examples
#' \dontrun{
#'     callSingleRead()
#' }
#'
#' @noRd
callSingleRead <- function(
        reference.genome,
        ensdb, 
        reads.obj,
        pathog.mut.obj,
        roi.start,
        roi.end,
        check.soft.clipping,
        softClippedReads.realign.match,
        softClippedReads.realign.mismatch,
        softClippedReads.realign.gapOpening,
        softClippedReads.realign.gapExtension,
        softClippedReads.realign.window.left,
        softClippedReads.realign.window.right,
        check.wildtype.reads
	){
    
    chr.name <- pathog.mut.obj$chr.name
    pathog.mut.type <- pathog.mut.obj$mut.type
    pathog.mut.start <- pathog.mut.obj$mut.start
    pathog.mut.end <- pathog.mut.obj$mut.end
    pathog.mut.ref <- pathog.mut.obj$mut.ref
    pathog.mut.alt <- pathog.mut.obj$mut.alt
    pathog.mut.hgvs <- pathog.mut.obj$mut.hgvs
    ref.genome.class <- class(reference.genome)
    
    is.reads.failed <- FALSE
	is.reads.target <- FALSE
	is.reads.wt <- FALSE
	is.reads.mt <- FALSE
	is.reads.rpm <- FALSE
	is.reversion <- FALSE
	rev.event <- NA
	rev.tbl <- NA
	
	realn.reads <- realignSingleRead(
	    reference.genome,
	    reads.obj,
	    pathog.mut.obj, 	
	    check.soft.clipping,
	    softClippedReads.realign.match,
	    softClippedReads.realign.mismatch,
	    softClippedReads.realign.gapOpening,
	    softClippedReads.realign.gapExtension,
	    softClippedReads.realign.window.left,
	    softClippedReads.realign.window.right,
	    indel.shifting="left" 
	)
	
	is.reads.target <- realn.reads$is.reads.target
	reads.obj.out <- realn.reads$realigned.reads.obj
	realn <- realn.reads$realigned.sequences
	aln.pos <- reads.obj.out$POS
	
	if( is.reads.target ){
		realn.cigar <- encodeCIGAR(realn[1], realn[2])
		realn.df <- decodeCIGAR(
			realn.cigar, 
			gsub("-","",realn[1]), 
			gsub("-","",realn[2])
		)
		realn.df$ref_pos <- aln.pos + realn.df$ref_index - 1
		ref.start <- aln.pos
		ref.end <- aln.pos + nchar(gsub("-","",realn[1])) - 1 
		
		if( ref.start <= pathog.mut.start & ref.end >= pathog.mut.end ){
    		roi.aln.df <- extractDetectionWindow(realn.df, ref.start, ref.end, roi.start, roi.end)
    		roi.rev <- scanAlnForRev(roi.aln.df, pathog.mut.obj, reference.genome, ensdb, check.wildtype.reads)
    		is.reads.wt <- roi.rev$is.reads.wt
    		is.reads.mt <- roi.rev$is.reads.mt
    		is.reads.rpm <- roi.rev$is.reads.rpm
    		is.reversion <- roi.rev$is.reversion
    		pathog.retained <- roi.rev$pathog.retained
    		rev.ref.df <- roi.rev$reversions
		}
		
		
		if( is.reversion ){
		    
		    rev.ref.df$delins <- ifelse( !is.na(rev.ref.df$del) & !is.na(rev.ref.df$ins), paste0(rev.ref.df$del, ">", rev.ref.df$ins), NA )
		    rev.ref.df[ !is.na(rev.ref.df$del) & !is.na(rev.ref.df$ins), c("del","ins") ] <- NA
			rev.tbl <- data.frame(
				mut_start_pos = rep( rev.ref.df$ref_pos, each=4 ),
				mut_type = rep( c("SNV","DEL","INS","DELINS"), nrow(rev.ref.df) ),
				mut_seq = as.vector( t(rev.ref.df[,c("snv","del","ins","delins")]) ),
				stringsAsFactors = FALSE  
			)
			rev.tbl <- rev.tbl[ !is.na(rev.tbl$mut_seq), ]
			
			rev.tbl$mut_length_change <- suppressWarnings(
			    ifelse( 
    			    rev.tbl$mut_type=="SNV", 
    			    0, 
    			    ifelse(
    			        rev.tbl$mut_type=="DEL",
    			        (-1) * as.integer(rev.tbl$mut_seq),
    			        ifelse(
    			            rev.tbl$mut_type=="INS",
    			            nchar(rev.tbl$mut_seq),
    			            sapply(rev.tbl$mut_seq, function(x){ nchar(strsplit(x, ">")[[1]][2]) - as.integer(strsplit(x, ">")[[1]][1]) }) 
    			        )
    			    )
                )
			)
			
			rev.tbl$mut_end_pos <- suppressWarnings(
			    ifelse( 
    			    rev.tbl$mut_type=="SNV", 
    			    rev.tbl$mut_start_pos, 
    			    ifelse(
    			        rev.tbl$mut_type=="DEL",
    			        rev.tbl$mut_start_pos + as.integer(rev.tbl$mut_seq) - 1,
    			        ifelse(
    			            rev.tbl$mut_type=="INS",
    			            rev.tbl$mut_start_pos + 1,
    			            apply(rev.tbl[,c("mut_start_pos","mut_seq")], 1, function(x){ as.integer(x[1]) + as.integer(strsplit(x[2], ">")[[1]][1]) }) 
    			        )
    			    )
			    )
			)
			
			rev.tbl$mut_coord <- suppressWarnings(
			    ifelse( 
    			    rev.tbl$mut_type=="SNV", 
    			    paste0(rev.tbl$mut_start_pos, rev.tbl$mut_seq), 
    			    ifelse(
    			        rev.tbl$mut_type=="DEL",
    			        ifelse( rev.tbl$mut_length_change==-1, paste0(rev.tbl$mut_start_pos,"del"), paste0(rev.tbl$mut_start_pos,"_",rev.tbl$mut_end_pos,"del") ),
    			        ifelse(
    			            rev.tbl$mut_type=="INS",
    			            paste0(rev.tbl$mut_start_pos,"_",rev.tbl$mut_end_pos,"ins",rev.tbl$mut_seq),
                            apply(
                                rev.tbl[,c("mut_start_pos","mut_seq")], 
                                1, 
                                function(x){
                                    ifelse(
                                        as.integer(strsplit(x[2], ">")[[1]][1])==1,
                                        paste0(as.integer(x[1]),"delins",strsplit(x[2], ">")[[1]][2]),
                                        paste0(as.integer(x[1]),"_",as.integer(x[1])+as.integer(strsplit(x[2], ">")[[1]][1])-1,"delins",strsplit(x[2], ">")[[1]][2])
                                    )
                                }  
                            )
    			        )
    			    )
			    )
			)
			
			rev.tbl$pathogenic_mut_retained <- rep( pathog.retained, nrow(rev.tbl) )
			rev.tbl$rev_mut_number <- 1:nrow(rev.tbl)
			rev.tbl$chr <- rep( chr.name, nrow(rev.tbl) )
			rev.tbl$pathog_mut_hgvs <- rep(pathog.mut.hgvs, nrow(rev.tbl))
			rev.tbl$pathog_mut_start <- rep(pathog.mut.start, nrow(rev.tbl))
			rev.tbl$pathog_mut_end <- rep(pathog.mut.end, nrow(rev.tbl))
			
			rev.tbl$dist_to_pathog_mut <- mapply(
				getMutationsDistance,
				rev.tbl$pathog_mut_start,
				rev.tbl$pathog_mut_end,
				rep(pathog.mut.type, nrow(rev.tbl)),
				rev.tbl$mut_start_pos,
				rev.tbl$mut_end_pos,
				rev.tbl$mut_type,
				SIMPLIFY = TRUE
			)
			
			if( ref.genome.class=="BSgenome" ){
    			rev.event <- paste0( pathog.retained, "|", chr.name, ":g.", paste(rev.tbl$mut_coord, collapse=";") )
    			rev.tbl$mut_hgvs <- paste0( chr.name, ":g.", rev.tbl$mut_coord )
			}else if( ref.genome.class=="DNAStringSet" ){
			    rev.event <- paste0( pathog.retained, "|", chr.name, ":", paste(rev.tbl$mut_coord, collapse=";") )
			    rev.tbl$mut_hgvs <- paste0( chr.name, ":", rev.tbl$mut_coord )
			}
			
			if(check.wildtype.reads){
			    rev.tbl$mut_type[rev.tbl$mut_seq=="MT>WT"] <- "WT"
			}
		}
	} 	
	
	flags <- list(
	    is.reads.failed = is.reads.failed,
		is.reads.target = is.reads.target,
		is.reads.wt = is.reads.wt,
		is.reads.mt = is.reads.mt,
		is.reads.rpm = is.reads.rpm,
		is.reversion = is.reversion
	)
	
	rev.call <- list(
		realigned.reads.obj = reads.obj.out,
		flags = flags,
		rev.event = rev.event,
		rev.tbl = rev.tbl
	)
	return(rev.call)
} 


