% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/initialise.R
\name{initialise}
\alias{initialise}
\title{Prepare Data Required To Perform A Simulation}
\usage{
initialise(
  n1_map,
  K_map,
  K_sd = 0,
  r,
  r_sd = 0,
  growth = "gompertz",
  A = NA,
  dens_dep = c("K2N", "K", "none"),
  border = c("reprising", "absorbing"),
  kernel_fun = "rexp",
  ...,
  max_dist = NA,
  calculate_dist = TRUE,
  dlist = NULL,
  progress_bar = TRUE,
  quiet = FALSE
)
}
\arguments{
\item{n1_map}{\code{\link[terra:SpatRaster-class]{SpatRaster}} object with one layer;
population numbers in every grid cell at the first time step}

\item{K_map}{\code{\link[terra:SpatRaster-class]{SpatRaster}} object with one layer;
carrying capacity map (if K is constant across time) or maps (if K is
time-varying)}

\item{K_sd}{numeric vector of length 1 with value \verb{>= 0} (default 0);
this parameter can be used if additional environmental stochasticity
is required; if \code{K_sd > 0}, random numbers are generated from a log-normal
distribution with the mean \code{K_map} and standard deviation \code{K_sd}}

\item{r}{numeric vector of length 1; intrinsic population growth rate}

\item{r_sd}{numeric vector of length 1 with value \verb{>= 0} (default \code{0});
if additional demographic stochasticity is required, \code{r_sd > 0} is
the standard deviation for a normal distribution around \code{r}
(defined for each time step)}

\item{growth}{character vector of length 1; the name of a population growth
function, either defined in \code{\link{growth}} or provided by
the user (case-sensitive, default \code{\link[=growth]{"gompertz"}})}

\item{A}{numeric vector of length 1; strength of the Allee effect
(see the \code{\link{growth}} function)}

\item{dens_dep}{character vector of length 1 specifying if the probability
of settling in a target grid cell is (case-sensitive, default \code{"K2N"}):
\itemize{
\item "none" - fully random,
\item "K" - proportional to the carrying capacity of a target cell,
\item "K2N" - density-dependent, i.e. proportional to the ratio of
carrying capacity of a target cell to the number of individuals
already present in a target cell
}}

\item{border}{character vector of length 1 defining how to deal
with borders (case-sensitive, default \code{"absorbing"}):
\itemize{
\item "reprising" - cells outside the study area are not allowed
as targets for dispersal
\item "absorbing" - individuals that disperse outside the study area
are removed from the population
}}

\item{kernel_fun}{character vector of length 1; name of a random number
generation function defining a dispersal kernel (case-sensitive,
default \code{"rexp"})}

\item{...}{any parameters required by \code{kernel_fun}}

\item{max_dist}{numeric vector of length 1; maximum distance of dispersal
to pre-calculate target cells}

\item{calculate_dist}{logical vector of length 1; determines if target cells
will be precalculated}

\item{dlist}{list; target cells at a specified distance calculated
for every cell within the study area}

\item{progress_bar}{logical vector of length 1; determines if progress bar
for calculating distances should be displayed}

\item{quiet}{logical vector of length 1; determines if messages should be displayed}
}
\value{
Object of class \code{sim_data} which inherits from \code{list}. This object
contains all necessary information to perform a simulation using
\code{\link{sim}} function.
}
\description{
This function generates a \code{sim_data} object containing all the necessary
information required to run a simulation by the \code{\link{sim}} function. The
input maps (\code{n1_map} and \code{K_map}) can be in the Cartesian or longitude/latitude
coordinate system.
}
\details{
The most time-consuming part of computations performed by the \code{\link{sim}}
function is the simulation of dispersal. To speed it up, a list containing
indexes of target cells at a specified distance from a focal cell
is calculated in advance and stored in a \code{dlist} slot.
The \code{max_dist} parameter sets
the maximum distance at which this pre-calculation is performed. If \code{max_dist}
is \code{NULL}, it is set to 0.99 quantile from the \code{kernel_fun}.
All distance calculations are always based on metres if the input maps are
latitude/longitude. For planar input maps, distances are calculated in map
units, which are typically metres, but check the \code{\link[terra:crs]{crs()}}
if in doubt.

If the input maps are in the Cartesian coordinate system and the grid cells
are squares, then
the distances between cells are calculated using the \code{\link[terra:distance]{distance}}
function from the \code{terra} package. These distances are later divided by the
resolution of the input maps.

For input maps with grid cells in shapes other than squares (e.g. with
rectangular cells or longitude/latitude coordinate system), the distance
resolution is calculated by finding the shortest distance between each
"queen" type neighbor. All distances calculated by the \code{\link[terra:distance]{distance}}
function are further divided by this distance resolution.
To avoid discontinuities in the distances at which the target cells are located,
an additional parameter \code{dist_bin} is calculated as half of the maximum
distance between each "queen" type neighbour. It is used to expand the
distances at which target cells are located from a single number to a range.

NA in the input maps represents cells outside the study area.

The \code{\link{K_get_interpolation}} function can be used to prepare \code{K_map} that changes
over time. This may be useful, when simulating environmental change or
exploring the effects of ecological disturbances.
}
\examples{
\donttest{

# input maps
library(terra)

n1_small <- rast(system.file("input_maps/n1_small.tif", package = "rangr"))
K_small <- rast(system.file("input_maps/K_small.tif", package = "rangr"))
K_small_changing <- rast(system.file("input_maps/K_small_changing.tif",
                         package = "rangr"))
n1_small_lon_lat <- rast(system.file("input_maps/n1_small_lon_lat.tif", package = "rangr"))
K_small_lon_lat <- rast(system.file("input_maps/K_small_lon_lat.tif", package = "rangr"))

# basic example
sim_data_1 <- initialise(
  n1_map = n1_small,
  K_map = K_small,
  r = log(2),
  rate = 1 / 1e3
)

# example with changing environment
K_interpolated <- K_get_interpolation(
  K_small_changing,
  K_time_points = c(1, 25, 50)
)

sim_data_2 <- initialise(
  n1_map = n1_small,
  K_map = K_interpolated,
  r = log(2),
  rate = 1 / 1e3
)

# example with lon/lat rasters
sim_data_3 <- initialise(
  n1_map = n1_small_lon_lat,
  K_map = K_small_lon_lat,
  r = log(2),
  rate = 1 / 1e3
)

# example without progress bar and messages
sim_data_4 <- initialise(
  n1_map = n1_small, K_map = K_small, K_sd = 0.1, r = log(5),
  r_sd = 4, growth = "ricker", rate = 1 / 200,
  max_dist = 5000, dens_dep = "K2N", progress_bar = FALSE, quiet = TRUE
)
}


}
\references{
Hijmans R (2024). terra: Spatial Data Analysis. R package version
1.7-81, \url{https://rspatial.github.io/terra/}, \url{https://rspatial.org/}

Solymos P, Zawadzki Z (2023). pbapply: Adding Progress Bar to '*apply' Functions. R
package version 1.7-2, \url{https://CRAN.R-project.org/package=pbapply}.
}
\seealso{
\link[=update.sim_data]{update}
}
