% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spec_interp_absolute_targets.R
\name{spec_interp_absolute_targets}
\alias{spec_interp_absolute_targets}
\title{Specify targets based on interpolating absolute thresholds}
\usage{
spec_interp_absolute_targets(
  rare_absolute_threshold,
  rare_relative_target,
  rare_absolute_target,
  rare_method,
  common_absolute_threshold,
  common_relative_target,
  common_absolute_target,
  common_method,
  cap_absolute_target,
  interp_method
)
}
\arguments{
\item{rare_absolute_threshold}{\code{numeric} value indicating the absolute
threshold for identifying rare features.
This value must be expressed in the same units as the feature data.
In particular, features with a total abundance
smaller than this value will be considered rare during the target setting
calculations.}

\item{rare_relative_target}{\code{numeric} value indicating the
relative target for rare features.
Note that this value must be a proportion between 0 and 1.
For example, a value of 0.1 corresponds to 10\%.}

\item{rare_absolute_target}{\code{numeric} value denoting the
absolute target for rare features.
This value must be expressed in the same units as the feature data.
To avoid setting an absolute target for rare features,
a missing (\code{NA}) value can be specified.}

\item{rare_method}{\code{character} value indicating how the target for rare
features should be calculated. Available options include \code{"min"} and \code{"max"}.
For example, a value of \code{"max"} means that the target for a rare features
is calculated as the maximum based on \code{rare_relative_target} and
\code{rare_absolute_target}. Note that \code{rare_method} will have no effect on
the target calculations if \code{rare_absolute_target} is a missing (\code{NA}) value.}

\item{common_absolute_threshold}{\code{numeric} value indicating the
absolute threshold for identifying common features.
This value must be expressed in the same units as the feature data.
In particular, features with a total abundance
greater than this value will be considered common during the target setting
calculations.}

\item{common_relative_target}{\code{numeric} value denoting the
relative target for common features.
Note that this value must be a proportion between 0 and 1.
For example, a value of 0.1 corresponds to 10\%.}

\item{common_absolute_target}{\code{numeric} value denoting the
absolute target for common features.
This value must be expressed in the same units as the feature data.
To avoid setting an absolute target for common features,
a missing (\code{NA}) value can be specified.}

\item{common_method}{\code{character} value indicating how the target for common
features should be calculated. Available options include \code{"min"} and \code{"max"}.
For example, a value of \code{"max"} means that the target for a common feature
is calculated as the maximum based on \code{common_relative_target} and
\code{common_absolute_target}. Note that \code{common_method} will have no effect on
the target calculations if \code{common_absolute_target} is a missing (\code{NA})
value.}

\item{cap_absolute_target}{\code{numeric} value denoting the absolute target
cap.
This value must be expressed in the same units as the feature data.
In particular, all targets are clamped to this value during target setting
calculations.
To avoid setting a target cap,
a missing (\code{NA}) value can be specified.}

\item{interp_method}{\code{character} value denoting the interpolation method.
Available options include \code{"linear"} for linear interpolation and
\code{"log10"} for log-linear interpolation.}
}
\value{
An object (\code{\linkS4class{TargetMethod}}) for specifying targets that
can be used with \code{\link[=add_auto_targets]{add_auto_targets()}} and \code{\link[=add_group_targets]{add_group_targets()}}
to add targets to a \code{\link[=problem]{problem()}}.
}
\description{
Specify targets by interpolating them between thresholds expressed as the
same values as the underlying feature data (ignoring any specified
feature units).
Briefly, this method involves
(i) setting target thresholds for rare features to a particular percentage
threshold, (ii) setting target thresholds for common features
to a particular percentage threshold, and (iii) interpolating
target thresholds for features with spatial distributions that
range between the those for the rare and common features.
Additionally, features can (optionally) have their targets capped at a
particular threshold.
This method is especially useful for setting targets based on
interpolation procedures when features do not have data expressed as an
area-based unit of measurement.
Note that this function is designed to be used with \code{\link[=add_auto_targets]{add_auto_targets()}}
and \code{\link[=add_group_targets]{add_group_targets()}}.
}
\details{
This method has been applied to set target thresholds at global and national
scales (e.g., Butchart \emph{et al.} 2015;
Rodrigues \emph{et al.} 2004; Polak \emph{et al.} 2015).
It is based on the rationale that species with a smaller geographic
distribution are at a greater risk of extinction, and so require
a larger percentage of their geographic distribution to be represented
by a prioritization (Rodrigues \emph{et al.} 2004).
When using this method in a planning exercise, it is important to ensure
that the threshold parameters reflect the stakeholder objectives.
Additionally, the threshold parameters may need to set according to
the spatial extent of the planning region.
}
\section{Mathematical formulation}{

This method provides a flexible approach for setting target thresholds based
on an interpolation procedure and the feature data.
To express this mathematically, we will define the following terminology.
Let \eqn{f} denote the total abundance of a feature,
\eqn{a} the threshold for identifying rare features
(per \code{rare_absolute_threshold}),
\eqn{b} the relative targets for rare features
(per \code{rare_relative_target}),
\eqn{c} the absolute targets for rare features
(per \code{rare_absolute_target}),
\eqn{d()} the function for calculating targets for rare features
as a maximum or minimum value (per \code{rare_method}),
\eqn{e} the threshold for identifying common features
(per \code{common_absolute_threshold}),
\eqn{g} the relative targets for common features
(per \code{common_relative_target}),
\eqn{h} the absolute targets for common features
(per \code{common_absolute_target}),
\eqn{i()} the method for calculating targets for common features
as a maximum or minimum value  (per \code{common_method}),
\eqn{j} the target cap (per \code{cap_absolute_target}), and
\eqn{k()} the interpolation method for features with a spatial distribution
that is larger than a rare features and smaller than a common feature
(per \code{interp_method}).
In particular, \eqn{k()} is either a linear or log-linear interpolation
procedure based on the thresholds for identifying rare and common features
as well as the relative targets for rare and common features.
Given this terminology, the target threshold (\eqn{t}) for the feature
is calculated as follows.
\itemize{
\item If \eqn{f < a}, then \eqn{
t = min(d(c, b \times f), j)}{
t = min(d(c, b * f), j)
}.
\item If \eqn{f > e}, then \eqn{
t = min(i(h, g \times f), j)}{.
t = min(i(h, g * f), j)
}.
\item If \eqn{a \leq f \leq e}{a <= f <= e}, then
\eqn{t = min(k(f, a, b, e, g), j)}.
}
}

\examples{
\dontrun{
# set seed for reproducibility
set.seed(500)

# load data
sim_pu_raster <- get_sim_pu_raster()
sim_features <- get_sim_features()

# this function sets targets based on the total abundance of the features
# (i.e., sum of planning unit values for the feature) and does not
# consider the spatial area covered by the planning units

# display the total abundance of the features
print(terra::global(get_sim_features(), "sum", na.rm = TRUE))

# create problem with interpolated targets.
# here, targets will be set as 70\% for features with a total abundance
# (i.e., sum of planning unit values for the feature) smaller than 50,
# 20\% for features with at total abundance greater than 70,
# linearly interpolated for features with an intermediate range size,
# and capped at a total abundance of 100
p1 <-
  problem(sim_pu_raster, sim_features) \%>\%
  add_min_set_objective() \%>\%
  add_auto_targets(
    method = spec_interp_absolute_targets(
      rare_absolute_threshold = 50,
      rare_relative_target = 0.7,
      rare_absolute_target = NA,            # not used
      rare_method = "max",                  # not used
      common_absolute_threshold = 70,
      common_relative_target = 0.2,
      common_absolute_target = NA,          # not used
      common_method = "max",                # not used
      cap_absolute_target = 100,
      interp_method = "linear"
    )
  ) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# solve problem
s1 <- solve(p1)

# plot solution
plot(s1, main = "solution", axes = FALSE)
}
}
\references{
Butchart SHM, Clarke M, Smith RJ, Sykes RE, Scharlemann JPW, Harfoot M,
Buchanan GM, Angulo A, Balmford A, Bertzky B, Brooks TM, Carpenter KE,
Comeros‐Raynal MT, Cornell J, Ficetola GF, Fishpool LDC, Fuller RA,
Geldmann J, Harwell H, Hilton‐Taylor C, Hoffmann M, Joolia A, Joppa L,
Kingston N, May I, Milam A, Polidoro B, Ralph G, Richman N, Rondinini C,
Segan DB, Skolnik B, Spalding MD, Stuart SN, Symes A, Taylor J, Visconti P,
Watson JEM, Wood L, Burgess ND (2015) Shortfalls and solutions for meeting
national and global conservation area targets. \emph{Conservation Letters},
8: 329--337.

Polak T, Watson JEM, Fuller RA, Joseph LN, Martin TG, Possingham HP,
Venter O, Carwardine J (2015) Efficient expansion of global protected areas
requires simultaneous planning for species and ecosystems.
\emph{Royal Society Open Science}, 2: 150107.

Rodrigues ASL, Akçakaya HR, Andelman SJ, Bakarr MI, Boitani L, Brooks TM,
Chanson JS, Fishpool LDC, Da Fonseca GAB, Gaston KJ, Hoffmann M, Marquet PA,
Pilgrim JD, Pressey RL, Schipper J, Sechrest W, Stuart SN, Underhill LG,
Waller RW, Watts MEJ, Yan X (2004)
Global gap analysis: priority regions for expanding the global
protected-area network. \emph{BioScience}, 54: 1092--1100.
}
\seealso{
Other target setting methods:
\code{\link{spec_absolute_targets}()},
\code{\link{spec_area_targets}()},
\code{\link{spec_duran_targets}()},
\code{\link{spec_interp_area_targets}()},
\code{\link{spec_jung_targets}()},
\code{\link{spec_max_targets}()},
\code{\link{spec_min_targets}()},
\code{\link{spec_polak_targets}()},
\code{\link{spec_pop_size_targets}()},
\code{\link{spec_relative_targets}()},
\code{\link{spec_rl_ecosystem_targets}()},
\code{\link{spec_rl_species_targets}()},
\code{\link{spec_rodrigues_targets}()},
\code{\link{spec_rule_targets}()},
\code{\link{spec_ward_targets}()},
\code{\link{spec_watson_targets}()},
\code{\link{spec_wilson_targets}()}
}
\concept{methods}
