% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/univariate.r
\name{powerprior_univariate}
\alias{powerprior_univariate}
\title{Compute Power Prior for Univariate Normal Data}
\usage{
powerprior_univariate(
  historical_data,
  a0,
  mu0 = NULL,
  kappa0 = NULL,
  nu0 = NULL,
  sigma2_0 = NULL
)
}
\arguments{
\item{historical_data}{Numeric vector of historical observations. Must contain
at least 2 observations. Missing values (NAs) are automatically removed.
The function assumes data are independent and identically distributed from
a normal distribution with unknown mean and variance.}

\item{a0}{Discounting parameter in \verb{[0, 1]}. Controls the degree of borrowing from
historical data. Specific values have intuitive interpretations:
\itemize{
\item a0 = 0: No borrowing from historical data; power prior equals the initial prior
\item a0 = 0.5: Moderate borrowing; historical data contribute with half weight
\item a0 = 1: Full borrowing; historical data weighted equally with current data
}
The parameter acts as a multiplicative discount on the historical likelihood.}

\item{mu0}{Prior mean for the normal distribution of the mean parameter.
Only used when specifying an informative initial prior. If NULL, a vague
(non-informative) prior is used instead. Represents the prior belief about
the center of the data distribution.}

\item{kappa0}{Prior precision parameter (inverse of scaled variance) for the mean.
Only used for informative priors. If NULL, vague prior is applied. Higher values
indicate greater prior confidence in mu0. Interpreted as the "prior sample size"
contributing to the mean estimate. For example, kappa0 = 1 is roughly equivalent
to one prior observation.}

\item{nu0}{Prior degrees of freedom for the inverse chi-squared distribution of
variance. Only used for informative priors. If NULL, vague prior is applied.
Higher values indicate greater prior confidence in sigma2_0. Typically set to
small positive values (e.g., 1-5) for weakly informative priors.}

\item{sigma2_0}{Prior scale parameter for the inverse chi-squared distribution.
Only used for informative priors. If NULL, vague prior is applied.
Represents the prior belief about the scale (spread) of the data. Should be
on the variance scale (not standard deviation).}
}
\value{
A list of class "powerprior_univariate" containing:
\item{mu_n}{Updated posterior mean parameter from power prior}
\item{kappa_n}{Updated posterior precision parameter (effective sample size)}
\item{nu_n}{Updated posterior degrees of freedom}
\item{sigma2_n}{Updated posterior scale parameter (variance scale)}
\item{a0}{Discounting parameter used}
\item{m}{Sample size of historical data}
\item{xbar}{Sample mean of historical data}
\item{Sx}{Sum of squared deviations of historical data}
\item{vague_prior}{Logical indicating if vague prior was used}
\item{mu0}{Prior mean (if informative prior used)}
\item{kappa0}{Prior precision (if informative prior used)}
\item{nu0}{Prior degrees of freedom (if informative prior used)}
\item{sigma2_0}{Prior scale parameter (if informative prior used)}
}
\description{
Computes the power prior for univariate normal data using a conjugate
Normal-Inverse-Chi-squared (NIX) representation.
}
\details{
\subsection{Background on Power Priors}{

Power priors provide a principled framework for incorporating historical information
in Bayesian analysis while controlling the degree of borrowing through a discounting
parameter. The power prior is defined as:

\deqn{P(\theta | x, a_0) = L(\theta | x)^{a_0} P(\theta)}

where

\itemize{
\item \eqn{L(\theta | x)} is the likelihood function evaluated on historical data \eqn{x}
\item \eqn{a_0 \in [0, 1]} is the discounting parameter
\item \eqn{P(\theta)} is the initial prior distribution
}

This approach is especially valuable in clinical trial design, where historical trial
data can improve statistical efficiency while maintaining appropriate skepticism about
whether historical and current populations are similar.
}

\subsection{Conjugacy and Computational Efficiency}{

Typically, power priors result in non-closed-form posterior distributions requiring
computationally expensive Markov Chain Monte Carlo (MCMC) sampling, especially
problematic for simulation studies requiring thousands of posterior samples.

This implementation exploits a key mathematical result: when applying power priors to
normal data with conjugate initial priors (Normal-Inverse-Chi-squared), the resulting
power prior and posterior distributions remain in closed form as NIX distributions.
This allows:

\itemize{
\item Direct computation without MCMC approximation
\item Closed-form parameter updates
\item Exact posterior sampling from standard distributions
\item Efficient sensitivity analyses across different a0 values
}
}

\subsection{Informative vs. Vague Priors}{

\strong{Informative Initial Prior (all of mu0, kappa0, nu0, sigma2_0 provided):}

Uses a Normal-Inverse-Chi-squared (NIX) conjugate prior with structure:

\deqn{\mu | \sigma^2 \sim N(\mu_0, \sigma^2 / \kappa_0)}
\deqn{\sigma^2 \sim \text{Inv-}\chi^2(\nu_0, \sigma^2_0)}

The power prior parameters are updated:

\deqn{\mu_n = \frac{a_0 m \bar{x} + \kappa_0 \mu_0}{a_0 m + \kappa_0}}

\deqn{\kappa_n = a_0 m + \kappa_0}

\deqn{\nu_n = a_0 m + \nu_0}

\deqn{\sigma^2_n = \frac{1}{\nu_n} \left[ a_0 S_x + \nu_0 \sigma^2_0 +
  \frac{a_0 m \kappa_0 (\bar{x} - \mu_0)^2}{a_0 m + \kappa_0} \right]}

where \eqn{m} is the sample size, \eqn{\bar{x}} is the sample mean, and
\eqn{S_x = \sum_{i=1}^m (x_i - \bar{x})^2} is the sum of squared deviations.

\strong{Vague (Non-informative) Initial Prior (all of mu0, kappa0, nu0, sigma2_0 are NULL):}

Uses a non-informative prior \eqn{P(\mu, \sigma^2) \propto \sigma^{-2}} that is
locally uniform in log-space. This places minimal prior constraints on the parameters.
The power prior parameters simplify to:

\deqn{\mu_n = \bar{x}}

\deqn{\kappa_n = a_0 m}

\deqn{\nu_n = a_0 m - 1}

\deqn{\sigma^2_n = \frac{a_0 S_x}{\nu_n}}

The vague prior approach is recommended when there is no strong prior information,
or when you want the analysis to be primarily driven by the (discounted) historical data.
}

\subsection{Parameter Interpretation}{

\strong{Effective Sample Size (kappa_n):} The updated precision parameter can be interpreted
as an effective sample size. Higher values indicate more concentrated posterior distributions
for the mean. The formula \eqn{\kappa_n = a_0 m + \kappa_0} shows that the historical
sample size \eqn{m} is discounted by \eqn{a_0} before combining with the prior's
contribution \eqn{\kappa_0}.

\strong{Posterior Mean (mu_n):} A weighted average of the historical sample mean \eqn{\bar{x}},
prior mean \eqn{\mu_0}, and their relative precision:
\eqn{\mu_n = \frac{a_0 m \bar{x} + \kappa_0 \mu_0}{a_0 m + \kappa_0}}
This naturally interpolates between the data and prior, with weights determined by
their precision.

\strong{Degrees of Freedom (nu_n):} Controls the tail behavior of posterior distributions
derived from the NIX. Higher values produce lighter tails, indicating greater confidence.

\strong{Scale Parameter (sigma2_n):} Estimates the variability in the data. The term involving
\eqn{(\bar{x} - \mu_0)^2} captures disagreement between the historical mean and prior mean,
which increases uncertainty in variance estimation when they conflict.
}
}
\examples{
# Generate historical data
historical <- rnorm(50, mean = 10, sd = 2)

# Compute power prior with informative initial prior
pp_inform <- powerprior_univariate(
  historical_data = historical,
  a0 = 0.5,
  mu0 = 10,
  kappa0 = 1,
  nu0 = 3,
  sigma2_0 = 4
)
print(pp_inform)

# Compute power prior with vague prior (no prior specification)
pp_vague <- powerprior_univariate(
  historical_data = historical,
  a0 = 0.5
)
print(pp_vague)

# Compare different discounting levels
pp_weak <- powerprior_univariate(historical_data = historical, a0 = 0.1)
pp_strong <- powerprior_univariate(historical_data = historical, a0 = 0.9)
cat("Strong discounting (a0=0.1) - kappa_n:", pp_weak$kappa_n, "\n")
cat("Weak discounting (a0=0.9) - kappa_n:", pp_strong$kappa_n, "\n")

}
\references{
Huang, Y., Yamaguchi, Y., Homma, G., Maruo, K., & Takeda, K. (2024).
"Conjugate Representation of Power Priors for Efficient Bayesian Analysis
of Normal Data." Statistical Science (unpublished).

Ibrahim, J. G., & Chen, M. H. (2000). "Power prior distributions for regression models."
Statistical Science, 15(1), 46-60.

Gelman, A., Carlin, J. B., Stern, H. S., et al. (2013).
Bayesian Data Analysis (3rd ed.). CRC Press.
}
