/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5ESmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5ESpkg.h"     
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 

H5FL_DEFINE_STATIC(H5ES_event_t);

H5ES_event_t *
H5ES__event_new(H5VL_connector_t *connector, void *token)
{
    H5ES_event_t  *ev        = NULL; 
    H5VL_object_t *request   = NULL; 
    H5ES_event_t  *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(connector);
    assert(token);

    
    if (NULL == (request = H5VL_create_object(token, connector))) {
        if (H5VL_request_free(token) < 0)
            HDONE_ERROR(H5E_EVENTSET, H5E_CANTFREE, NULL, "can't free request");
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTINIT, NULL, "can't create vol object for request token");
    } 

    
    if (NULL == (ev = H5FL_CALLOC(H5ES_event_t)))
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTALLOC, NULL, "can't allocate event object");

    
    ev->request = request;

    
    ret_value = ev;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5ES__event_free(H5ES_event_t *ev)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(ev);

    
    ev->op_info.api_name = NULL;
    if (ev->op_info.api_args)
        H5MM_xfree_const(ev->op_info.api_args);
    ev->op_info.app_file_name = NULL;
    ev->op_info.app_func_name = NULL;
    if (ev->request) {
        
        if (H5VL_request_free(ev->request) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTFREE, FAIL, "unable to free request");

        
        if (H5VL_free_object(ev->request) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTRELEASE, FAIL, "can't free VOL request object");
    } 

    H5FL_FREE(H5ES_event_t, ev);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5ES__event_completed(H5ES_event_t *ev, H5ES_event_list_t *el)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(ev);

    
    H5ES__list_remove(el, ev);

    
    if (H5ES__event_free(ev) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTFREE, FAIL, "unable to free event");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
