% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tab_create_modify.R
\name{tab_row_group}
\alias{tab_row_group}
\title{Add a row group to a \strong{gt} table}
\usage{
tab_row_group(data, label, rows, id = label, others_label = NULL, group = NULL)
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{label}{\emph{Row group label text}

\verb{scalar<character>} // \strong{required}

The text to use for the row group label. We can optionally use \code{\link[=md]{md()}} or
\code{\link[=html]{html()}} to style the text as Markdown or to retain HTML elements in the
text.}

\item{rows}{\emph{Rows to target}

\verb{<row-targeting expression>} // \strong{required}

The rows to be made components of the row group. We can supply a vector of
row ID values within \code{c()}, a vector of row indices, or use select helpers
(e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}). We can also use expressions to filter
down to the rows we need (e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{id}{\emph{Row group ID}

\verb{scalar<character>} // \emph{default:} \code{label}

The ID for the row group. When accessing a row group through
\code{\link[=cells_row_groups]{cells_row_groups()}} (when using \code{\link[=tab_style]{tab_style()}} or \code{\link[=tab_footnote]{tab_footnote()}}) the
\code{id} value is used as the reference (and not the \code{label}). If an \code{id} is
not explicitly provided here, it will be taken from the \code{label} value. It
is advisable to set an explicit \code{id} value if you plan to access this cell
in a later function call and the label text is complicated (e.g., contains
markup, is lengthy, or both). Finally, when providing an \code{id} value you
must ensure that it is unique across all ID values set for row groups (the
function will stop if \code{id} isn't unique).}

\item{others_label}{\emph{\link{Deprecated} Label for default row group}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

This argument is deprecated. Instead use
\verb{tab_options(row_group.default_label = <label>)}.}

\item{group}{\emph{\link{Deprecated} The group label}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

This argument is deprecated. Instead use \code{label}.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
We can create a row group from a collection of rows with \code{tab_row_group()}.
This requires specification of the rows to be included, either by supplying
row labels, row indices, or through use of a select helper function like
\code{\link[=starts_with]{starts_with()}}. To modify the order of row groups, we can use
\code{\link[=row_group_order]{row_group_order()}}.

To set a default row group label for any rows not formally placed in a row
group, we can use a separate call to \verb{tab_options(row_group.default_label = <label>)}. If this is not done and there are rows that haven't been placed
into a row group (where one or more row groups already exist), those rows
will be automatically placed into a row group without a label.
}
\section{Examples}{


Using a subset of the \code{\link{gtcars}} dataset, let's create a simple \strong{gt} table
with row labels (from the \code{model} column) inside of a stub. This eight-row
table begins with no row groups at all but with a single use of
\code{tab_row_group()}, we can specify a row group that will contain any rows
where the car model begins with a number.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{gtcars |>
  dplyr::select(model, year, hp, trq) |>
  dplyr::slice(1:8) |>
  gt(rowname_col = "model") |>
  tab_row_group(
    label = "numbered",
    rows = matches("^[0-9]")
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_tab_row_group_1.png" alt="This image of a table was generated from the first code example in the `tab_row_group()` help file." style="width:100\%;">
}}

This actually makes two row groups since there are row labels that don't
begin with a number. That second row group is a catch-all \code{NA} group, and it
doesn't display a label at all. Rather, it is set off from the other group
with a double line. This may be a preferable way to display the arrangement
of one distinct group and an 'others' or default group. If that's the case
but you'd like the order reversed, you can use \code{\link[=row_group_order]{row_group_order()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{gtcars |>
  dplyr::select(model, year, hp, trq) |>
  dplyr::slice(1:8) |>
  gt(rowname_col = "model") |>
  tab_row_group(
    label = "numbered",
    rows = matches("^[0-9]")
  ) |>
  row_group_order(groups = c(NA, "numbered"))
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_tab_row_group_2.png" alt="This image of a table was generated from the second code example in the `tab_row_group()` help file." style="width:100\%;">
}}

Two more options include: (1) setting a default label for the 'others' group
(done through \code{\link[=tab_options]{tab_options()}}), and (2) creating row groups until there are
no more unaccounted for rows. Let's try the first option in the next example:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{gtcars |>
  dplyr::select(model, year, hp, trq) |>
  dplyr::slice(1:8) |>
  gt(rowname_col = "model") |>
  tab_row_group(
    label = "numbered",
    rows = matches("^[0-9]")
  ) |>
  row_group_order(groups = c(NA, "numbered")) |>
  tab_options(row_group.default_label = "others")
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_tab_row_group_3.png" alt="This image of a table was generated from the third code example in the `tab_row_group()` help file." style="width:100\%;">
}}

The above use of the \code{row_group.default_label} in \code{\link[=tab_options]{tab_options()}} gets the
job done and provides a default label. One drawback is that the default/\code{NA}
group doesn't have an ID, so it can't as easily be styled with \code{\link[=tab_style]{tab_style()}};
however, row groups have indices and the index for the \code{"others"} group here
is \code{1}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{gtcars |>
  dplyr::select(model, year, hp, trq) |>
  dplyr::slice(1:8) |>
  gt(rowname_col = "model") |>
  tab_row_group(
    label = "numbered",
    rows = matches("^[0-9]")
  ) |>
  row_group_order(groups = c(NA, "numbered")) |>
  tab_options(row_group.default_label = "others") |>
  tab_style(
    style = cell_fill(color = "bisque"),
    locations = cells_row_groups(groups = 1)
  ) |>
  tab_style(
    style = cell_fill(color = "lightgreen"),
    locations = cells_row_groups(groups = "numbered")
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_tab_row_group_4.png" alt="This image of a table was generated from the fourth code example in the `tab_row_group()` help file." style="width:100\%;">
}}

Now let's try using \code{tab_row_group()} with our \code{\link{gtcars}}-based table such
that all rows are formally assigned to different row groups. We'll define two
row groups with the (Markdown-infused) labels \code{"**Powerful Cars**"} and
\code{"**Super Powerful Cars**"}. The distinction between the groups is whether
\code{hp} is lesser or greater than \code{600} (and this is governed by the expressions
provided to the \code{rows} argument).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{gtcars |>
  dplyr::select(model, year, hp, trq) |>
  dplyr::slice(1:8) |>
  gt(rowname_col = "model") |>
  tab_row_group(
    label = md("**Powerful Cars**"),
    rows = hp < 600,
    id = "powerful"
  ) |>
  tab_row_group(
    label = md("**Super Powerful Cars**"),
    rows = hp >= 600,
    id = "v_powerful"
  ) |>
  tab_style(
    style = cell_fill(color = "gray85"),
    locations = cells_row_groups(groups = "powerful")
  ) |>
  tab_style(
    style = list(
      cell_fill(color = "gray95"),
      cell_text(size = "larger")
    ),
    locations = cells_row_groups(groups = "v_powerful")
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_tab_row_group_5.png" alt="This image of a table was generated from the fifth code example in the `tab_row_group()` help file." style="width:100\%;">
}}

Setting the \code{id} values for each of the row groups makes things easier since
you will have clean, markup-free ID values to reference in later calls (as
was done with the \code{\link[=tab_style]{tab_style()}} invocations in the example above). The use of
the \code{\link[=md]{md()}} helper function makes it so that any Markdown provided for the
\code{label} of a row group is faithfully rendered.
}

\section{Function ID}{

2-4
}

\section{Function Introduced}{

\code{v0.2.0.5} (March 31, 2020)
}

\seealso{
Other part creation/modification functions: 
\code{\link{tab_caption}()},
\code{\link{tab_footnote}()},
\code{\link{tab_header}()},
\code{\link{tab_info}()},
\code{\link{tab_options}()},
\code{\link{tab_source_note}()},
\code{\link{tab_spanner}()},
\code{\link{tab_spanner_delim}()},
\code{\link{tab_stub_indent}()},
\code{\link{tab_stubhead}()},
\code{\link{tab_style}()},
\code{\link{tab_style_body}()}
}
\concept{part creation/modification functions}
