% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_data.R
\name{fmt_fraction}
\alias{fmt_fraction}
\title{Format values as mixed fractions}
\usage{
fmt_fraction(
  data,
  columns = everything(),
  rows = everything(),
  accuracy = NULL,
  simplify = TRUE,
  layout = c("inline", "diagonal"),
  use_seps = TRUE,
  pattern = "{x}",
  sep_mark = ",",
  system = c("intl", "ind"),
  locale = NULL
)
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{columns}{\emph{Columns to target}

\code{\link[=rows-columns]{<column-targeting expression>}} // \emph{default:} \code{everything()}

Can either be a series of column names provided in \code{c()}, a vector of
column indices, or a select helper function (e.g. \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=num_range]{num_range()}} and \code{\link[=everything]{everything()}}).}

\item{rows}{\emph{Rows to target}

\code{\link[=rows-columns]{<row-targeting expression>}} // \emph{default:} \code{everything()}

In conjunction with \code{columns}, we can specify which of their rows should
undergo formatting. The default \code{\link[=everything]{everything()}} results in all rows in
\code{columns} being formatted. Alternatively, we can supply a vector of row
captions within \code{c()}, a vector of row indices, or a select helper
function (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}). We can also use expressions to filter
down to the rows we need (e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{accuracy}{\emph{Accuracy of fractions}

\verb{singl-kw:[low|med|high]|scalar<numeric|integer>(val>=1)} // \emph{default:} \code{"low"}

The type of fractions to generate. This can either be one of the keywords
\code{"low"}, \code{"med"}, or \code{"high"} (to generate fractions with denominators of
up to 1, 2, or 3 digits, respectively) or an integer value greater than
zero to obtain fractions with a fixed denominator (\code{2} yields halves, \code{3}
is for thirds, \code{4} is quarters, etc.). For the latter option, using
\code{simplify = TRUE} will simplify fractions where possible (e.g., \code{2/4} will
be simplified as \code{1/2}). By default, the \code{"low"} option is used.}

\item{simplify}{\emph{Simplify the fraction}

\verb{scalar<logical>} // \emph{default:} \code{TRUE}

If choosing to provide a numeric value for \code{accuracy}, the option to
simplify the fraction (where possible) can be taken with \code{TRUE} (the
default). With \code{FALSE}, denominators in fractions will be fixed to the
value provided in \code{accuracy}.}

\item{layout}{\emph{Layout of fractions in HTML output}

\verb{singl-kw:[inline|diagonal]} // \emph{default:} \code{"inline"}

For HTML output, the \code{"inline"} layout is the default. This layout places
the numerals of the fraction on the baseline and uses a standard slash
character. The \code{"diagonal"} layout will generate fractions that are typeset
with raised/lowered numerals and a virgule.}

\item{use_seps}{\emph{Use digit group separators}

\verb{scalar<logical>} // \emph{default:} \code{TRUE}

An option to use digit group separators. The type of digit group separator
is set by \code{sep_mark} and overridden if a locale ID is provided to \code{locale}.
This setting is \code{TRUE} by default.}

\item{pattern}{\emph{Specification of the formatting pattern}

\verb{scalar<character>} // \emph{default:} \code{"{x}"}

A formatting pattern that allows for decoration of the formatted value. The
formatted value is represented by the \code{{x}} (which can be used multiple
times, if needed) and all other characters will be interpreted as string
literals.}

\item{sep_mark}{\emph{Separator mark for digit grouping}

\verb{scalar<character>} // \emph{default:} \code{","}

The string to use as a separator between groups of digits. For example,
using \code{sep_mark = ","} with a value of \code{1000} would result in a formatted
value of \code{"1,000"}. This argument is ignored if a \code{locale} is supplied
(i.e., is not \code{NULL}).}

\item{system}{\emph{Numbering system for grouping separators}

\verb{singl-kw:[intl|ind]} // \emph{default:} \code{"intl"}

The international numbering system (keyword: \code{"intl"}) is widely used and
its grouping separators (i.e., \code{sep_mark}) are always separated by three
digits. The alternative system, the Indian numbering system (keyword:
\code{"ind"}), uses grouping separators that correspond to thousand, lakh,
crore, and higher quantities.}

\item{locale}{\emph{Locale identifier}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

An optional locale identifier that can be used for formatting values
according to the locale's rules. Examples include \code{"en"} for English (United
States) and \code{"fr"} for French (France). We can call \code{\link[=info_locales]{info_locales()}} for a
useful reference for all of the locales that are supported. A locale ID can
be also set in the initial \code{\link[=gt]{gt()}} function call (where it would be used
automatically by any function with a \code{locale} argument) but a \code{locale}
value provided here will override that global locale.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
With numeric values in a \strong{gt} table, we can perform mixed-fraction-based
formatting. There are several options for setting the accuracy of the
fractions. Furthermore, there is an option for choosing a layout (i.e.,
typesetting style) for the mixed-fraction output.

The following options are available for controlling this type of formatting:
\itemize{
\item accuracy: how to express the fractional part of the mixed fractions; there
are three keyword options for this and an allowance for arbitrary denominator
settings
\item simplification: an option to simplify fractions whenever possible
\item layout: We can choose to output values with diagonal or inline fractions
\item digit grouping separators: options to enable/disable digit separators
and provide a choice of separator symbol for the whole number portion
\item pattern: option to use a text pattern for decoration of the formatted
mixed fractions
\item locale-based formatting: providing a locale ID will result in number
formatting specific to the chosen locale
}
}
\section{Compatibility of formatting function with data values}{


\code{fmt_fraction()} is compatible with body cells that are of the \code{"numeric"} or
\code{"integer"} types. Any other types of body cells are ignored during
formatting. This is to say that cells of incompatible data types may be
targeted, but there will be no attempt to format them.
}

\section{Compatibility of arguments with the \code{from_column()} helper function}{


\code{\link[=from_column]{from_column()}} can be used with certain arguments of \code{fmt_fraction()} to
obtain varying parameter values from a specified column within the table.
This means that each row could be formatted a little bit ifferently. These
arguments provide support for \code{\link[=from_column]{from_column()}}:
\itemize{
\item \code{accuracy}
\item \code{simplify}
\item \code{layout}
\item \code{use_seps}
\item \code{pattern}
\item \code{sep_mark}
\item \code{system}
\item \code{locale}
}

Please note that for all of the aforementioned arguments, a \code{\link[=from_column]{from_column()}}
call needs to reference a column that has data of the correct type (this is
different for each argument). Additional columns for parameter values can be
generated with \code{\link[=cols_add]{cols_add()}} (if not already present). Columns that contain
parameter data can also be hidden from final display with \code{\link[=cols_hide]{cols_hide()}}.
Finally, there is no limitation to how many arguments the \code{\link[=from_column]{from_column()}}
helper is applied so long as the arguments belong to this closed set.
}

\section{Adapting output to a specific \code{locale}}{


This formatting function can adapt outputs according to a provided \code{locale}
value. Examples include \code{"en"} for English (United States) and \code{"fr"} for
French (France). The use of a valid locale ID here means separator and
decimal marks will be correct for the given locale. Should any value be
provided in \code{sep_mark}, it will be overridden by the locale's preferred
values.

Note that a \code{locale} value provided here will override any global locale
setting performed in \code{\link[=gt]{gt()}}'s own \code{locale} argument (it is settable there as
a value received by all other functions that have a \code{locale} argument). As a
useful reference on which locales are supported, we can call \code{\link[=info_locales]{info_locales()}}
to view an info table.
}

\section{Examples}{


Using a summarized version of the \code{\link{pizzaplace}} dataset, let's create a
\strong{gt} table. With \code{fmt_fraction()} we can format the \code{f_sold} and \code{f_income}
columns to display fractions. As for how the fractions are represented, we
are electing to use \code{accuracy = 10}. This gives all fractions as tenths.
We won't simplify the fractions (by using \code{simplify = FALSE}) and this means
that a fraction like \code{5/10} won't become \code{1/2}. With \code{layout ="diagonal"},
we get a diagonal display of all fractions.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pizzaplace |>
  dplyr::group_by(type, size) |>
  dplyr::summarize(
    sold = dplyr::n(),
    income = sum(price),
    .groups = "drop_last"
  ) |>
  dplyr::group_by(type) |>
  dplyr::mutate(
    f_sold = sold / sum(sold),
    f_income = income / sum(income),
  ) |>
  dplyr::arrange(type, dplyr::desc(income)) |>
  gt(rowname_col = "size") |>
  tab_header(
    title = "Pizzas Sold in 2015",
    subtitle = "Fraction of Sell Count and Revenue by Size per Type"
  ) |>
  fmt_integer(columns = sold) |>
  fmt_currency(columns = income) |>
  fmt_fraction(
    columns = starts_with("f_"),
    accuracy = 10,
    simplify = FALSE,
    layout = "diagonal"
  ) |>
  sub_missing(missing_text = "") |>
  tab_spanner(
    label = "Sold",
    columns = contains("sold")
  ) |>
  tab_spanner(
    label = "Revenue",
    columns = contains("income")
  ) |>
  text_transform(
    locations = cells_body(),
    fn = function(x) \{
      dplyr::case_when(
        x == 0 ~ "<em>nil</em>",
        x != 0 ~ x
      )
    \}
  ) |>
  cols_label(
    sold = "Amount",
    income = "Amount",
    f_sold = md("_f_"),
    f_income = md("_f_")
  ) |>
  cols_align(align = "center", columns = starts_with("f")) |>
  tab_options(
    table.width = px(400),
    row_group.as_column = TRUE
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_fraction_1.png" alt="This image of a table was generated from the first code example in the `fmt_fraction()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-8
}

\section{Function Introduced}{

\code{v0.4.0} (February 15, 2022)
}

\seealso{
The vector-formatting version of this function:
\code{\link[=vec_fmt_fraction]{vec_fmt_fraction()}}.

Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bins}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_chem}()},
\code{\link{fmt_country}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_date}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_email}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_icon}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_number_si}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_spelled_num}()},
\code{\link{fmt_tf}()},
\code{\link{fmt_time}()},
\code{\link{fmt_units}()},
\code{\link{fmt_url}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
