% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gdal_mdim.R
\name{mdim_as_classic}
\alias{mdim_as_classic}
\title{Return a view of an MDArray as a "classic" GDALDataset (i.e., 2D)}
\usage{
mdim_as_classic(
  dsn,
  array_name,
  idx_xdim,
  idx_ydim,
  read_only = TRUE,
  group_name = NULL,
  view_expr = NULL,
  allowed_drivers = NULL,
  open_options = NULL
)
}
\arguments{
\item{dsn}{Character string giving the data source name of the
multidimensional raster (e.g., file, VSI path).}

\item{array_name}{Character string giving the name of the MDarray in
\code{dsn}.}

\item{idx_xdim}{Integer value giving the index of the dimension that will be
used as the X/width axis (0-based).}

\item{idx_ydim}{Integer value giving the index of the dimension that will be
used as the Y/height axis (0-based).}

\item{read_only}{Logical value, \code{TRUE} to open the dataset in read-only mode
(the default).}

\item{group_name}{Optional character string giving the fully qualified name
of a group containing \code{array_name}.}

\item{view_expr}{Optional character string giving an expression for basic
array slicing and indexing, or field access (see section \verb{View Expressions}).}

\item{allowed_drivers}{Optional character vector of driver short names that
must be considered. By default, all known multidimensional raster drivers are
considered.}

\item{open_options}{Optional character vector of format-specific dataset open
options as \code{"NAME=VALUE"} pairs.}
}
\value{
An object of class \code{GDALRaster}.
}
\description{
\code{mdim_as_classic()} returns a 2D raster view on an MDArray in a GDAL
Multidimensional Raster dataset, as an object of class \code{GDALRaster}. Only 2D
or more arrays are supported. In the case of > 2D arrays, additional
dimensions will be represented as raster bands. Requires GDAL >= 3.2.
}
\note{
The indexing of array dimensions is 0-based consistent with the
\verb{<ARRAY-SPEC>} notation that may be used with GDAL CLI commands, e.g.,
\code{gdal_usage("mdim convert")} (CLI bindings require GDAL > 3.11.3).
See \url{https://gdal.org/en/stable/programs/gdal_mdim_convert.html}.

Once the returned \code{GDALRaster} object has been closed, it cannot be re-opened
with its \verb{$open()} method.
}
\section{View Expressions}{

A character string can be passed in argument \code{view_expr} to specify array
slicing or field access. The slice expression uses the same syntax as NumPy
basic slicing and indexing (0-based), or it can use field access by name.
See \url{https://numpy.org/doc/stable/user/basics.indexing.html}.

GDAL support for view expression on an MDArray is documented for
\code{GDALMDArray::GetView()} (see
\url{https://gdal.org/en/stable/api/gdalmdarray_cpp.html}) and copied here:

Multiple [] bracket elements can be concatenated, with a slice expression or
field name inside each.

For basic slicing and indexing, inside each [] bracket element, a list of
indexes that apply to successive source dimensions, can be specified, using
integer indexing (e.g. 1), range indexing (start:stop:step), ellipsis (...)
or newaxis, using a comma separator.

Example expressions with a 2-dimensional array whose content is
\verb{[[0,1,2,3],[4,5,6,7]]}.
\itemize{
\item \code{"[1][2]"}: returns a 0-dimensional/scalar array with the value at
index 1 in the first dimension, and index 2 in the second dimension from the
source array. That is, \code{5}.
\item \code{"[1,2]"}: same as above, but a bit more performant.
\item \code{"[1]"}: returns a 1-dimensional array, sliced at index \code{1} in the
first dimension. That is \verb{[4,5,6,7]}.
\item \code{"[:,2]"}: returns a 1-dimensional array, sliced at index \code{2} in the
second dimension. That is \verb{[2,6]}.
\item \code{"[:,2:3:]"}: returns a 2-dimensional array, sliced at index \code{2} in
the second dimension. That is \verb{[[2],[6]]}.
\item \code{"[::,2]"}: Same as above.
\item \code{"[...,2]"}: same as above, in that case, since the ellipsis only
expands to one dimension here.
\item \code{"[:,::2]"}: returns a 2-dimensional array, with even-indexed
elements of the second dimension. That is \verb{[[0,2],[4,6]]}.
\item \code{"[:,1::2]"}: returns a 2-dimensional array, with odd-indexed
elements of the second dimension. That is \verb{[[1,3],[5,7]]}.
\item \code{"[:,1:3:]"}: returns a 2-dimensional array, with elements of the
second dimension with index in the range \verb{[1,3]}. That is \verb{[[1,2],[5,6]]}.
\item \code{"[::-1,:]"}: returns a 2-dimensional array, with the values in
first dimension reversed. That is \verb{[[4,5,6,7],[0,1,2,3]]}.
\item \code{"[newaxis,...]"}: returns a 3-dimensional array, with an additional
dimension of size \code{1} put at the beginning. That is
\verb{[[[0,1,2,3],[4,5,6,7]]]}.
}

One difference with NumPy behavior is that ranges that would result in zero
elements are not allowed (dimensions of size 0 not being allowed in the GDAL
multidimensional model).

For field access, the syntax to use is \code{"['field_name']"}. Multiple field
specification is not supported currently. Both type of access can be
combined, e.g. \code{"[1]['field_name']"}.
}

\examples{
\dontshow{if (gdal_version_num() >= gdal_compute_version(3, 2, 0) && isTRUE(gdal_formats("netCDF")$multidim_raster)) withAutoprint(\{ # examplesIf}
f <- system.file("extdata/byte.nc", package="gdalraster")
# mdim_info(f)

(ds <- mdim_as_classic(f, "Band1", 1, 0))

plot_raster(ds, interpolate = FALSE, legend = TRUE, main = "Band1")

ds$close()
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=GDALRaster]{GDALRaster-class}}, \code{\link[=mdim_info]{mdim_info()}}, \code{\link[=mdim_translate]{mdim_translate()}}
}
