% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transformations.R
\name{make.tran}
\alias{make.tran}
\alias{inverse}
\title{Response-transformation extensions}
\usage{
make.tran(type = c("genlog", "power", "boxcox", "sympower", "asin.sqrt",
  "atanh", "bcnPower", "scale"), alpha = 1, beta = 0, param, y, inner, ...)

inverse(y)
}
\arguments{
\item{type}{The name of a standard transformation supported by \code{stat::make.link},
or of a special transformation described under Details.}

\item{alpha, beta}{Numeric parameters needed for special transformations.}

\item{param}{If non-missing, this specifies either
\code{alpha} or \code{c(alpha, beta)} (provided for backward compatibility).
Also, for the same reason, if \code{alpha} is of length more than 1,
it is taken as \code{param}.}

\item{y}{A numeric response variable used (\emph{and required}) with \code{type = "scale"}, 
where \code{scale(y)} determines \code{alpha} and \code{beta}.}

\item{inner}{another transformation. See the section on compound transformations}

\item{...}{Additional arguments passed to other functions/methods}
}
\value{
A \code{list} having at least the same elements as those returned by
  \code{\link{make.link}}. The \code{linkfun} component is the transformation
  itself. Each of the functions is associated with an environment where any 
  parameter values are defined.

\code{inverse} returns the reciprocal of its argument. It allows
  the \code{"inverse"} link to be auto-detected as a response transformation.
}
\description{
The \code{make.tran} function creates the needed information to perform
transformations of the response
variable, including inverting the transformation and estimating variances of
back-transformed predictions via the delta method. \code{make.tran} is
similar to \code{\link{make.link}}, but it covers additional transformations.
The result can be used as an environment in which the model is fitted, or as
the \code{tran} argument in \code{\link{update.emmGrid}} (when the given
transformation was already applied in an existing model).
}
\note{
The \code{genlog} transformation is technically unneeded, because
  a response transformation of the form \code{log(y + c)} is now auto-detected 
  by \code{\link{ref_grid}}.

We modify certain \code{\link{make.link}} results in transformations
  where there is a restriction on valid prediction values, so that reasonable
  inverse predictions are obtained, no matter what. For example, if a
  \code{sqrt} transformation was used but a predicted value is negative, the
  inverse transformation is zero rather than the square of the prediction. A
  side effect of this is that it is possible for one or both confidence
  limits, or even a standard error, to be zero.
}
\section{Details}{

The \code{make.tran} function returns a
suitable list of functions for several popular transformations. Besides being
usable with \code{update}, the user may use this list as an enclosing
environment in fitting the model itself, in which case the transformation is
auto-detected when the special name \code{linkfun} (the transformation
itself) is used as the response transformation in the call. See the examples
below.

The primary purpose of \code{make.tran} is to support transformations that
require additional parameters, specified as \code{alpha} and \code{beta};
these are the onse shown in the argument-matching list. However, standard
transformations supported by \code{stats::make.link} are also supported.
In the following discussion of ones requiring parameters, 
we use \eqn{\alpha} and \eqn{\beta} to
denote \code{alpha} and \code{beta}, and \eqn{y} to denote the response variable.
The \code{type} argument specifies the following transformations:
\describe{
\item{\code{"genlog"}}{Generalized logarithmic transformation: \eqn{\log_\beta(y +
  \alpha)}, where \eqn{y > -\alpha}.
  When \eqn{\beta = 0} (the default), we use \eqn{\log_e(y + \alpha)}}
\item{\code{"power"}}{Power transformation: \eqn{(y-\beta)^\alpha}, where \eqn{y > \beta}.
  When \eqn{\alpha = 0}, \eqn{\log(y-\beta)} is used instead.}
\item{\code{"boxcox"}}{The Box-Cox transformation (unscaled by the geometric
  mean): \eqn{((y - \beta)^\alpha - 1) / \alpha}, where \eqn{y > \beta}. 
  When \eqn{\alpha = 0}, \eqn{\log(y - \beta)}
  is used.}
\item{\code{"sympower"}}{A symmetrized power transformation on the whole real
  line:
  \eqn{|y - \beta|^\alpha\cdot sign(y - \beta)}. There are no restrictions on \eqn{y}, but we
  require \eqn{\alpha > 0} in order for the transformation to be monotone and
  continuous.}
\item{\code{"asin.sqrt"}}{Arcsin-square-root transformation:
  \eqn{\sin^{-1}(y/\alpha)^{1/2}}. Typically, \code{alpha} will be either 1 (default) or 100.}
\item{\code{"atanh"}}{Arctanh transformation:
  \eqn{\tanh^{-1}(y/\alpha)}. Typically, \code{alpha} will be either 1 (default) or 100.}
\item{\code{"bcnPower"}}{Box-Cox with negatives allowed, as described for the 
  \code{bcnPower} function in the \pkg{car} package. It is defined as the Box-Cox
  transformation \eqn{(z^\alpha - 1) / \alpha} of the variable \eqn{z = y + (y^2+\beta^2)^{1/2}}. 
  Note that this requires both parameters and that \code{beta > 0}.}
\item{\code{"scale"}}{This one is a little different than the others, in that
  \code{alpha} and \code{beta} are ignored; instead, they are determined by calling 
  \code{scale(y, ...)}. The user should give as \code{y} the response variable in the
  model to be fitted to its scaled version.}

}

Note that with the \code{"power"}, \code{"boxcox"}, or \code{"sympower"} transformations, 
the argument \code{beta} specifies a location shift. 
In the \code{"genpower"} transformation, \code{beta} specifies
the base of the logarithm -- however, quirkily, the default of \code{beta = 0}
is taken to be the natural logarithm. For example,
\code{make.tran(0.5, 10)} sets up the \eqn{\log_{10}(y + \frac12)}
transformation. In the \code{"bcnPower"} transformation, \code{beta}
must be specified as a positive value.

For purposes of back-transformation, the \samp{sqrt(y) + sqrt(y+1)}
transformation is treated exactly the same way as \samp{2*sqrt(y)}, because
both are regarded as estimates of \eqn{2\sqrt\mu}.
}

\section{Cases where \code{make.tran} may not be needed}{

For standard transformations with no parameters, we usually don't need to use
\code{make.tran}; just the name of the transformation is all that is needed.
The functions \code{\link{emmeans}}, \code{\link{ref_grid}}, and related ones
automatically detect response transformations that are recognized by
examining the model formula. These are \code{log}, \code{log2}, \code{log10},
\code{log1p},
\code{sqrt}, \code{logit}, \code{probit}, \code{cauchit}, \code{cloglog}; as
well as (for a response variable \code{y}) \code{asin(sqrt(y))},
\code{asinh(sqrt(y))}, \code{atanh(y)}, and \code{sqrt(y) + sqrt(y+1)}. 
In addition, any
constant multiple of these (e.g., \code{2*sqrt(y)}) is auto-detected and
appropriately scaled (see also the \code{tran.mult} argument in
\code{\link{update.emmGrid}}).

A few additional transformations may be specified as character strings and
are auto-detected: \code{"identity"}, \code{"1/mu^2"},
\code{"inverse"}, \code{"reciprocal"}, \code{"log10"}, \code{"log2"},
\code{"asin.sqrt"}, \code{"asinh.sqrt"}, and \code{"atanh"}.
}

\section{Compound transformations}{

A transformation that is a function of another function can be created by
specifying \code{inner} for the other function. For example, the
transformation \eqn{1/\sqrt{y}} can be created either by
\code{make.tran("inverse", inner = "sqrt")} or by \code{make.tran("power",
-0.5)}. In principle, transformations can be compounded to any depth.
Also, if \code{type} is \code{"scale"}, \code{y} is replaced by 
\code{inner$linkfun(y)}, because that will be the variable that is scaled.
}

\examples{
# Fit a model using an oddball transformation:
bctran <- make.tran("boxcox", 0.368)
warp.bc <- with(bctran, 
    lm(linkfun(breaks) ~ wool * tension, data = warpbreaks))
# Obtain back-transformed LS means:    
emmeans(warp.bc, ~ tension | wool, type = "response")

### Using a scaled response...
# Case where it is auto-detected:
mod <- lm(scale(yield[, 1]) ~ Variety, data = MOats)
emmeans(mod, "Variety", type = "response")

# Case where scaling is not auto-detected -- and what to do about it:
copt <- options(contrasts = c("contr.sum", "contr.poly"))
mod.aov <- aov(scale(yield[, 1]) ~ Variety + Error(Block), data = MOats)
emm.aov <- suppressWarnings(emmeans(mod.aov, "Variety", type = "response"))

# Scaling was not retrieved, but we can do:
emm.aov <- update(emm.aov, tran = make.tran("scale", y = MOats$yield[, 1]))
emmeans(emm.aov, "Variety", type = "response")

### Compound transformations
# The following amount to the same thing:
t1 <- make.tran("inverse", inner = "sqrt")
t2 <- make.tran("power", -0.5)

options(copt)


\dontrun{
### An existing model 'mod' was fitted with a y^(2/3) transformation...
  ptran = make.tran("power", 2/3)
  emmeans(mod, "treatment", tran = ptran)
}

pigs.lm <- lm(inverse(conc) ~ source + factor(percent), data = pigs)
emmeans(pigs.lm, "source", type = "response")
}
