#' @title Decomposition of the equitability index
#'
#' @description Computes equitability and its decomposition
#'
#' @param x Vector of dimension S (spicies) with the numbers of observed individuals in each spicy. NA values are allowed. 0 values are converted to NA.
#' @param groups Vector of dimension S of factors indicating the groups.
#'
#' @return
#' \itemize{
#'   \item \code{equitability}: Equitability index.
#'   \item \code{within}: Within groups equitability.
#'   \item \code{between}: Between groups equitability.
#' }
#'
#' @seealso \link{dec_shannon}
#'
#' @references
#' "Quantifying Diversity through Entropy Decomposition: Insights into Hominin Occupation and Carcass Processing at Qesem cave"
#'
#' @examples
#' data(Qesem_s)
#' dec_equit(Qesem_s$HU, Qesem_s$Group)
#'
#' @export

# function to compute equitability and its decomposition
dec_equit <- function(x, groups){
  d <- dec_shannon(x, groups)
  g <- d$groups
  S <- sum(g$S)
  return(list(equitability = equitability(x),
              within = d$within/log(S),
              between = d$between/log(S)))
}
