% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/corrSelect.R
\name{corrSelect}
\alias{corrSelect}
\title{Select Variable Subsets with Low Correlation (Data Frame Interface)}
\usage{
corrSelect(
  df,
  threshold = 0.7,
  method = NULL,
  force_in = NULL,
  cor_method = c("pearson", "spearman", "kendall", "bicor", "distance", "maximal"),
  ...
)
}
\arguments{
\item{df}{A data frame. Only numeric columns are used.}

\item{threshold}{A numeric value in (0, 1). Maximum allowed absolute correlation. Defaults to 0.7.}

\item{method}{Character. Selection algorithm to use. One of \code{"els"} or
\code{"bron-kerbosch"}. If not specified, the function chooses automatically:
\code{"els"} when \code{force_in} is provided, otherwise \code{"bron-kerbosch"}.}

\item{force_in}{Optional character vector or numeric indices of columns to force into all subsets.}

\item{cor_method}{Character string indicating which correlation method to use.
One of \code{"pearson"} (default), \code{"spearman"}, \code{"kendall"},
\code{"bicor"}, \code{"distance"}, or \code{"maximal"}.}

\item{...}{Additional arguments passed to \code{\link{MatSelect}()}, e.g., \code{use_pivot}.}
}
\value{
An object of class \code{\link{CorrCombo}}, containing selected subsets and correlation statistics.
}
\description{
Identifies combinations of numeric variables in a data frame such that all pairwise absolute correlations
fall below a specified threshold. This function is a wrapper around \code{\link{MatSelect}()}
and accepts data frames, tibbles, or data tables with automatic preprocessing.
}
\details{
Only numeric columns are used for correlation analysis. Non‐numeric columns (factors, characters,
logicals, etc.) are ignored, and their names and types are printed to inform the user. These can be
optionally reattached later using \code{\link{corrSubset}()} with \code{keepExtra = TRUE}.

Rows with missing values are removed before computing correlations. A warning is issued if any rows are dropped.

The \code{cor_method} controls how the correlation matrix is computed:
\itemize{
\item \code{"pearson"}: Standard linear correlation.
\item \code{"spearman"}: Rank-based monotonic correlation.
\item \code{"kendall"}: Kendall's tau.
\item \code{"bicor"}: Biweight midcorrelation (WGCNA::bicor).
\item \code{"distance"}: Distance correlation (energy::dcor).
\item \code{"maximal"}: Maximal information coefficient (minerva::mine).
}

For \code{"bicor"}, \code{"distance"}, and \code{"maximal"}, the corresponding
package must be installed.
}
\examples{
set.seed(42)
n <- 100

# Create 20 variables: 5 blocks of correlated variables + some noise
block1 <- matrix(rnorm(n * 4), ncol = 4)
block2 <- matrix(rnorm(n), ncol = 1)
block2 <- matrix(rep(block2, 4), ncol = 4) + matrix(rnorm(n * 4, sd = 0.1), ncol = 4)
block3 <- matrix(rnorm(n * 4), ncol = 4)
block4 <- matrix(rnorm(n * 4), ncol = 4)
block5 <- matrix(rnorm(n * 4), ncol = 4)

df <- as.data.frame(cbind(block1, block2, block3, block4, block5))
colnames(df) <- paste0("V", 1:20)

# Add a non-numeric column to be ignored
df$label <- factor(sample(c("A", "B"), n, replace = TRUE))

# Basic usage
corrSelect(df, threshold = 0.8)

# Try Bron–Kerbosch with pivoting
corrSelect(df, threshold = 0.6, method = "bron-kerbosch", use_pivot = TRUE)

# Force in a specific variable and use Spearman correlation
corrSelect(df, threshold = 0.6, force_in = "V10", cor_method = "spearman")

}
\seealso{
\code{\link[=assocSelect]{assocSelect()}}, \code{\link[=MatSelect]{MatSelect()}}, \code{\link[=corrSubset]{corrSubset()}}
}
