% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semTools.R
\name{SEMdci}
\alias{SEMdci}
\title{SEM-based differential network analysis}
\usage{
SEMdci(graph, data, group, type = "ace", method = "BH", alpha = 0.05, ...)
}
\arguments{
\item{graph}{Input network as an igraph object.}

\item{data}{A matrix or data.frame. Rows correspond to subjects, and
columns to graph nodes (variables).}

\item{group}{A binary vector. This vector must be as long as the number
of subjects. Each vector element must be 1 for cases and 0 for control
subjects.}

\item{type}{Average Causal Effect (ACE) with two-group, "parents"
(back-door) adjustement set, and "direct" effects (\code{type = "ace"},
default), or CGGM solver with two-group using a clustering method.
If \code{type = "tahc"}, network modules are generated using the tree
agglomerative hierarchical clustering method, or non-tree clustering
methods from igraph package, i.e., \code{type = "wtc"} (walktrap community
structure with short random walks), \code{type ="ebc"} (edge betweeness
clustering), \code{type = "fgc"} (fast greedy method), \code{type = "lbc"}
(label propagation method), \code{type = "lec"} (leading eigenvector method),
\code{type = "loc"} (multi-level optimization), \code{type = "opc"} (optimal
community structure), \code{type = "sgc"} (spinglass statistical mechanics),
\code{type = "none"} (no breaking network structure into clusters).}

\item{method}{Multiple testing correction method. One of the values
available in \code{\link[stats]{p.adjust}}. By default, method is set
to "BH" (i.e., FDR multiple test correction).}

\item{alpha}{Significance level (default = 0.05) for edge set selection.}

\item{...}{Currently ignored.}
}
\value{
An igraph object.
}
\description{
Differential Connected Inference (DCI) via a sub-network with
perturbed edges obtained from the output of \code{\link[SEMgraph]{SEMace}},
comparable to the procedure in Jablonski et al (2022), or \code{\link[SEMgraph]{SEMrun}}
with two-group and CGGM solver, comparable to the algorithm 2 in Belyaeva et al (2021). 
To increase the efficiency of computations for large graphs, users can
select to break the network structure into clusters, and select the
topological clustering method (see \code{\link[SEMgraph]{clusterGraph}}).
The function \code{\link[SEMgraph]{SEMrun}} is applied iteratively on
each cluster (with size min > 10 and max < 500) to obtain the graph
with the full list of perturbed edges.
}
\examples{

\dontrun{

#load SEMdata package for ALS data with 17K genes:
#devtools::install_github("fernandoPalluzzi/SEMdata")
#library(SEMdata)

# Nonparanormal(npn) transformation
data.npn<- transformData(alsData$exprs)$data
dim(data.npn)

# Selection of FTD-ALS pathways from KEGG pathways

paths.name <- c("MAPK signaling pathway",
                "Protein processing in endoplasmic reticulum",
                "Endocytosis",
                "Wnt signaling pathway",
                "Neurotrophin signaling pathway",
                "Amyotrophic lateral sclerosis")

j <- which(names(kegg.pathways) \%in\% paths.name)

# Neuro interactome (max component)
gU <- Reduce(igraph::union, kegg.pathways[j])
gU <- properties(gU)[[1]]
#summary(gU)

# Modules with ALS perturbed edges using fast gready clustering
gD<- SEMdci(gU, data.npn, alsData$group, type="fgc")
summary(gD)
gcD<- properties(gD)

par(mfrow=c(2,2), mar=rep(2,4))
gplot(gcD[[1]], l="fdp", main="max component")
gplot(gcD[[2]], l="fdp", main="2nd component")
gplot(gcD[[3]], l="fdp", main="3rd component")
gplot(gcD[[4]], l="fdp", main="4th component")

}

}
\references{
Belyaeva A, Squires C, Uhler C (2021). DCI: learning causal differences
between gene regulatory networks. Bioinformatics, 37(18): 3067–3069.
<https://doi: 10.1093/bioinformatics/btab167>

Jablonski K, Pirkl M, Ćevid D, Bühlmann P, Beerenwinkel N (2022).
Identifying cancer pathway dysregulations using differential
causal effects. Bioinformatics, 38(6):1550–1559.
<https://doi.org/10.1093/bioinformatics/btab847>
}
\author{
Mario Grassi \email{mario.grassi@unipv.it}
}
