#' Check if the root scan directories contain everything necessary
#'
#' \code{checkInput} - This function checks a set of root scan
#' directories if they each contain two pictures, X.segmentation.png and
#' X.skeleton.png, where X is the name of the directory.
#'
#' @param data_dir (Optional, default = NULL) String specifying the name
#' (full path) of the directory  (full path) containing all root scan
#' directories of interest.
#' @param data_dirs (Optional, default = NULL) Character vector specifying all
#' of the individual root scan directories of interest (full paths). This is
#' only used if \code{data_dir} is set to NULL.
#'
#' @return \code{checkInput} A list containing two TRUE-FALSE-vectors showing
#' if the root scan directories each contain the two root pictures.
#'
#' @export
#' @rdname checkInput
#'
#' @examples
#' # Replace NULL with path to directory.
#' DATA_DIR <- NULL
#' # Apply the function directly...
#' cI1 <- checkInput(data_dir = DATA_DIR)
#' # ... or use it in combination with \code{getOverviewInput}:
#' cI2 <- checkInput(data_dirs = getOverviewInput(data_dir = DATA_DIR,
#'                                      naming_conv = "standard")$dir_name_full)
# Check input: -----------------------------------------------------------------
checkInput <- function(data_dir = NULL, data_dirs = NULL){
  if(!is.null(data_dir)){
    # List all files in the directory.
    root_image_dirs <- list.dirs(data_dir, full.names = TRUE, recursive = FALSE)
  } else if(!is.null(data_dirs)){
    root_image_dirs <- data_dirs
  } else {
    message("No directories specified.\n")
    return(NULL)
  }
  standardizePaths(root_image_dirs)

  # Check if each contains two scans (standard and skeletonized).
  root_pic_exist <- sapply(root_image_dirs,
                            FUN = function(X){
                              file.exists(paste0(X,"/",basename(X),
                                                 ".segmentation.png"))})
  root_skel_exist <- sapply(root_image_dirs,
                            FUN = function(X){
                              file.exists(paste0(X,"/",basename(X),
                                                 ".skeleton.png"))})
  # Message the results
  if(sum(!root_pic_exist)>0 || sum(!root_skel_exist)>0){
    if(sum(!root_pic_exist)>0){
      message(paste0("Root image(s) missing in the following ",
                     "directories:\n",
                     paste0(root_image_dirs[!root_pic_exist],"/",
                            collapse = "\n")))
    }
    if(sum(!root_skel_exist)>0){
      message(paste0("Root skeleton image(s) missing in the following ",
                     "directories:\n",
                     paste0(root_image_dirs[!root_skel_exist],"/",
                            collapse = "\n")))
    }
  } else {
    message("Great. All directories seem to contain everything necessary.")
  }
  return(list(root_pic_exist, root_skel_exist))
}
