% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/htlr.R
\name{htlr}
\alias{htlr}
\title{Fit a HTLR Model}
\usage{
htlr(
  X,
  y,
  fsel = 1:ncol(X),
  stdx = TRUE,
  prior = "t",
  df = 1,
  iter = 2000,
  warmup = floor(iter/2),
  thin = 1,
  init = "lasso",
  leap = 50,
  leap.warm = floor(leap/10),
  leap.stepsize = 0.3,
  cut = 0.05,
  verbose = FALSE,
  rep.legacy = FALSE,
  keep.warmup.hist = FALSE,
  ...
)
}
\arguments{
\item{X}{Input matrix, of dimension nobs by nvars; each row is an observation vector.}

\item{y}{Vector of response variables. Must be coded as non-negative integers, 
e.g., 1,2,\ldots,C for C classes, label 0 is also allowed.}

\item{fsel}{Subsets of features selected before fitting, such as by univariate screening.}

\item{stdx}{Logical; if \code{TRUE}, the original feature values are standardized to have \code{mean = 0} 
and \code{sd = 1}.}

\item{prior}{The prior to be applied to the model. Either a list of hyperparameter settings 
returned by \code{\link{htlr_prior}} or a character string from "t" (student-t), "ghs" (horseshoe), 
and "neg" (normal-exponential-gamma).}

\item{df}{The degree freedom of t/ghs/neg prior for coefficients. Will be ignored if the 
configuration list from \code{\link{htlr_prior}} is passed to \code{prior}.}

\item{iter}{A positive integer specifying the number of iterations (including warmup).}

\item{warmup}{A positive integer specifying the number of warmup (aka burnin). 
The number of warmup iterations should not be larger than iter and the default is \code{iter / 2}.}

\item{thin}{A positive integer specifying the period for saving samples.}

\item{init}{The initial state of Markov Chain; it accepts three forms:
\itemize{ 
\item a previously fitted \code{fithtlr} object, 
\item  a user supplied initial coeficient matrix of (p+1)*K, where p is the number of features, K is the number of classes in y minus 1, 
\item a character string matches the following:  
\itemize{
  \item "lasso" - (Default) Use Lasso initial state with \code{lambda} chosen by 
  cross-validation. Users may specify their own candidate \code{lambda} values via 
  optional argument \code{lasso.lambda}. Further customized Lasso initial 
  states can be generated by \code{\link{lasso_deltas}}.    
  \item "bcbc" - Use initial state generated by package \code{BCBCSF} 
  (Bias-corrected Bayesian classification). Further customized BCBCSF initial 
  states can be generated by \code{\link{bcbcsf_deltas}}. WARNING: This type of 
  initial states can be used for continuous features such as gene expression profiles, 
  but it should not be used for categorical features such as SNP profiles.
  \item "random" - Use random initial values sampled from N(0, 1).     
}

}}

\item{leap}{The length of leapfrog trajectory in sampling phase.}

\item{leap.warm}{The length of leapfrog trajectory in burnin phase.}

\item{leap.stepsize}{The integrator step size used in the Hamiltonian simulation.}

\item{cut}{The coefficients smaller than this criteria will be fixed in each HMC updating step.}

\item{verbose}{Logical; setting it to \code{TRUE} for tracking MCMC sampling iterations.}

\item{rep.legacy}{Logical; if \code{TRUE}, the output produced in \code{HTLR} versions up to 
legacy-3.1-1 is reproduced. The speed will be typically slower than non-legacy mode on
multi-core machine. Default is \code{FALSE}.}

\item{keep.warmup.hist}{Warmup iterations are not recorded by default, set \code{TRUE} to enable it.}

\item{...}{Other optional parameters:
\itemize{
  \item rda.alpha - A user supplied alpha value for \code{\link{bcbcsf_deltas}}. Default: 0.2.
  \item lasso.lambda - A user supplied lambda sequence for \code{\link{lasso_deltas}}. 
  Default: \{.01, .02, \ldots, .05\}. Will be ignored if \code{rep.legacy} is set to \code{TRUE}.
}}
}
\value{
An object with S3 class \code{htlr.fit}.
}
\description{
This function trains linear logistic regression models with HMC in restricted Gibbs sampling.
}
\examples{
set.seed(12345)
data("colon")

## fit HTLR models with selected features, note that the chain length setting is for demo only

## using t prior with 1 df and log-scale fixed to -10 
fit.t <- htlr(X = colon$X, y = colon$y, fsel = 1:100,
              prior = htlr_prior("t", df = 1, logw = -10), 
              init = "bcbc", iter = 20, thin = 1)

## using NEG prior with 1 df and log-scale fixed to -10 
fit.neg <- htlr(X = colon$X, y = colon$y, fsel = 1:100,
                prior = htlr_prior("neg", df = 1, logw = -10), 
                init = "bcbc", iter = 20, thin = 1)

## using horseshoe prior with 1 df and auto-selected log-scale   
fit.ghs <- htlr(X = colon$X, y = colon$y, fsel = 1:100,
                prior = "ghs", df = 1, init = "bcbc",
                iter = 20, thin = 1)

}
\references{
Longhai Li and Weixin Yao (2018). Fully Bayesian Logistic Regression 
with Hyper-Lasso Priors for High-dimensional Feature Selection.
\emph{Journal of Statistical Computation and Simulation} 2018, 88:14, 2827-2851.
}
