% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jointSpectralRadius.R
\name{bound_jsr_G}
\alias{bound_jsr_G}
\title{Calculate upper bound for the joint spectral radius of a set of matrices}
\usage{
bound_jsr_G(
  S,
  epsilon = 0.01,
  adaptive_eps = FALSE,
  ncores = 2,
  print_progress = TRUE
)
}
\arguments{
\item{S}{the set of matrices the bounds should be calculated for in an array, in STVAR applications,
all \eqn{((dp)x(dp))} "bold A" (companion form) matrices in a 3D array, so that \code{[, , m]} gives the matrix
the regime \code{m}.}

\item{epsilon}{a strictly positive real number that approximately defines the goal of length of the interval between the lower
and upper bounds. A smaller epsilon value results in a narrower interval, thus providing better accuracy for the bounds,
but at the cost of increased computational effort. Note that the bounds are always wider than \code{epsilon} and it is not
obvious what \code{epsilon} should be chosen obtain bounds of specific tightness.}

\item{adaptive_eps}{logical: if \code{TRUE}, starts with a large epsilon and then decreases it gradually whenever the progress
of the algorithm requires, until the value given in the argument \code{epsilon} is reached. Usually speeds up the algorithm
substantially but is an unconventional approach, and there is no guarantee that the algorithm converges appropriately towards
bounds with the tightness given by the argument \code{epsilon}.}

\item{ncores}{the number of cores to be used in parallel computing.}

\item{print_progress}{logical: should the progress of the algorithm be printed?}
}
\value{
Returns an upper bound for the joint spectral radius of the "companion form AR matrices" of the regimes.
}
\description{
\code{bound_jsr_G} calculates lower and upper bounds for the joint spectral radious of a set of square matrices,
 typically the "bold A" matrices, using the algorithm by Gripenberg (1996).
}
\details{
The upper and lower bounds are calculated using the Gripenberg's (1996) branch-and-bound method, which is also discussed
 in Chang and Blondel (2013). This function can be generally used for approximating the JSR of a set of square matrices, but the
 main intention is STVAR applications (for models created with \code{sstvars}, the function \code{bound_JSR} should be preferred).
 Specifically, Kheifets and Saikkonen (2020) show that if the joint spectral radius of the companion form AR matrices of the regimes
 is smaller than one, the STVAR process is ergodic stationary. Virolainen (2025) shows the same result for his parametrization of
 of threshold and smooth transition vector autoregressive models. Therefore, if the upper bound is smaller than one, the process is
 stationary ergodic. However, as the condition is not necessary but sufficient and also because the bound might be too conservative,
 upper bound larger than one does not imply that the process is not ergodic stationary. You can try higher accuracy, and if the bound
 is still larger than one, the result does not tell whether the process is ergodic stationary or not.

 Note that with high precision (small \code{epsilon}), the computational effort required are substantial and
 the estimation may take long, even though the function takes use of parallel computing. This is because
 with small epsilon the the number of candidate solutions in each iteration may grow exponentially and a large
 number of iterations may be required. For this reason, \code{adaptive_eps=TRUE} can be considered for large matrices,
 in which case the algorithm starts with a large epsilon, and then decreases it when new candidate solutions are
 not found, until the epsilon given by the argument \code{epsilon} is reached.
}
\examples{
\donttest{
## Below examples take approximately 5 seconds to run.

# A set of two (5x5) square matrices:
set.seed(1); S1 <- array(rnorm(20*20*2), dim=c(5, 5, 2))

# Bound the joint spectral radius of the set of matrices S1, with the
# approximate tightness epsilon=0.01:
bound_jsr_G(S1, epsilon=0.01, adaptive_eps=FALSE)

# Obtain bounds faster with adaptive_eps=TRUE:
bound_jsr_G(S1, epsilon=0.01, adaptive_eps=TRUE)
# Note that the upper bound is not the same as with adaptive_eps=FALSE.

# A set of three (3x3) square matrices:
set.seed(2); S2 <- array(rnorm(3*3*3), dim=c(3, 3, 3))

# Bound the joint spectral radius of the set of matrices S2:
bound_jsr_G(S2, epsilon=0.01, adaptive_eps=FALSE)

# Larger epsilon terminates the iteration earlier and results in wider bounds:
bound_jsr_G(S2, epsilon=0.05, adaptive_eps=FALSE)

# A set of eight (2x2) square matrices:
set.seed(3); S3 <- array(rnorm(2*2*8), dim=c(2, 2, 8))

# Bound the joint spectral radius of the set of matrices S3:
bound_jsr_G(S3, epsilon=0.01, adaptive_eps=FALSE)
}
}
\references{
\itemize{
 \item C-T Chang and V.D. Blondel. 2013 . An experimental study of approximation algorithms for the joint spectral radius.
     \emph{Numerical algorithms}, \strong{64}, 181-202.
   \item Gripenberg, G. 1996. Computing the joint spectral radius. \emph{Linear Algebra and its Applications},
     234, 43–60.
   \item I.L. Kheifets, P.J. Saikkonen. 2020. Stationarity and ergodicity of Vector STAR models.
     \emph{Econometric Reviews}, \strong{39}:4, 407-414.
   \item Virolainen S. 2025. Identification by non-Gaussianity in structural
      smooth transition vector autoregressive models. Unpublished working
      paper, available as arXiv:2404.19707.
 }
}
\seealso{
\code{\link{bound_JSR}}
}
