% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convenienceFunctions.R
\name{semPower.powerMI}
\alias{semPower.powerMI}
\title{semPower.powerMI}
\usage{
semPower.powerMI(type, comparison = NULL, nullEffect = NULL, ...)
}
\arguments{
\item{type}{type of power analysis, one of \code{'a-priori'}, \code{'post-hoc'}, \code{'compromise'}.}

\item{comparison}{comparison model, either \code{'saturated'} or one of \code{'configural'}, \code{'metric'}, \code{'scalar'}, \code{'covariances'}, or a vector of restrictions in \code{lavaan} format (with \code{'none'} for no restrictions). See details.}

\item{nullEffect}{defines the hypothesis (i.e., level of invariance) of interest. One of \code{'metric'}, \code{'scalar'}, \code{'residual'}, \code{'covariances'}, \code{'means'} or a vector of restrictions in \code{lavaan} format. See details.}

\item{...}{mandatory further parameters related to the specific type of power analysis requested, see \code{\link[=semPower.aPriori]{semPower.aPriori()}}, \code{\link[=semPower.postHoc]{semPower.postHoc()}}, and \code{\link[=semPower.compromise]{semPower.compromise()}}, and parameters specifying the factor model. See details.}
}
\value{
a list. Use the \code{summary} method to obtain formatted results. Beyond the results of the power analysis and a number of effect size measures, the list contains the following components:
\item{\code{Sigma}}{the population covariance matrix. A list for multiple group models.}
\item{\code{mu}}{the population mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{SigmaHat}}{the H0 model implied covariance matrix. A list for multiple group models.}
\item{\code{muHat}}{the H0 model implied mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{modelH0}}{\code{lavaan} H0 model string.}
\item{\code{modelH1}}{\code{lavaan} H1 model string or \code{NULL} when the comparison refers to the saturated model.}
\item{\code{simRes}}{detailed simulation results when a simulated power analysis (\code{simulatedPower = TRUE}) was performed.}
}
\description{
Convenience function for performing power analyses for hypothesis arising
in multigroup measurement invariance models concerning a specific level of invariance.
This requires the lavaan package.
}
\details{
This function performs a power analysis to reject various hypotheses arising
in the context of multigroup measurement invariance. Multigroup invariance models
fit the specified model simultaneously to various groups and place increasingly
restrictive cross-group equality constrains on the model parameters. The typical - but not in all parts necessary -
sequence is (a) configural, (b) metric, (c) scalar, and (d) residual invariance, where each level of invariance is
compared against the previous level (e.g., scalar vs. metric). Power analysis provides
the power (or the required N) to reject a particular level of invariance.

For hypotheses regarding longitudinal invariance, see \code{\link[=semPower.powerLI]{semPower.powerLI()}}.

The models defined in the \code{comparison} and the \code{nullEffect} arguments can be specified in two ways. Either specify
a specific level of invariance that includes all previous levels:
\itemize{
\item \code{'configural'}: no invariance constraints. Shows the same fit as the saturated model, so only the delta df differ.
\item \code{'metric'}: all loadings are restricted to equality.
\item \code{'scalar'}: all loadings and (indicator-)intercepts are restricted to equality.
\item \code{'residual'}: all loadings, (indicator-)intercepts, and (indicator-)residuals are restricted to equality.
\item \code{'covariances'}: all loadings, (indicator-)intercepts, and (indicator-)residuals, and latent covariances are restricted to equality.
\item \code{'means'}: all loadings, (indicator-)intercepts, (indicator-)residuals, latent covariances, and latent means are restricted to equality.
}

For example, setting \code{comparison = 'metric'} and \code{nullEffect = 'scalar'} determines power
to reject the hypothesis that the constraints placed in the scalar invariance model
(restricting loadings and intercepts) over the
metric invariance model (restricting only the loadings) are defensible.

For greater flexibility, the models can also be defined using \code{lavaan} style \code{group.equal} restrictions as a vector:
\itemize{
\item \code{'none'}: no invariance constraints and thus representing a configural invariance model. Shows the same fit as the saturated model, so only the delta df differ.
\item \code{c('loadings')}: all loadings are restricted to equality.
\item \code{c('loadings', 'intercepts')}: all loadings and (indicator-)intercepts are restricted to equality.
\item \code{c('loadings', 'intercepts', 'residuals')}: all loadings, (indicator-)intercepts, and (indicator-)residuals are restricted to equality.
\item \code{c('loadings', 'residuals')}: all loadings and (indicator-)residuals are restricted to equality.
\item \code{c('loadings', 'intercepts', 'means')}: all loadings, (indicator-)intercepts, and latent factor means are restricted to equality.
}

For example, setting \code{comparison = c('loadings')} and \verb{nullEffect = 'c('loadings', 'intercepts')'}
determines power to reject the hypothesis that the constraints placed in the scalar invariance model
(restricting loadings and intercepts) over the  metric invariance model (restricting only the loadings) are defensible.
Note that variance scaling is used, so invariance of variances (\code{'lv.variances'}) is always met.

Beyond the arguments explicitly contained in the function call, additional arguments
are required specifying the factor model and the requested type of power analysis.

Additional arguments related to the \strong{definition of the factor model}:
\itemize{
\item \code{Lambda}: The factor loading matrix (with the number of columns equaling the number of factors).
\item \code{loadings}: Can be used instead of \code{Lambda}: Defines the primary loadings for each factor in a list structure, e. g. \code{loadings = list(c(.5, .4, .6), c(.8, .6, .6, .4))} defines a two factor model with three indicators loading on the first factor by .5, , 4., and .6, and four indicators loading on the second factor by .8, .6, .6, and .4.
\item \code{nIndicator}: Can be used instead of \code{Lambda}: Used in conjunction with \code{loadM}. Defines the number of indicators by factor, e. g., \code{nIndicator = c(3, 4)} defines a two factor model with three and four indicators for the first and second factor, respectively. \code{nIndicator} can also be a single number to define the same number of indicators for each factor.
\item \code{loadM}: Can be used instead of \code{Lambda}: Used in conjunction with \code{nIndicator}. Defines the loading either for all indicators (if a single number is provided) or separately for each factor (if a vector is provided), e. g. \code{loadM = c(.5, .6)} defines the loadings of the first factor to equal .5 and those of the second factor do equal .6.
\item \code{Theta}: Variance-covariance matrix of the indicator residuals, which should be a diagonal matrix. Required when residual non-invariance is to be detected. When \code{NULL}, Theta is a diagonal matrix with elements such that all variances are 1.
\item \code{tau}: Defines the item intercepts, required whenever a model involves hypotheses about means (e.g., scalar invariance). If \code{NULL} and \code{Alpha} is set, all intercepts are assumed to equal zero.
\item \code{Alpha}: Defines the latent means, required whenever a model involves hypotheses about latent means (e.g., latent mean invariance). If \code{NULL} and \code{tau} is set, all latent means are assumed to equal zero. Because variance scaling is used so that all factor variances are 1, latent mean differences can be interpreted akin to Cohen's d as standardized mean differences.
}

So either \code{Lambda}, or \code{loadings}, or \code{nIndicator} and \code{loadM} always need to be defined,
and \code{Theta}, \code{tau} and \code{Alpha} need to be defined for particular levels of invariance.
As this function operates on multiple groups, either argument is a list whenever there are
group differences in the respective parameters. When no list is provided, the same
parameter values are assumed for all groups.

Additional arguments related to the requested type of \strong{power analysis}:
\itemize{
\item \code{alpha}: The alpha error probability. Required for \code{type = 'a-priori'} and \code{type = 'post-hoc'}.
\item Either \code{beta} or \code{power}: The beta error probability and the statistical power (1 - beta), respectively. Only for \code{type = 'a-priori'}.
\item \code{N}: The sample size. Always required for \code{type = 'post-hoc'} and \code{type = 'compromise'}. For \code{type = 'a-priori'} and multiple group analysis, \code{N} is a list of group weights.
\item \code{abratio}: The ratio of alpha to beta. Only for \code{type = 'compromise'}.
}

If a \strong{simulated power analysis} (\code{simulatedPower = TRUE}) is requested, optional arguments can be provided as a list to \code{simOptions}:
\itemize{
\item \code{nReplications}: The targeted number of simulation runs. Defaults to 250, but larger numbers greatly improve accuracy at the expense of increased computation time.
\item \code{minConvergenceRate}:  The minimum convergence rate required, defaults to .5. The maximum actual simulation runs are increased by a factor of 1/minConvergenceRate.
\item \code{type}: specifies whether the data should be generated from a population assuming multivariate normality (\code{'normal'}; the default), or based on an approach generating non-normal data (\code{'IG'}, \code{'mnonr'}, \code{'RC'}, or \code{'VM'}).
The approaches generating non-normal data require additional arguments detailed below.
\item \code{missingVars}: vector specifying the variables containing missing data (defaults to NULL).
\item \code{missingVarProp}: can be used instead of \code{missingVars}: The proportion of variables containing missing data (defaults to zero).
\item \code{missingProp}: The proportion of missingness for variables containing missing data (defaults to zero), either a single value or a vector giving the probabilities for each variable.
\item \code{missingMechanism}: The missing data mechanism, one of \code{MCAR} (the default), \code{MAR}, or \code{NMAR}.
\item \code{nCores}: The number of cores to use for parallel processing. Defaults to 1 (= no parallel processing). This requires the \code{doSNOW} package.
}

\code{type = 'IG'} implements the independent generator approach (IG, Foldnes & Olsson, 2016) approach
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors. This requires the \code{covsim} package.

\code{type = 'mnonr'} implements the approach suggested by Qu, Liu, & Zhang (2020) and requires provision of  Mardia's multivariate skewness (\code{skewness})  and kurtosis  (\code{kurtosis}), where
skewness must be non-negative and kurtosis must be at least 1.641 skewness + p (p + 0.774), where p is the number of variables. This requires the \code{mnonr} package.

\code{type = 'RK'} implements the approach suggested by Ruscio & Kaczetow (2008) and requires provision of the population distributions
of each variable (\code{distributions}). \code{distributions} must be a list (if all variables shall be based on the same population distribution) or a list of lists.
Each component must specify the population distribution (e.g. \code{rchisq}) and additional arguments (\code{list(df = 2)}).

\code{type = 'VM'} implements the third-order polynomial method (Vale & Maurelli, 1983)
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors.  This requires the \code{semTools} package.
}
\examples{
\dontrun{
# obtain the required N to reject the hypothesis of metric invariance
# in comparison to the configural invariance model 
# with a power of 95\% on alpha = 5\% 
# assuming equally sized groups (N = list(1, 1)) 
# for a factor model involving a single factor which 
# is measured by 5 indicators (in both groups)
# loading by .5 each in the first group and 
# loading by .6 each in the second group.
powerMI <- semPower.powerMI(type = 'a-priori',
                            comparison = 'configural', 
                            nullEffect = 'metric',
                            nIndicator = list(5, 5),
                            loadM = list(.5, .6),
                            alpha = .05, beta = .05, N = list(1, 1))

# show summary
summary(powerMI)
# optionally use lavaan to verify the model was set-up as intended
lavaan::sem(powerMI$modelH1, sample.cov = list(powerMI$Sigma[[1]], powerMI$Sigma[[2]]),
            sample.nobs = as.list(powerMI$requiredN.g), sample.cov.rescale = FALSE)
lavaan::sem(powerMI$modelH0, sample.cov = list(powerMI$Sigma[[1]], powerMI$Sigma[[2]]),
            sample.nobs = as.list(powerMI$requiredN.g), sample.cov.rescale = FALSE)

# same as above, but determine power with N = 500 in each group on alpha = .05
powerMI <- semPower.powerMI(type = 'post-hoc',
                            comparison = 'configural', 
                            nullEffect = 'metric',
                            nIndicator = 5,
                            loadM = list(.5, .6),
                            alpha = .05, N = list(500, 500))

# same as above, but determine the critical chi-square with N = 500 in each 
# group so that alpha = beta
powerMI <- semPower.powerMI(type = 'compromise',
                            comparison = 'configural', 
                            nullEffect = 'metric',
                            nIndicator = 5,
                            loadM = list(.5, .6),
                            abratio = 1, N = list(500, 500))

# same as above, but compare to the saturated model
# (rather than to the configural invariance model)
powerMI <- semPower.powerMI(type = 'a-priori',
                            comparison = 'saturated', 
                            nullEffect = 'metric',
                            nIndicator = 5,
                            loadM = list(.5, .6),
                            alpha = .05, beta = .05, N = list(1, 1))

# same as above, but provide individual factor loadings by group using a 
# reduced loading matrix to define a  single factor model with three indicators
# loading by .4, .6, .5 in the first group and 
# loading by .5, .6, .7 in the second group
powerMI <- semPower.powerMI(type = 'a-priori',
                            comparison = 'saturated', 
                            nullEffect = 'metric',
                            loadings = list(
                              list(c(.4, .6, .5)), 
                              list(c(.5, .6, .7))),
                            alpha = .05, beta = .05, N = list(1, 1))

# same as above, but make first group twice as large as the second group 
powerMI <- semPower.powerMI(type = 'a-priori',
                            comparison = 'saturated', 
                            nullEffect = 'metric',
                            loadings = list(
                              list(c(.4, .6, .5)), 
                              list(c(.5, .6, .7))),
                            alpha = .05, beta = .05, N = list(2, 1))

# obtain the required N to reject the hypothesis of scalar invariance
# in comparison to the metric invariance model 
# with a power of 95\% on alpha = 5\% 
# assuming equally sized groups (N = list(1, 1)) 
# for a two factor model, where both factors are  
# measured by 3 indicators each and all loadings equal .5 (in both groups),
# the factor correlation is .3 in both groups, and the
# all intercepts are 0.0 in the first group, but
# the intercepts are .1, .2, .3, .4, .5, .6 in the second group
powerMI <- semPower.powerMI(type = 'a-priori',
                            comparison = 'metric', 
                            nullEffect = 'scalar',
                            Phi = list(.3, .3),
                            nIndicator = list(
                              c(3, 3), 
                              c(3, 3)),
                            loadM = .5,
                            tau = list(
                              rep(0.0, 6), 
                              seq(.1, .6, .1) 
                            ),
                            alpha = .05, beta = .05, N = list(1, 1))

# same as above, but use lavaan group.equal strings 
powerMI <- semPower.powerMI(type = 'a-priori',
                            comparison = c('loadings'), 
                            nullEffect = c('loadings', 'intercepts'),
                            Phi = list(.3, .3),
                            nIndicator = list(
                              c(3, 3), 
                              c(3, 3)),
                            loadM = .5,
                            tau = list(
                              rep(0.0, 6), 
                              seq(.1, .6, .1) 
                            ),
                            alpha = .05, beta = .05, N = list(1, 1))

# same as above, but
# obtain the required N to reject the hypothesis of equal latent means
# in comparison to the scalar invariance model;
# all intercepts are zero in both groups, 
# in the first group, the latent means equal 0.0, 
# in the second group, the latent mean of the factors are 0.0 and 0.5
powerMI <- semPower.powerMI(type = 'a-priori',
                            comparison = c('loadings', 'intercepts'), 
                            nullEffect = c('loadings', 'intercepts', 'means'),
                            Phi = list(.3, .3),
                            nIndicator = list(
                              c(3, 3), 
                              c(3, 3)),
                            loadM = .5,
                            tau = list(
                              rep(0.0, 6), 
                              rep(0.0, 6) 
                            ),
                            Alpha = list(
                              c(0.0, 0.0),
                              c(0.0, 0.5)
                            ),
                            alpha = .05, beta = .05, N = list(1, 1))

# request a simulated post-hoc power analysis with 500 replications
# to reject the hypothesis of metric invariance.
set.seed(300121)
powerMI <- semPower.powerMI(type = 'post-hoc',
                            comparison = 'configural', 
                            nullEffect = 'metric',
                            nIndicator = list(5, 5),
                            loadM = list(.5, .6),
                            alpha = .05, N = list(500, 500), 
                            simulatedPower = TRUE, 
                            simOptions = list(nReplications = 500))
                             
}
}
\seealso{
\code{\link[=semPower.genSigma]{semPower.genSigma()}} \code{\link[=semPower.aPriori]{semPower.aPriori()}} \code{\link[=semPower.postHoc]{semPower.postHoc()}} \code{\link[=semPower.compromise]{semPower.compromise()}}
}
