#' Retrieve GWAS credible set data
#'
#' Provided with a study ID and a lead variant ID, this function returns a
#' data frame consisting of all the associated credible set tag variants
#'  with the corresponding statistical data.
#'
#' @param study_id Character: Study ID(s) generated by Open Targets Genetics (e.g GCST90002357).
#' @param variant_id Character: generated ID for variants by Open Targets Genetics (e.g. 1_154119580_C_A) or rsId (rs2494663).
#'
#' @return Returns a data frame of results from the credible set of variants for a specific lead variant with the following columns:
#' \itemize{
#'   \item{\code{tagVariant.id}:} \emph{Data frame}. A table of IDs of the tag variant.
#'   \item{\code{tagVariant.rsId}:} \emph{Character vector}. rsID of the tag variant.
#'   \item{\code{beta}:} \emph{Numeric}. Beta value.
#'   \item{\code{postProb}:} \emph{Numeric}. Posterior probability.
#'   \item{\code{pval}:} \emph{Numeric}. P-value.
#'   \item{\code{se}:} \emph{Numeric}. Standard error.
#'   \item{\code{MultisignalMethod}:} \emph{Character vector}. Multisignal method.
#'   \item{\code{logABF}:} \emph{Numeric}. Logarithm of approximate Bayes factor.
#'   \item{\code{is95}:} \emph{Logical}. Indicates if the variant has a 95% credible set.
#'   \item{\code{is99}:} \emph{Logical}. Indicates if the variant has a 99% credible set.
#' }
#' @examples
#' \dontrun{
#' result <- gwasCredibleSet(study_id="GCST90002357", variant_id="1_154119580_C_A")
#' result <- gwasCredibleSet(study_id="GCST90002357", variant_id="rs2494663")
#' }
#' @import dplyr
#' @importFrom magrittr %>%
#' @export
#'

gwasCredibleSet <- function(study_id, variant_id) {

  ## Query for GWAS study locus details
tryCatch({
  cli::cli_progress_step("Connecting to the Open Targets Genetics GrpahQL API...", spinner = TRUE)
  otg_cli <- ghql::GraphqlClient$new(url = "https://api.genetics.opentargets.org/graphql")
  otg_qry <- ghql::Query$new()

  # Check variant id format
  if (grepl(pattern = "rs\\d+", variant_id)) {

    # Convert rs id to variant id
    query_searchid <- "query ConvertRSIDtoVID($queryString:String!) {
      search(queryString:$queryString){
        totalVariants
        variants{
          id
        }
      }
    }"

    variables <- list(queryString = variant_id)
    otg_qry$query(name = "convertid", x = query_searchid)
    id_result <- jsonlite::fromJSON(otg_cli$exec(otg_qry$queries$convertid, variables), flatten = TRUE)$data
    input_variant_id <- id_result$search$variants$id
  }
  else if (grepl(pattern = "\\d+_\\d+_[a-zA-Z]+_[a-zA-Z]+", variant_id)) {
    input_variant_id <- variant_id
  }
  else {
    stop("\n Please provide a variant Id")
  }

  query <- "query credsetQuery($studyId: String!, $variantId: String!) {
    gwasCredibleSet(studyId: $studyId, variantId: $variantId) {
      tagVariant {
        id
        rsId
      }
      beta
      postProb
      pval
      se
      MultisignalMethod
      logABF
      is95
      is99
    }
  }"

  ## Execute the query

  output <- data.frame()
  variables <- list(studyId = study_id, variantId = input_variant_id)

  otg_qry$query(name = "credset_query", x = query)

  cli::cli_progress_step("Downloading the data...", spinner = TRUE)
  result <- jsonlite::fromJSON(otg_cli$exec(otg_qry$queries$credset_query, variables, flatten = TRUE))$data

  output <- result$gwasCredibleSet %>% as.data.frame()

  return(output)

}, error = function(e) {
  # Handling connection timeout
  if(grepl("Timeout was reached", e$message)) {
    stop("Connection timeout reached while connecting to the Open Targets Genetics GraphQL API.")
  } else {
    stop(e) # Handle other types of errors
  }
})
}
