% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aov.b.R
\name{aov.b}
\alias{aov.b}
\title{Between-Subject Analysis of Variance}
\usage{
aov.b(formula, data, posthoc = FALSE, conf.level = 0.95, hypo = TRUE,
      descript = TRUE, effsize = FALSE, weighted = FALSE, correct = FALSE,
      digits = 2, p.digits = 3, as.na = NULL, plot = FALSE, bar = TRUE,
      point = FALSE, ci = TRUE, jitter = FALSE, adjust = TRUE,
      point.size = 3, errorbar.width = 0.1, jitter.size = 1.25,
      jitter.width = 0.05, jitter.height = 0, jitter.alpha = 0.1,
      xlab = NULL, ylab = "y", ylim = NULL, ybreaks = ggplot2::waiver(),
      title = NULL, subtitle = "Confidence Interval", filename = NULL,
      width = NA, height = NA, units = c("in", "cm", "mm", "px"), dpi = 600,
      write = NULL, append = TRUE, check = TRUE, output = TRUE)
}
\arguments{
\item{formula}{a formula of the form \code{y ~ group} where \code{y} is
a numeric variable giving the data values and \code{group}
a numeric variable, character variable or factor with more
than two values or factor levels giving the corresponding
groups.}

\item{data}{a matrix or data frame containing the variables in the
formula \code{formula}.}

\item{posthoc}{logical: if \code{TRUE}, Tukey HSD post hoc test for
multiple comparison is conducted.}

\item{conf.level}{a numeric value between 0 and 1 indicating the confidence
level of the interval.}

\item{hypo}{logical: if \code{TRUE} (default), null and alternative hypothesis
are shown on the console.}

\item{descript}{logical: if \code{TRUE} (default), descriptive statistics are shown
on the console.}

\item{effsize}{logical: if \code{TRUE}, effect size measures \eqn{\eta^2}
and \eqn{\omega^2} for the ANOVA and Cohen's d for the post
hoc tests are shown on the console.}

\item{weighted}{logical: if \code{TRUE}, the weighted pooled standard
deviation is used to compute Cohen's d.}

\item{correct}{logical: if \code{TRUE}, correction factor to remove
positive bias in small samples is used.}

\item{digits}{an integer value indicating the number of decimal places
to be used for displaying descriptive statistics and
confidence interval.}

\item{p.digits}{an integer value indicating the number of decimal places
to be used for displaying the \emph{p}-value.}

\item{as.na}{a numeric vector indicating user-defined missing values,
i.e. these values are converted to \code{NA} before conducting
the analysis.}

\item{plot}{logical: if \code{TRUE}, a plot showing error bars for
confidence intervals is drawn.}

\item{bar}{logical: if \code{TRUE} (default), bars representing means
for each groups are drawn.}

\item{point}{logical: if \code{TRUE}, points representing means for
each groups are drawn.}

\item{ci}{logical: if \code{TRUE} (default), error bars representing
confidence intervals are drawn.}

\item{jitter}{logical: if \code{TRUE}, jittered data points are drawn.}

\item{adjust}{logical: if \code{TRUE} (default), difference-adjustment
for the confidence intervals is applied.}

\item{point.size}{a numeric value indicating the \code{size} aesthetic for
the point representing the mean value.}

\item{errorbar.width}{a numeric value indicating the horizontal bar width of
the error bar.}

\item{jitter.size}{a numeric value indicating the \code{size} aesthetic
for the jittered data points.}

\item{jitter.width}{a numeric value indicating the amount of horizontal jitter.}

\item{jitter.height}{a numeric value indicating the amount of vertical jitter.}

\item{jitter.alpha}{a numeric value between 0 and 1 for specifying the
\code{alpha} argument in the \code{geom_histogram}
function for controlling the opacity of the jittered
data points.}

\item{xlab}{a character string specifying the labels for the x-axis.}

\item{ylab}{a character string specifying the labels for the y-axis.}

\item{ylim}{a numeric vector of length two specifying limits of the
limits of the y-axis.}

\item{ybreaks}{a numeric vector specifying the points at which tick-marks
are drawn at the y-axis.}

\item{title}{a character string specifying the text for the title of
the plot.}

\item{subtitle}{a character string specifying the text for the subtitle of
the plot.}

\item{filename}{a character string indicating the \code{filename}
argument including the file extension in the \code{ggsave}
function. Note that one of \code{".eps"}, \code{".ps"},
\code{".tex"}, \code{".pdf"} (default),
\code{".jpeg"}, \code{".tiff"}, \code{".png"},
\code{".bmp"}, \code{".svg"} or \code{".wmf"} needs
to be specified as file extension in the \code{filename}
argument. Note that plots can only be saved when
\code{plot = TRUE}.}

\item{width}{a numeric value indicating the \code{width} argument
(default is the size of the current graphics device)
in the \code{ggsave} function.}

\item{height}{a numeric value indicating the \code{height} argument
(default is the size of the current graphics device)
in the \code{ggsave} function.}

\item{units}{a character string indicating the \code{units} argument
(default is \code{in}) in the \code{ggsave} function.}

\item{dpi}{a numeric value indicating the \code{dpi} argument
(default is \code{600}) in the \code{ggsave} function.}

\item{write}{a character string naming a text file with file extension
\code{".txt"} (e.g., \code{"Output.txt"}) for writing the
output into a text file.}

\item{append}{logical: if \code{TRUE} (default), output will be appended
to an existing text file with extension \code{.txt} specified
in \code{write}, if \code{FALSE} existing text file will be
overwritten.}

\item{check}{logical: if \code{TRUE} (default), argument specification is checked.}

\item{output}{logical: if \code{TRUE} (default), output is shown on the console.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:
  \item{\code{call}}{function call}
  \item{\code{type}}{type of analysis}
  \item{\code{data}}{data frame with variables used in the current analysis}
  \item{\code{formula}}{formula of the current analysis}
  \item{\code{args}}{specification of function arguments}
  \item{\code{plot}}{ggplot2 object for plotting the results}
  \item{\code{result}}{list with result tables, i.e., \code{descript} for descriptive
                       statistics, \code{test} for the ANOVA table, \code{posthoc}
                       for post hoc tests, and \code{aov} for the return object
                       of the \code{aov} function}
}
\description{
This function performs an one-way between-subject analysis of variance (ANOVA)
including Tukey HSD post hoc tests for multiple comparison and provides descriptive
statistics, effect size measures, and a plot showing bars representing means
for each group and error bars for difference-adjusted confidence intervals.
}
\details{
\describe{
\item{\strong{Post Hoc Test}}{Tukey HSD post hoc test reports Cohen's d based
on the non-weighted standard deviation (i.e., \code{weighted = FALSE}) when
requesting an effect size measure  (i.e., \code{effsize = TRUE}) following the
recommendation by Delacre et al. (2021).
}
\item{\strong{Confidence Intervals}}{Cumming and Finch (2005) pointed out that
when 95\% confidence intervals (CI) for two separately plotted means overlap,
it is still possible that the CI for the difference would not include zero.
Baguley (2012) proposed to adjust the width of the CIs by the factor of
\eqn{\sqrt{2}} to reflect the correct width of the CI for a mean difference:

\deqn{\hat{\mu}_{j} \pm t_{n - 1, 1 - \alpha/2} \frac{\sqrt{2}}{2} \hat{\sigma}_{{\hat{\mu}}_j}}}

These difference-adjusted CIs around the individual means can be interpreted
as if it were a CI for their difference. Note that the width of these intervals
is sensitive to differences in the variance and sample size of each sample,
i.e., unequal population variances and unequal \eqn{n} alter the interpretation
of difference-adjusted CIs.
}
}
\examples{
# Example 1: Between-subject ANOVA
aov.b(mpg ~ gear, data = mtcars)

# Example 2: Between-subject ANOVA
# print effect size measures
aov.b(mpg ~ gear, data = mtcars, effsize = TRUE)

# Example 3: Between-subject ANOVA
# do not print hypotheses and descriptive statistics,
aov.b(mpg ~ gear, data = mtcars, descript = FALSE, hypo = FALSE)

# Example 4: Between-subject ANOVA
# plot results
aov.b(mpg ~ gear, data = mtcars, plot = TRUE)

\dontrun{
# Example 5: Write Results into a text file
aov.b(mpg ~ gear, data = mtcars, write = "ANOVA.txt")

# Example 6: Save plot
aov.b(mpg ~ gear, data = mtcars, plot = TRUE, filename = "Between-Subject_ANOVA.png",
      width = 7, height = 6)

# Example 7: Draw plot in line with the default setting of aov.b()
library(ggplot2)

object <- aov.b(mpg ~ gear, data = mtcars, output = FALSE)

ggplot(object$data, aes(group, y)) +
  geom_bar(stat = "summary", fun = "mean") +
  geom_errorbar(data = ci.mean(object$data, y, group = "group", adjust = TRUE,
                output = FALSE)$result,
                aes(group, m, ymin = low, ymax = upp), width = 0.1) +
  scale_x_discrete(name = NULL) +
  labs(subtitle = "Two-Sided Difference-Adjusted Confidence Interval") +
  theme_bw() + theme(plot.subtitle = element_text(hjust = 0.5))
}
}
\references{
Baguley, T. S. (2012a). \emph{Serious stats: A guide to advanced statistics for
the behavioral sciences}. Palgrave Macmillan.

Cumming, G., and Finch, S. (2005) Inference by eye: Confidence intervals, and
how to read pictures of data. \emph{American Psychologist, 60}, 170–80.

Delacre, M., Lakens, D., Ley, C., Liu, L., & Leys, C. (2021). Why Hedges' g*s
based on the non-pooled standard deviation should be reported with Welch's t-test.
https://doi.org/10.31234/osf.io/tu6mp

Rasch, D., Kubinger, K. D., & Yanagida, T. (2011). \emph{Statistics in psychology
- Using R and SPSS}. John Wiley & Sons.
}
\seealso{
\code{\link{aov.w}}, \code{\link{test.t}}, \code{\link{test.z}},
\code{\link{test.levene}}, \code{\link{test.welch}}, \code{\link{cohens.d}},
\code{\link{ci.mean.diff}}, \code{\link{ci.mean}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
