% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/confint-methods.R
\name{confint.gam}
\alias{confint.gam}
\alias{confint.gamm}
\alias{confint.list}
\title{Point-wise and simultaneous confidence intervals for smooths}
\usage{
\method{confint}{gam}(
  object,
  parm,
  level = 0.95,
  data = newdata,
  n = 100,
  type = c("confidence", "simultaneous"),
  nsim = 10000,
  shift = FALSE,
  transform = FALSE,
  unconditional = FALSE,
  ncores = 1,
  partial_match = FALSE,
  ...,
  newdata = NULL
)

\method{confint}{gamm}(object, ...)

\method{confint}{list}(object, ...)
}
\arguments{
\item{object}{an object of class \code{"gam"} or \code{"gamm"}.}

\item{parm}{which parameters (smooth terms) are to be given intervals as a
vector of terms. If missing, all parameters are considered, although this
is not currently implemented.}

\item{level}{numeric, \verb{0 < level < 1}; the confidence level of the point-wise
or simultaneous interval. The default is \code{0.95} for a 95\% interval.}

\item{data}{data frame; new values of the covariates used in the model fit.
The selected smooth(s) will be evaluated at the supplied values.}

\item{n}{numeric; the number of points to evaluate smooths at.}

\item{type}{character; the type of interval to compute. One of \code{"confidence"}
for point-wise intervals, or \code{"simultaneous"} for simultaneous intervals.}

\item{nsim}{integer; the number of simulations used in computing the
simultaneous intervals.}

\item{shift}{logical; should the constant term be add to the smooth?}

\item{transform}{logical; should the smooth be evaluated on a transformed
scale? For generalised models, this involves applying the inverse of the
link function used to fit the model. Alternatively, the name of, or an
actual, function can be supplied to transform the smooth and it's
confidence interval.}

\item{unconditional}{logical; if \code{TRUE} (and \code{freq == FALSE}) then the
Bayesian smoothing parameter uncertainty corrected covariance matrix is
returned, if available.}

\item{ncores}{number of cores for generating random variables from a
multivariate normal distribution. Passed to \code{\link[mvnfast:rmvn]{mvnfast::rmvn()}}.
Parallelization will take place only if OpenMP is supported (but appears
to work on Windows with current \code{R}).}

\item{partial_match}{logical; should matching \code{parm} use a partial match or
an exact match? Can only be used if \code{length(parm)} is \code{1}.}

\item{...}{additional arguments for methods}

\item{newdata}{DEPRECATED! data frame; containing new values of the
covariates used in the model fit. The selected smooth(s) will be evaluated
at the supplied values.}
}
\value{
a tibble with components:
\enumerate{
\item \code{.smooth}; character indicating to which term each row relates,
\item \code{.type}; the type of smooth,
\item \code{.by} the name of the by variable if a by smooth, \code{NA} otherwise,
\item one or more vectors of values at which the smooth was evaluated, named as
per the variables in the smooth,
\item zero or more variables containing values of the by variable,
\item \code{.estimate}; estimated value of the smooth,
\item \code{.se}; standard error of the estimated value of the smooth,
\item \code{.crit}; critical value for the \code{100 * level}\% confidence interval.
\item \code{.lower_ci}; lower limit of the confidence or simultaneous interval,
\item \code{.upper_ci}; upper limit of the confidence or simultaneous interval,
}
}
\description{
Calculates point-wise confidence or simultaneous intervals for the smooth
terms of a fitted GAM.
}
\examples{
load_mgcv()
\dontshow{
op <- options(pillar.sigfig = 2, cli.unicode = FALSE)
}
dat <- data_sim("eg1", n = 1000, dist = "normal", scale = 2, seed = 2)
mod <- gam(y ~ s(x0) + s(x1) + s(x2) + s(x3), data = dat, method = "REML")

# new data to evaluate the smooths at, say over the middle 50\% of range
# of each covariate
middle <- function(x, n = 50, coverage = 0.5) {
  v <- (1 - coverage) / 2
  q <- quantile(x, prob = c(0 + v, 1 - v), type = 8)
  seq(q[1], q[2], length = n)
}
new_data <- sapply(dat[c("x0", "x1", "x2", "x3")], middle)
new_data <- data.frame(new_data)

## point-wise interval for smooth of x2
ci <- confint(mod, parm = "s(x2)", type = "confidence", data = new_data)
ci
\dontshow{
options(op)
}
}
\author{
Gavin L. Simpson
}
