% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{qkw}
\alias{qkw}
\title{Quantile Function of the Kumaraswamy (Kw) Distribution}
\usage{
qkw(p, alpha, beta, lower_tail = TRUE, log_p = FALSE)
}
\arguments{
\item{p}{Vector of probabilities (values between 0 and 1).}

\item{alpha}{Shape parameter \code{alpha} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{beta}{Shape parameter \code{beta} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{lower_tail}{Logical; if \code{TRUE} (default), probabilities are \eqn{p = P(X \le q)},
otherwise, probabilities are \eqn{p = P(X > q)}.}

\item{log_p}{Logical; if \code{TRUE}, probabilities \code{p} are given as
\eqn{\log(p)}. Default: \code{FALSE}.}
}
\value{
A vector of quantiles corresponding to the given probabilities \code{p}.
The length of the result is determined by the recycling rule applied to
the arguments (\code{p}, \code{alpha}, \code{beta}).
Returns:
\itemize{
\item \code{0} for \code{p = 0} (or \code{p = -Inf} if \code{log_p = TRUE},
when \code{lower_tail = TRUE}).
\item \code{1} for \code{p = 1} (or \code{p = 0} if \code{log_p = TRUE},
when \code{lower_tail = TRUE}).
\item \code{NaN} for \code{p < 0} or \code{p > 1} (or corresponding log scale).
\item \code{NaN} for invalid parameters (e.g., \code{alpha <= 0},
\code{beta <= 0}).
}
Boundary return values are adjusted accordingly for \code{lower_tail = FALSE}.
}
\description{
Computes the quantile function (inverse CDF) for the two-parameter
Kumaraswamy (Kw) distribution with shape parameters \code{alpha} (\eqn{\alpha})
and \code{beta} (\eqn{\beta}). It finds the value \code{q} such that
\eqn{P(X \le q) = p}.
}
\details{
The quantile function \eqn{Q(p)} is the inverse of the CDF \eqn{F(q)}. The CDF
for the Kumaraswamy distribution is \eqn{F(q) = 1 - (1 - q^\alpha)^\beta}
(see \code{\link{pkw}}). Inverting this equation for \eqn{q} yields the
quantile function:
\deqn{
Q(p) = \left\{ 1 - (1 - p)^{1/\beta} \right\}^{1/\alpha}
}
The function uses this closed-form expression and correctly handles the
\code{lower_tail} and \code{log_p} arguments by transforming \code{p}
appropriately before applying the formula. This is equivalent to the general
GKw quantile function (\code{\link{qgkw}}) evaluated with \eqn{\gamma=1, \delta=0, \lambda=1}.
}
\examples{
\donttest{
# Example values
p_vals <- c(0.1, 0.5, 0.9)
alpha_par <- 2.0
beta_par <- 3.0

# Calculate quantiles using qkw
quantiles <- qkw(p_vals, alpha_par, beta_par)
print(quantiles)

# Calculate quantiles for upper tail probabilities P(X > q) = p
quantiles_upper <- qkw(p_vals, alpha_par, beta_par, lower_tail = FALSE)
print(quantiles_upper)
# Check: qkw(p, ..., lt=F) == qkw(1-p, ..., lt=T)
print(qkw(1 - p_vals, alpha_par, beta_par))

# Calculate quantiles from log probabilities
log_p_vals <- log(p_vals)
quantiles_logp <- qkw(log_p_vals, alpha_par, beta_par, log_p = TRUE)
print(quantiles_logp)
# Check: should match original quantiles
print(quantiles)

# Compare with qgkw setting gamma = 1, delta = 0, lambda = 1
quantiles_gkw <- qgkw(p_vals, alpha = alpha_par, beta = beta_par,
                     gamma = 1.0, delta = 0.0, lambda = 1.0)
print(paste("Max difference:", max(abs(quantiles - quantiles_gkw)))) # Should be near zero

# Verify inverse relationship with pkw
p_check <- 0.75
q_calc <- qkw(p_check, alpha_par, beta_par)
p_recalc <- pkw(q_calc, alpha_par, beta_par)
print(paste("Original p:", p_check, " Recalculated p:", p_recalc))
# abs(p_check - p_recalc) < 1e-9 # Should be TRUE

# Boundary conditions
print(qkw(c(0, 1), alpha_par, beta_par)) # Should be 0, 1
print(qkw(c(-Inf, 0), alpha_par, beta_par, log_p = TRUE)) # Should be 0, 1

}

}
\references{
Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.

Jones, M. C. (2009). Kumaraswamy's distribution: A beta-type distribution
with some tractability advantages. \emph{Statistical Methodology}, \emph{6}(1), 70-81.
}
\seealso{
\code{\link{qgkw}} (parent distribution quantile function),
\code{\link{dkw}}, \code{\link{pkw}}, \code{\link{rkw}} (other Kw functions),
\code{\link[stats]{qbeta}}, \code{\link[stats]{qunif}}
}
\author{
Lopes, J. E.
}
\keyword{distribution}
\keyword{kumaraswamy}
\keyword{quantile}
