% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg.R
\name{plot.gkwreg}
\alias{plot.gkwreg}
\title{Diagnostic Plots for Generalized Kumaraswamy Regression Models}
\usage{
\method{plot}{gkwreg}(
  x,
  which = 1:6,
  caption = c("Residuals vs. Observation Indices", "Cook's Distance Plot",
    "Generalized Leverage vs. Fitted Values", "Residuals vs. Linear Predictor",
    "Half-Normal Plot of Residuals", "Predicted vs. Observed Values"),
  sub.caption = paste(deparse(x$call), collapse = "\\n"),
  main = "",
  ask = prod(par("mfcol")) < length(which) && dev.interactive(),
  ...,
  type = c("quantile", "pearson", "deviance"),
  family = NULL,
  nsim = 100,
  level = 0.9,
  use_ggplot = FALSE,
  arrange_plots = FALSE,
  sample_size = NULL,
  theme_fn = ggplot2::theme_minimal,
  save_diagnostics = FALSE
)
}
\arguments{
\item{x}{An object of class \code{"gkwreg"}, typically the result of a
call to \code{\link{gkwreg}}.}

\item{which}{Integer vector specifying which diagnostic plots to produce.
If a subset of the plots is required, specify a subset of the numbers 1:6.
Defaults to \code{1:6}. The plots correspond to:
\enumerate{
\item Residuals vs. Observation Indices: Checks for time trends or patterns.
\item Cook's Distance Plot: Helps identify influential observations.
\item Generalized Leverage vs. Fitted Values: Identifies points with high leverage.
\item Residuals vs. Linear Predictor: Checks for non-linearity and heteroscedasticity.
\item Half-Normal Plot of Residuals (with simulated envelope): Assesses normality
of residuals, comparing against simulated quantiles.
\item Predicted vs. Observed Values: Checks overall model prediction accuracy.
}}

\item{caption}{Character vector providing captions (titles) for the plots.
Its length must be at least \code{max(which)}. Defaults are provided for plots 1-6.}

\item{sub.caption}{Character string used as a common subtitle positioned above all plots
(especially when multiple plots are arranged). Defaults to the deparsed model call.}

\item{main}{An optional character string to be prepended to the individual plot captions
(from the \code{caption} argument).}

\item{ask}{Logical. If \code{TRUE} (and using base R graphics with multiple plots
on an interactive device), the user is prompted before displaying each plot.
Defaults to \code{TRUE} if more plots are requested than fit on the current screen layout.}

\item{...}{Additional arguments passed to the underlying plotting functions
(e.g., graphical parameters like \code{col}, \code{pch}, \code{cex} for base R plots).}

\item{type}{Character string indicating the type of residuals to be used for plotting.
Defaults to \code{"quantile"}. Valid options are:
\itemize{
\item \code{"quantile"}: Randomized quantile residuals (Dunn & Smyth, 1996).
Recommended for bounded responses as they should be approximately N(0,1)
if the model is correctly specified.
\item \code{"pearson"}: Pearson residuals (response residual standardized by
estimated standard deviation). Useful for checking the variance function.
\item \code{"deviance"}: Deviance residuals. Related to the log-likelihood
contribution of each observation.
}}

\item{family}{Character string specifying the distribution family assumptions
to use when calculating residuals and other diagnostics. If \code{NULL} (default),
the family stored within the fitted \code{object} is used. Specifying a different
family can be useful for diagnostic comparisons. Available options match those
in \code{\link{gkwreg}}: \code{"gkw", "bkw", "kkw", "ekw", "mc", "kw", "beta"}.}

\item{nsim}{Integer. Number of simulations used to generate the envelope in the
half-normal plot (\code{which = 5}). Defaults to 100. Must be positive.}

\item{level}{Numeric. The confidence level (between 0 and 1) for the simulated
envelope in the half-normal plot (\code{which = 5}). Defaults to 0.90.}

\item{use_ggplot}{Logical. If \code{TRUE}, plots are generated using the \code{ggplot2}
package. If \code{FALSE} (default), base R graphics are used. Requires the
\code{ggplot2} package to be installed if set to \code{TRUE}.}

\item{arrange_plots}{Logical. Only relevant if \code{use_ggplot = TRUE} and multiple
plots are requested (\code{length(which) > 1}). If \code{TRUE}, attempts to arrange
the generated \code{ggplot} objects into a grid using either the \code{gridExtra}
or \code{ggpubr} package (requires one of them to be installed). Defaults to \code{FALSE}.}

\item{sample_size}{Integer or \code{NULL}. If specified as an integer less than the
total number of observations (\code{x$nobs}), a random sample of this size is
used for calculating diagnostics and plotting. This can be useful for speeding up
plots with very large datasets. Defaults to \code{NULL} (use all observations).}

\item{theme_fn}{A function. Only relevant if \code{use_ggplot = TRUE}. Specifies a
\code{ggplot2} theme function to apply to the plots (e.g., \code{theme_bw},
\code{theme_classic}). Defaults to \code{ggplot2::theme_minimal}.}

\item{save_diagnostics}{Logical. If \code{TRUE}, the function invisibly returns a list
containing the calculated diagnostic measures (residuals, leverage, Cook's distance, etc.)
instead of the model object. If \code{FALSE} (default), the function invisibly
returns the original model object \code{x}.}
}
\value{
Invisibly returns either the original fitted model object \code{x}
(if \code{save_diagnostics = FALSE}) or a list containing the calculated
diagnostic measures used for plotting (if \code{save_diagnostics = TRUE}).
Primarily called for its side effect of generating plots.
}
\description{
Produces a set of diagnostic plots for assessing the adequacy of a fitted
Generalized Kumaraswamy (GKw) regression model (objects of class \code{"gkwreg"}).
Options allow selection of specific plots, choice of residual type, and plotting
using either base R graphics or \code{ggplot2}.
}
\details{
Diagnostic plots are essential for evaluating the assumptions and adequacy of
fitted regression models. This function provides several standard plots adapted
for \code{gkwreg} objects.

The choice of residual type (\code{type}) is important. For models with bounded
responses like the GKw family, quantile residuals (\code{type = "quantile"}) are
generally preferred as they are constructed to be approximately normally distributed
under a correctly specified model, making standard diagnostic tools like QQ-plots
more directly interpretable.

The plots help to assess:
\itemize{
\item Plot 1 (Residuals vs. Index): Potential patterns or autocorrelation over time/index.
\item Plot 2 (Cook's Distance): Observations with disproportionately large influence
on the estimated coefficients.
\item Plot 3 (Leverage vs. Fitted): Observations with unusual predictor combinations
(high leverage) that might influence the fit.
\item Plot 4 (Residuals vs. Linear Predictor): Non-linearity in the predictor-response
relationship or non-constant variance (heteroscedasticity).
\item Plot 5 (Half-Normal Plot): Deviations from the assumed residual distribution
(ideally normal for quantile residuals). Points outside the simulated envelope
are potentially problematic.
\item Plot 6 (Predicted vs. Observed): Overall goodness-of-fit and potential systematic
over- or under-prediction.
}
The function relies on internal helper functions to calculate the necessary diagnostic
quantities and generate the plots using either base R or \code{ggplot2}.
}
\examples{
\donttest{
# Assume 'mydata' exists with response 'y' and predictors 'x1', 'x2'
# and that rgkw() is available and data is appropriate (0 < y < 1).
set.seed(456)
n <- 150
x1 <- runif(n, -1, 1)
x2 <- rnorm(n)
alpha <- exp(0.5 + 0.2 * x1)
beta <- exp(0.8 - 0.3 * x1 + 0.1 * x2)
gamma <- exp(0.6)
delta <- plogis(0.0 + 0.2 * x1)
lambda <- exp(-0.2 + 0.1 * x2)
# Use stats::rbeta as placeholder if rgkw is not available
y <- stats::rbeta(n, shape1 = gamma * alpha, shape2 = delta * beta) # Approximation
y <- pmax(pmin(y, 1 - 1e-7), 1e-7)
mydata <- data.frame(y = y, x1 = x1, x2 = x2)

# Fit a GKw model
model <- gkwreg(y ~ x1 | x1 + x2 | 1 | x1 | x2, data = mydata, family = "gkw")

# --- Generate default base R plots (prompts for each plot) ---
plot(model)

# --- Generate specific plots using base R ---
plot(model, which = c(1, 5), type = "quantile") # Residuals vs Index, Half-Normal

# --- Generate plots using ggplot2 (requires ggplot2 package) ---
# Ensure ggplot2 is installed: install.packages("ggplot2")
plot(model, which = c(4, 6), use_ggplot = TRUE) # Res vs Lin Pred, Pred vs Obs

# --- Generate all ggplot2 plots and arrange them (requires gridExtra or ggpubr) ---
# Ensure gridExtra is installed: install.packages("gridExtra")
# plot(model, use_ggplot = TRUE, arrange_plots = TRUE, ask = FALSE)

# --- Generate plots using Pearson residuals ---
plot(model, which = 4, type = "pearson") # Res vs Lin Pred using Pearson residuals

# --- Save diagnostic measures instead of plotting ---
diagnostics <- plot(model, save_diagnostics = TRUE)
head(diagnostics$residuals)
head(diagnostics$cooks_distance)
}

}
\seealso{
\code{\link{gkwreg}}, \code{\link{residuals.gkwreg}},
\code{\link{summary.gkwreg}}, \code{\link[stats]{plot.lm}},
\code{\link[ggplot2]{ggplot}}, \code{\link[gridExtra]{grid.arrange}},
\code{\link[ggpubr]{ggarrange}}
}
\author{
Lopes, J. E.
}
\keyword{diagnostics}
\keyword{hplot}
\keyword{methods}
\keyword{plot}
\keyword{regression}
