% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwfit.R
\name{confint.gkwfit}
\alias{confint.gkwfit}
\title{Compute Confidence Intervals for gkwfit Parameters}
\usage{
\method{confint}{gkwfit}(object, parm, level = 0.95, ...)
}
\arguments{
\item{object}{An object of class \code{"gkwfit"}, typically the result of a call to \code{\link{gkwfit}}.
The object must contain valid coefficient estimates and a corresponding
variance-covariance matrix (usually requires fitting with \code{hessian = TRUE}).}

\item{parm}{A specification of which parameters are to be given confidence intervals,
either a vector of numbers (indices) or a vector of names. If missing,
confidence intervals are computed for all parameters that have a valid
standard error available. Parameter indices refer to the order of parameters
for which standard errors could be calculated.}

\item{level}{The confidence level required (default: 0.95).}

\item{...}{Additional arguments (currently ignored).}
}
\value{
A matrix with columns giving lower and upper confidence limits for each
parameter specified in \code{parm}. The columns are labeled with quantile percentages
(e.g., \code{"2.5 \%"} and \code{"97.5 \%"} for \code{level = 0.95}). Row names
are taken from the parameter names. Returns \code{NULL} or stops with an error
if coefficients or a valid variance-covariance matrix cannot be extracted.
}
\description{
Computes confidence intervals for one or more parameters in a model fitted by
\code{\link{gkwfit}}. It uses the Wald method based on the estimated coefficients
and their standard errors derived from the variance-covariance matrix.
}
\details{
This function calculates confidence intervals using the Wald method:
\eqn{Estimate \pm z \times SE}, where \eqn{z} is the appropriate quantile
from the standard normal distribution for the given confidence \code{level}.

It relies on the results from \code{\link{coef.gkwfit}} and \code{\link{vcov.gkwfit}}
(or directly accesses \code{object$coefficients} and \code{object$vcov} if those
methods aren't defined). It checks for the validity of the variance-covariance
matrix before proceeding.

Since all parameters of the GKw family distributions are constrained to be positive,
the lower bound of the confidence interval is truncated at a small positive value
(\code{.Machine$double.eps^0.5}) if the calculated lower bound is non-positive.

If \code{parm} is specified, it selects the parameters for which to compute intervals.
Numeric indices in \code{parm} refer to the parameters that have calculable standard
errors, not necessarily all parameters in the model (if some were fixed or had
estimation issues).
}
\examples{
\donttest{
# Generate data and fit model
set.seed(2203)
y <- rkw(50, alpha = 2, beta = 3)
fit <- gkwfit(data = y, family = "kw", plot = FALSE, hessian = TRUE)

# Calculate confidence intervals for all parameters
ci <- confint(fit)
print(ci)

# 90\% confidence interval
ci_90 <- confint(fit, level = 0.90)
print(ci_90)

# Confidence interval for specific parameter
ci_alpha <- confint(fit, parm = "alpha")
print(ci_alpha)
}

}
\seealso{
\code{\link{gkwfit}}, \code{\link[stats]{confint}}, \code{\link{coef.gkwfit}}, \code{\link{vcov.gkwfit}}
}
\author{
Lopes, J. E.
}
\keyword{methods}
\keyword{models}
