% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convenience-functions.R
\name{sim_sar}
\alias{sim_sar}
\title{Simulate spatially autocorrelated data}
\source{
LeSage, J. and Pace, R. K. (2009). \emph{An Introduction to Spatial Econometrics}. CRC Press.
}
\usage{
sim_sar(
  m = 1,
  mu = rep(0, nrow(w)),
  rho,
  sigma = 1,
  w,
  type = c("SEM", "SLM"),
  approx = FALSE,
  K = 20,
  ...
)
}
\arguments{
\item{m}{The number of samples required. Defaults to \code{m=1} to return an \code{n}-length vector; if \code{m>1}, an \code{m x n} matrix is returned (i.e. each row will contain a sample of auto-correlated values).}

\item{mu}{An \code{n}-length vector of mean values. Defaults to a vector of zeros with length equal to \code{nrow(w)}.}

\item{rho}{Spatial autocorrelation parameter in the range (-1, 1). A single numeric value.}

\item{sigma}{Scale parameter (standard deviation). Defaults to \code{sigma = 1}. A single numeric value.}

\item{w}{\code{n x n} spatial weights matrix; typically row-standardized.}

\item{type}{Type of SAR model: spatial error model ("SEM") or spatial lag model ("SLM").}

\item{approx}{Use power of matrix W to approximate the inverse term?}

\item{K}{Number of matrix powers to use if \code{approx = TRUE}.}

\item{...}{further arguments passed to \code{MASS::mvrnorm}.}
}
\value{
If \code{m = 1} then \code{sim_sar} returns a vector of the same length as \code{mu}, otherwise an \code{m x length(mu)} matrix with one sample in each row.
}
\description{
Given a spatial weights matrix and degree of autocorrelation, returns autocorrelated data.
}
\details{
This function takes \code{n = nrow(w)} draws from the normal distribution using \code{rnorm} to obtain vector \code{x}; if \verb{type = 'SEM', it then pre-multiplies }x\verb{by the inverse of the matrix}(I - rho * W)\verb{to obtain spatially autocorrelated values. For}type = 'SLM', the multiplier matrix is applied to \code{x + mu} to produce the desired values.

The \code{approx} method approximates the matrix inversion using the method described by LeSage and Pace (2009, p. 40). For high values of rho, larger values of K are required for the approximation to suffice; you want \code{rho^K} to be near zero.
}
\examples{
# spatially autocorrelated data on a regular grid
library(sf)
row = 10
col = 10
sar_parts <- prep_sar_data2(row = row, col = col)
w <- sar_parts$W
x <- sim_sar(rho = 0.65, w = w)
dat <- data.frame(x = x)

# create grid 
sfc = st_sfc(st_polygon(list(rbind(c(0,0), c(col,0), c(col,row), c(0,0)))))
grid <- st_make_grid(sfc, cellsize = 1, square = TRUE)
st_geometry(dat) <- grid
plot(dat)

# draw form SAR (SEM) model
z <- sim_sar(rho = 0.9, w = w)
moran_plot(z, w)
grid$z <- z

# multiple sets of observations
# each row is one N-length draw from the SAR model
x <- sim_sar(rho = 0.7, w = w, m = 4)
nrow(w)
dim(x)
apply(x, 1, aple, w = w)
apply(x, 1, mc, w = w)

# Spatial lag model (SLM): y = rho*Wy + beta*x + epsilon
x <- sim_sar(rho = 0.5, w = w)
y <- sim_sar(mu = x, rho = 0.7, w = w, type = "SLM")

# Spatial Durbin lag model (SLM with spatial lag of x)
# SDLM: y = rho*Wy + beta*x + gamma*Wx + epsilon
x = sim_sar(w = w, rho = 0.5)
mu <- -0.5*x + 0.5*(w \%*\% x)[,1]
y <- sim_sar(mu = mu, w = w, rho = 0.6, type = "SLM")


}
\seealso{
\code{\link[geostan]{aple}}, \code{\link[geostan]{mc}}, \code{\link[geostan]{moran_plot}}, \code{\link[geostan]{lisa}}, \code{\link[geostan]{shape2mat}}
}
