#' Plot Runs Test Results
#'
#' Generates a residual plot of Sen's estimator applied to annual maximum series
#' data (or the variability of the data) with a horizontal dashed line at 
#' zero and an annotation indicating the p-value of the Runs test.
#'
#' @param results A list of runs test results generated by [eda_runs_test()].
#'
#' @param ... Optional named arguments: 'title', 'xlabel', and 'ylabel'.
#'
#' @return `ggplot`; a plot containing:
#' - Black points for the residual at each year.
#' - A red dashed horizontal line at \eqn{y = 0}.
#' - A text annotation “Runs p-value: X.XXX” in the plot area.
#'
#' @seealso [eda_runs_test()]
#'
#' @examples
#' # Initialize data and years
#' data <- rnorm(n = 100, mean = 100, sd = 10)
#' years <- seq(from = 1901, to = 2000)
#'
#' # Generate the runs test plot 
#' sens_trend <- eda_sens_trend(data, years)
#' results <- eda_runs_test(sens_trend$residuals, years)
#' plot_runs_test(results)
#'
#' @import ggplot2
#' @importFrom grid textGrob grobTree rectGrob gpar
#' @export

plot_runs_test <- function(results, ...) {

	# Generate dataframes for the trend estimate, data, and residuals
	df_residuals <- data.frame(x = results$years, y = results$values)

	# Capture optional arguments
	args <- list(...)

    # Set default values
    title <- args$title %||% "Residual Plot"
    xlabel <- args$xlabel %||% "Year"
    ylabel <- args$ylabel %||% "Residual Value"

	# First subplot: Plot of residuals
	p1 <- ggplot(df_residuals, aes(x = .data$x, y = .data$y)) +
		geom_point(color = "black", size = 2.25) + 
	    geom_hline(yintercept = 0, color = "red", linetype = "dashed", linewidth = 1.2) +
		labs(title = title, x = xlabel, y = ylabel)

	# Add  annotation
	runs_label <- sprintf("Runs p-value: %.3f", results$p_value)
	p1 <- add_annotation(p1, runs_label)

	# Return the plot with added theme
	add_theme(add_scales(p1))

}

 
