% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/easyViz_1.0.0.R
\name{easyViz}
\alias{easyViz}
\title{Easy Visualization of Conditional Effects from Regression Models}
\usage{
easyViz(
  model,
  data,
  predictor,
  by = NULL,
  pred_type = "response",
  pred_range_limit = TRUE,
  pred_on_top = FALSE,
  num_conditioning = "median",
  cat_conditioning = "mode",
  fix_values = NULL,
  backtransform_response = NULL,
  re.form = NULL,
  xlim = NULL,
  ylim = NULL,
  xlab = NULL,
  ylab = NULL,
  font_family = "",
  las = 1,
  bty = "o",
  plot_args = list(),
  show_data_points = TRUE,
  binary_data_type = "plain",
  bins = 10,
  jitter_data_points = FALSE,
  point_col = rgb(0, 0, 0, alpha = 0.4),
  point_pch = 16,
  point_cex = 0.75,
  pred_line_col = "black",
  pred_line_lty = c(1, 2, 3, 4),
  pred_line_lwd = 2,
  ci_type = "polygon",
  ci_polygon_col = c("gray", "black", "lightgray", "darkgray"),
  ci_line_col = "black",
  ci_line_lty = c(1, 2, 3, 4),
  ci_line_lwd = 1,
  pred_point_col = c("black", "gray", "darkgray", "lightgray"),
  pred_point_pch = 16,
  pred_point_cex = 1,
  ci_bar_col = "black",
  ci_bar_lty = 1,
  ci_bar_lwd = 1,
  ci_bar_caps = 0.1,
  cat_labels = NULL,
  add_legend = FALSE,
  legend_position = "top",
  legend_text_size = 0.9,
  legend_labels = NULL
)
}
\arguments{
\item{model}{[required] A fitted model object (e.g., \code{model = your.model}).
Supported models include a wide range of regression types, including linear, robust linear, nonlinear, generalized least squares,
generalized linear, mixed-effects, and generalized additive (mixed) models.
Compatible model-fitting functions include:
\code{stats::lm}, \code{MASS::rlm}, \code{stats::nls}, \code{nlme::gls}, \code{stats::glm}, \code{MASS::glm.nb},
\code{lme4::lmer}, \code{lme4::glmer}, \code{lme4::glmer.nb}, \code{glmmTMB::glmmTMB}, and \code{mgcv::gam}.}

\item{data}{[required] The data frame used to fit the model (e.g., \code{data = your.data}).
This data frame is used internally for generating predictions.
\strong{All variables used in the model formula (including predictors, offsets, grouping variables, and interaction terms) must be present in this data frame}.
If the model was fitted without using a \code{data} argument (e.g., using variables from the global environment),
you must ensure that \code{data} includes all required variables.
Otherwise, prediction may fail or produce incorrect results.}

\item{predictor}{[required] The name of the target explanatory variable to be plotted (e.g., \code{predictor = "x1"}).}

\item{by}{The name of an interaction or additional variable for conditioning (e.g., \code{by = "x2"}). 
If a continuous variable is used, cross-sections are taken at the 10th, 50th, and 90th quantiles.
If a categorical variable is used, a separate line or point will be plotted for each level.
This can also be used to visualize group-level random effects all at once:
namely, when \code{by} corresponds to a grouping variable used in a random effect term (e.g., if \code{by = "group"} when 
the random term is specified as \code{(1|group)} or \code{s(group, bs="re")})
and \code{re.form = NULL}, predictions are conditional on each group's estimated random effect.
Although \code{easyViz} does not natively support direct visualization of three-way interactions in a multi-panel plot,
this can be easily achieved by combining the \code{by} and \code{fix_values} arguments.
For example, if your model includes a term like \code{x1*x2*x3}, you can visualize the effect of \code{x1}
across levels of \code{x2} by setting \code{predictor = "x1"}, \code{by = "x2"}, and fixing \code{x3}
at a specific value using \code{fix_values = c(x3 = ...)}.
Repeating this with different values of \code{x3} produces multiple plots that can be arranged to visualize the full three-way interaction.}

\item{pred_type}{Character string indicating the type of predictions to plot. 
Either \code{"response"} (default), which returns predictions on the original outcome scale 
by applying the inverse of the model's link function (e.g., probabilities for binary models), 
or \code{"link"}, which returns predictions on the linear predictor (link) scale 
(e.g., log-odds, log-counts, or other transformed scales depending on the model).}

\item{pred_range_limit}{Logical. Applies only when the predictor is numeric and a categorical \code{by} variable is specified. 
If \code{TRUE} (default), the prediction range for each level of the \code{by} variable is limited to the range of the \code{predictor} 
observed within that level. This avoids extrapolating predictions beyond the available data for each subgroup. 
If \code{FALSE}, predictions span the entire range of the predictor across all levels of the \code{by} variable.
If the \code{by} variable is numeric, \code{pred_range_limit} is automatically set to \code{FALSE}, 
since numeric \code{by} values are treated as continuous rather than grouping factors.}

\item{pred_on_top}{Logical. If \code{TRUE}, prediction lines (and their confidence intervals) for numeric predictors are drawn after raw data, 
so they appear on top. Default is \code{FALSE}, which draws predictions underneath the data. 
This has no effect for categorical predictors — for those, predictions are always drawn on top of raw data.}

\item{num_conditioning}{How to condition non-target numeric predictors. Either \code{"median"} (default) or \code{"mean"}. 
This determines how numeric variables that are not directly plotted are held constant during prediction, 
while varying the predictor of interest — a common approach when visualizing effects in multivariable models. 
To fix specific variables at custom values instead, use the \code{fix_values} argument.}

\item{cat_conditioning}{How to condition non-target categorical predictors. 
Either \code{"mode"} (default) or \code{"reference"}. As for \code{"num_conditioning"}, 
conditioning means holding these variables constant while varying the predictor of interest.
If multiple levels are equally frequent when \code{"mode"} is selected, the level chosen will be the first in the factor's level order 
(which by default is alphabetical and typically coincides with the reference level, unless explicitly re-leveled).
This behavior also applies to grouping variables used as random effects when \code{re.form = NULL}.
To fix categorical variables (including grouping variables) at specific levels, use \code{fix_values}.}

\item{fix_values}{A named vector or named list specifying fixed values for one or more variables during prediction.
Supports both numeric and categorical variables.
For numeric variables, specify a fixed value (e.g., \code{fix_values = c(x = 1)}).
For categorical variables (factors), provide the desired level as a character string or factor 
(e.g., \code{fix_values = c(group = "levelA")} or \code{fix_values = list(group = levels(data$group)[1])}).
This overrides the default conditioning behavior specified via \code{num_conditioning} and \code{cat_conditioning}.
\strong{Note:} This argument also applies to grouping variables used as random effects: when \code{re.form = NULL}, 
predictions are conditional on the level specified in \code{fix_values}; 
if not specified, the level is chosen based on \code{cat_conditioning}.
This argument is useful for setting offsets, forcing predictions at specific values, or ensuring consistent conditioning across models.
For example, it is particularly useful when you want to visualize the effect of a predictor
at a specific level of an interacting variable, without conditioning on all levels.
E.g., to plot the conditional effect of a continuous predictor \code{x1} 
at a specific value of another variable \code{x2} (numeric or categorical), simply set
\code{fix_values = c(x2 = ...)} and omit the \code{by} argument. 
This creates a clean single-effect plot for \code{x1} at the desired level of \code{x2},
without plotting multiple lines or groups as \code{by} would.
This argument can also be used to visualize three-way interactions when combined with \code{by}.
See the \code{by} argument description for details and an example of how to apply this approach.}

\item{backtransform_response}{A custom function to back-transform predictions for transformed response variables 
(e.g., \code{exp} for log-transformed responses, or \code{function(x) x^2} for square root-transformed responses).
\strong{Note:} If you wish to model a transformed response, it is recommended to apply the transformation 
directly in the model formula (e.g., \code{log(y)}), rather than modifying the response variable in the data set. 
This ensures that observed data points are correctly plotted on the original (back-transformed) scale. 
Otherwise, raw data and predicted values may not align properly in the plot.}

\item{re.form}{A formula specifying which random effects to include when generating predictions.
This argument is relevant for mixed-effects models only (e.g., from \code{lme4}, \code{glmmTMB}, or \code{mgcv::gam}).
Use \code{re.form = NULL} (default) to include group-specific predictions,
conditional on the random-effect levels present in the data.
By default, \code{easyViz} fixes grouping variables at their mode (i.e., the most frequent level), 
so that, when \code{re.form = NULL}, the prediction reflects the conditional estimate for that group level.
However, you can explicitly fix the level of the grouping variable using the \code{fix_values} argument —
this allows you to visualize group-specific predictions for a specific level of the random term (e.g., \code{fix_values = c(group = "levelA")}).
If all levels are equally frequent and no value is specified via \code{fix_values}, the first level (in factor order) is used,
which typically follows alphabetical order unless manually re-leveled.
Use \code{re.form = NA} or \code{re.form = ~0} to obtain population-level predictions based only on fixed effects —
this means that random effects are part of the model fit but are excluded from the prediction, resulting in population-level (i.e., marginal) 
predictions based solely on fixed effects. This is equivalent to assuming the random effects are zero — i.e., an ‘average’ group or subject.
For \code{mgcv::gam()} models, random effects are typically modeled using smooth terms such as \code{s(group, bs = "re")}.
Although \code{predict.gam()} does not support a \code{re.form} argument, \code{easyViz} emulates its behavior:
\code{re.form = NULL} includes random-effect smooths in the prediction, while \code{re.form = NA} or \code{re.form = ~0} excludes them 
by internally using the \code{exclude} argument in \code{predict.gam()}.
For all types of mixed models, when \code{re.form = NULL} and \code{by} corresponds to a grouping variable used in a random effect term,
group-specific (i.e., conditional) predictions are visualized for all levels of the grouping variable.
\strong{Note:} For models fitted with \code{lme4} (e.g., \code{lmer()}, \code{glmer()}), standard errors are not available when \code{re.form = NULL}.}

\item{xlim}{x-axis limits for the plot (e.g., \code{xlim = c(0, 10)}). Defaults to automatic scaling based on the data range.}

\item{ylim}{y axis limits for the plot (e.g., \code{ylim = c(10, 20)}). Defaults to automatic scaling based on the data and prediction range.}

\item{xlab}{x axis labels (e.g., \code{xlab = "x"}). Defaults to \code{"predictor"}.}

\item{ylab}{y axis labels (e.g., \code{ylab = "y"}). Defaults to \code{"response"}.}

\item{font_family}{Font family for the plot. E.g., \code{"sans"} (default), \code{"serif"}, \code{"mono"}.}

\item{las}{Text orientation for axis labels (default: \code{1}).}

\item{bty}{Box type around the plot. E.g., \code{"o"} (default), \code{"n"}, \code{"L"}.}

\item{plot_args}{A named list of additional graphical parameters passed to base R's \code{plot()} function.
These arguments allow users to override default appearance settings in a flexible way.
Common options include axis label size, color, margin settings, font family, and tick mark style.
Common \code{plot()} parameters you may override:
\itemize{
  \item {Label/Text size and style:} \code{cex.lab}, \code{cex.axis}, \code{cex.main}, \code{font.lab}, \code{font.axis}, \code{font.main}
  \item {Colors:} \code{col.lab}, \code{col.axis}, \code{col.main}, \code{col.sub}, \code{col}, \code{bg}, \code{fg}
  \item {Label/Text content:} \code{xlab}, \code{ylab}, \code{main}, \code{sub}
  \item {Margins and layout:} \code{mar}, \code{oma}, \code{mgp}, \code{tcl}, \code{las}, \code{adj}
  \item {Box and axis rendering:} \code{bty}, \code{axes}, \code{frame.plot}, \code{ann}
  \item {Coordinate settings:} \code{xlim}, \code{ylim}, \code{xaxs}, \code{yaxs}, \code{asp}, \code{xlog}, \code{ylog}
}
This is a flexible alternative to manually setting individual plot parameters in the function signature.
For a full list of supported parameters, see \code{?par} and \code{?plot.default}.
Example usage: \cr
\code{plot_args = list(main = "Title", cex.lab = 1.2, col.axis = "gray40", mar = c(5, 4, 4, 2), las = 1)}.}

\item{show_data_points}{Logical. Whether to display raw data points (default: \code{TRUE}).
For binomial models where the response is expressed in the formula as \code{cbind(successes, failures)} or as \code{successes / trials},
the raw data points plotted on the y-axis are based on the calculated proportions:
\code{successes / (successes + failures)} or \code{successes / trials}, respectively.
These proportions are computed internally from the original data and temporarily added to the data set for visualization purposes.}

\item{binary_data_type}{For binary responses, how to display raw data points in the plot. 
Either \code{"plain"} (default), which plots each individual 0/1 observation as-is, 
or \code{"binned"}, which groups observations into intervals (bins) of the predictor and plots 
the proportion of 0s and 1s within each bin. This makes it easier to visualize trends in binary outcomes, 
especially when many points overlap.}

\item{bins}{Number of bins for displaying binary response raw data (default: \code{10}).}

\item{jitter_data_points}{Logical. If \code{TRUE}, raw data points are jittered horizontally 
to reduce overplotting. Applies to both categorical and numeric predictors. 
Default is \code{FALSE}. For categorical predictors, jittering helps distinguish overlapping points. 
For numeric predictors, it can be useful when many data points share the same x-value (e.g., integers or rounding).}

\item{point_col}{Point color for raw data (default: \code{rgb(0,0,0, alpha = 0.4)}). 
Can be specified as a color name (e.g., \code{"gray"}), 
an integer (e.g., \code{1}), or an RGB (e.g., \code{rgb(0,0,0,alpha=0.4)}) or hex string (e.g., \code{"#808080"}). 
Dynamic: accepts multiple values when points are plotted for different values/levels of a variable. \cr
\strong{Tip:} For large data sets with many overlapping data points, 
it is recommended to use semi-transparent colors to reduce overplotting. 
You can achieve this by setting a low alpha value (e.g., \code{rgb(1,0,0, alpha = 0.1}), 
or by using \code{adjustcolor()} with the argument \code{alpha.f} 
(e.g., \code{adjustcolor("red", alpha.f = 0.1)}). 
In such cases, consider setting \code{pred_on_top = TRUE} to ensure that prediction lines 
and confidence intervals remain clearly visible above the dense cloud of raw points.}

\item{point_pch}{Point shape for raw data (default: \code{16}). 
Dynamic: accepts multiple values when points are plotted for different values/levels of a variable.}

\item{point_cex}{Point size for raw data (default: \code{0.75}). 
Dynamic: accepts multiple values when points are plotted for different values/levels of a variable.}

\item{pred_line_col}{Color of the predicted line for numerical predictors (default: \code{"black"}). 
Can be specified as a color name, number or RGB/hex string. 
Dynamic: accepts multiple values (e.g., \code{c("red", "green", "blue")}) when multiple lines are plotted (i.e., when \code{by} is specified).}

\item{pred_line_lty}{Type of the predicted line for numerical predictors (default: \code{1}). 
Dynamic: accepts multiple values (e.g., \code{c(1, 2, 3)}) when multiple lines are plotted (i.e., when \code{by} is specified).}

\item{pred_line_lwd}{Width of the predicted line for numerical predictors (default: \code{2}). 
Dynamic: accepts multiple values (e.g., \code{c(1, 2, 3)}) when multiple lines are plotted (i.e., when \code{by} is specified).}

\item{ci_type}{Type of 95 percent confidence intervals for numerical predictors. \code{"polygon"} (default) or \code{"lines"}.}

\item{ci_polygon_col}{Color for 95 percent confidence interval polygon (default: \code{"gray"}). 
Requires \code{ci_type = "polygon"}. Can be specified as a color name, number or RGB/hex string. 
Dynamic: accepts multiple values (e.g., \code{c("red", "green", "blue")}) 
when 95 percent CIs are plotted for multiple lines (i.e., when \code{by} is specified).
\strong{Tip:} To hide confidence bands entirely, set this to \code{rgb(0,0,0,0)}.}

\item{ci_line_col}{Color for 95 percent confidence interval lines (default: \code{"black"}). 
Requires \code{ci_type = "lines"}. Can be specified as a color name, number or RGB/hex string. 
Dynamic: accepts multiple values (e.g., \code{c("red", "green", "blue")}) 
when 95 percent CIs are plotted for multiple lines (i.e., when \code{by} is specified).}

\item{ci_line_lty}{Type for 95 percent confidence interval lines (default: \code{1}). 
Requires \code{ci_type = "lines"}. Dynamic: accepts multiple values (e.g., \code{c(1, 2, 3)}) 
when 95 percent CIs are plotted for multiple lines (i.e., when \code{by} is specified).}

\item{ci_line_lwd}{Width for 95 percent confidence interval lines (default: \code{1}). 
Requires \code{ci_type = "lines"}. Dynamic: accepts multiple values (e.g., \code{c(1, 2, 3)}) 
when 95 percent CIs are plotted for multiple lines (i.e., when \code{by} is specified).}

\item{pred_point_col}{Color for predicted point values of categorical predictors (default: \code{"black"}). 
Can be specified as a color name, number or RGB/hex string. 
Dynamic: accepts multiple values (e.g., \code{c("red", "green", "blue")}) when points are plotted for an interaction (i.e., when \code{by} is specified).}

\item{pred_point_pch}{Shape for predicted point values of categorical predictors (default: \code{16}). 
Dynamic: accepts multiple values (e.g., \code{c(1, 2, 3)}) when points are plotted for an interaction (i.e., when \code{by} is specified).}

\item{pred_point_cex}{Size for predicted point values of categorical predictors (default: \code{1}). 
Dynamic: accepts multiple values (e.g., \code{c(1, 2, 3)}) when points are plotted for an interaction (i.e., when \code{by} is specified).}

\item{ci_bar_col}{Color for 95 percent confidence interval bars (default: \code{"black"}). 
Applies only when the predictor is categorical. Can be specified as a color name, number, or RGB/hex string.}

\item{ci_bar_lty}{Type for 95 percent confidence interval bars (default: \code{1}). Applies only when the predictor is categorical.}

\item{ci_bar_lwd}{Width for 95 percent confidence interval bars (default: \code{1}). Applies only when the predictor is categorical.}

\item{ci_bar_caps}{Size of the caps on 95 percent confidence interval bars (default: \code{0.1}).
Increase for more visible caps, set to 0 to remove caps and draw plain vertical bars.}

\item{cat_labels}{Custom labels for levels of a categorical predictor (e.g., \code{cat_labels = c("Level A", "Level B", "Level C")}).}

\item{add_legend}{Logical. Whether to add a legend for \code{by} variable levels (default: \code{FALSE}).}

\item{legend_position}{Legend position. Either a named position string (\code{"top"}, \code{"bottom"}, \code{"left"}, 
\code{"right"}, \code{"topleft"}, \code{"topright"}, \code{"bottomleft"}, \code{"bottomright"}) or a numeric vector \code{c(x, y)} 
specifying exact coordinates for the legend placement.}

\item{legend_text_size}{Legend text size (default: \code{0.9}).}

\item{legend_labels}{Custom labels for the legend (e.g., \code{legend_labels = c("Label 1", "Label 2", "Label 3")}).}
}
\value{
A base R plot visualizing the conditional effect of a predictor on the response variable.
Additionally, a data frame is invisibly returned containing the predictor values, conditioning variables, 
predicted values (\code{fit}), and their 95 percent confidence intervals (\code{lower}, \code{upper}).
To extract prediction data for further use (e.g., custom plotting or tabulation), assign the output to an object:
\code{pred_df <- easyViz(...)}. You can then inspect it using \code{head(pred_df)} or save it with \code{write.csv(pred_df, ...)}.
}
\description{
\code{easyViz} offers a flexible and user-friendly interface for visualizing conditional effects 
from a broad range of regression and mixed-effects models using base R graphics.
}
\details{
This function provides an easy-to-use yet highly flexible tool for visualizing conditional effects 
from a wide range of regression models, including mixed-effects and generalized additive (mixed) models. 
Compatible model types include \code{lm}, \code{rlm}, \code{glm}, \code{glm.nb}, and \code{mgcv::gam}; 
nonlinear models via \code{nls}; and generalized least squares via \code{gls}. 
Mixed-effects models with random intercepts and/or slopes can be fitted using \code{lmer}, \code{glmer}, \code{glmer.nb}, 
\code{glmmTMB}, or \code{mgcv::gam} (via smooth terms). 
The function handles nonlinear relationships (e.g., splines, polynomials), two-way interactions, 
and supports visualization of three-way interactions via conditional plots.
Plots are rendered using base R graphics with extensive customization options available through the \code{plot_args} argument.
Users can pass any valid graphical parameters accepted by \code{plot} or \code{par},
enabling full control over axis labels, font styles, colors, margins, tick orientation, and more.
The arguments \code{model}, \code{data}, and \code{predictor} are required.
The function will return an error if any of them is missing or invalid.
}
\examples{
#------------------------------------------
# Load required packages
#------------------------------------------

library(nlme)
library(MASS)
library(lme4)
library(glmmTMB)
library(mgcv)

#------------------------------------------
# Simulate dataset
#------------------------------------------
set.seed(123)
n <- 100
x1 <- rnorm(n)
x2 <- rnorm(n)
x3 <- runif(n, 0, 5)
x4 <- factor(sample(letters[1:3], n, replace = TRUE))
group_levels <- paste0("G", 1:10)
group <- factor(sample(group_levels, n, replace = TRUE))

# Generate random intercepts for each group
group_effects <- rnorm(length(group_levels), mean = 0, sd = 2)  # non-zero variance
names(group_effects) <- group_levels
group_intercept <- group_effects[as.character(group)]

# Non-linear continuous response
true_y <- 5 * sin(x3) + 3 * x1 + group_intercept + model.matrix(~x4)[, -1] \%*\% c(2, -2)
noise <- rnorm(n, sd = 3)
y <- as.vector(true_y + noise)

# Binary response with group effect added to logit
logit_p <- 2 * x1 - 1 + group_intercept
p <- 1 / (1 + exp(-logit_p))
binary_y <- rbinom(n, size = 1, prob = p)

# Binomial response: number of successes and failures
y3 <- sample(10:30, n, replace = TRUE)
logit_p_prop <- -1.5 * scale(x1)  
p_prop <- 1 / (1 + exp(-logit_p_prop))
y1 <- rbinom(n, size = y3, prob = p_prop) # successes
y2 <- y3 - y1  # failures

# Count response with group effect in log(mu)
mu_count <- exp(1 + 0.8 * x2 - 0.5 * (x4 == "b") + group_intercept)
size <- 1.2
count_y <- rnbinom(n, size = size, mu = mu_count)
# Offset variable
offset_var <- log(runif(n, 1, 10))

# Assemble dataset
sim_data <- data.frame(x1, x2, x3, x4, group, y, binary_y, y1, y2, y3, count_y, offset_var)

#------------------------------------------
# 1. Linear model (lm)
#------------------------------------------
mod_lm <- lm(y ~ x1 + x4, 
             data = sim_data)
easyViz(model = mod_lm, data = sim_data, predictor = "x1", 
        by = "x4",
        pred_range_limit = FALSE,
        pred_on_top = TRUE,
        bty = "n",
        xlab = "Predictor x1", 
        ylab = "Response y",
        point_col = ifelse(sim_data$x4=="a", "red", 
                           ifelse(sim_data$x4=="b", "orange", 
                                  "yellow")),
        point_cex = 0.5,
        pred_line_col = c("red", "orange", "yellow"),
        pred_line_lty = 1,
        ci_polygon_col = c(rgb(1,0,0,0.5), 
                           rgb(1,0.5,0,0.5), 
                           rgb(1,1,0,0.5)),
        add_legend = TRUE, 
        legend_position = "topleft",
        legend_labels = c("a", "b", "c"))

# Extract prediction data
pred_df <- easyViz(model = mod_lm, data = sim_data, predictor = "x1", by = "x4")
head(pred_df)

mod_lm2 <- lm(sqrt(x3) ~ x1 * x4, 
              data = sim_data)
easyViz(model = mod_lm2, data = sim_data, predictor = "x1", 
        by="x4",
        backtransform_response = function(x) x^2,
        ylim = c(0,8),
        show_data_points = FALSE,
        add_legend = TRUE)

mod_lm3 <- lm(y ~ poly(x3, 3), 
              data = sim_data)
easyViz(model = mod_lm3, data = sim_data, predictor = "x3", 
        pred_on_top = TRUE,
        font_family = "mono",
        point_col = rgb(1,0,0,0.3),
        point_pch = "+",
        ci_type = "lines",
        ci_line_lty = 2)

#------------------------------------------
# 2. Robust linear model (rlm)
#------------------------------------------
mod_rlm <- rlm(y ~ x1 + x4, 
               data = sim_data)
easyViz(model = mod_rlm, data = sim_data, predictor = "x1", 
        by = "x4",
        pred_on_top = TRUE,
        bty = "n",
        xlab = "Predictor x1", 
        ylab = "Response y",
        point_col = ifelse(sim_data$x4=="a", "red", 
                           ifelse(sim_data$x4=="b", "orange", 
                                  "yellow")),
        point_cex = 0.5,
        pred_line_col = c("red", "orange", "yellow"),
        pred_line_lty = 1,
        ci_polygon_col = c(rgb(1,0,0,0.5), 
                           rgb(1,0.5,0,0.5), 
                           rgb(1,1,0,0.5)),
        add_legend = TRUE, 
        legend_position = c(1.25,-1),
        legend_labels = c("a", "b", "c"))

#------------------------------------------
# 3. Generalized least squares (gls)
#------------------------------------------
mod_gls <- gls(y ~ x1 + x2 + x4, 
               correlation = corAR1(form = ~1|group), 
               data = sim_data)
easyViz(model = mod_gls, data = sim_data, predictor = "x4",
        jitter_data_points = TRUE,
        bty = "n",
        xlab = "Predictor x4", 
        ylab = "Response y",
        point_col = rgb(0,0,1,0.2),
        pred_point_col = "blue",
        cat_labels = c("group A", "group B", "group C"))

sim_data$x5 <- sample(c(rep("CatA", 50), rep("CatB", 50)))
mod_gls2 <- gls(y ~ x1 + x2 + x4 * x5, 
                correlation = corAR1(form = ~1|group), 
                data = sim_data)
easyViz(model = mod_gls2, data = sim_data, predictor = "x4",
        by = "x5",
        jitter_data_points = TRUE,
        bty = "n",
        ylim = c(-15,20),
        xlab = "Predictor x4", 
        ylab = "Response y",
        point_col = c(rgb(0,0,1,0.2), rgb(1,0,0,0.2)),
        pred_point_col = c("blue", "red"),
        ci_bar_caps = 0,
        cat_labels = c("group A", "group B", "group C"),
        add_legend = TRUE,
        legend_position = "topright",
        legend_labels = c("Category A", "Category B"))

#------------------------------------------
# 4. Nonlinear least squares (nls)
#------------------------------------------
mod_nls <- nls(y ~ a * sin(b * x3) + c,
               data = sim_data,
               start = list(a = 5, b = 1, c = 0))
summary(mod_nls)
easyViz(model = mod_nls, data = sim_data, predictor = "x3",
        pred_on_top = TRUE,
        font_family = "serif",
        bty = "n",
        xlab = "Predictor x3", 
        ylab = "Response y",
        point_col = rgb(0,1,0,0.7),
        point_pch = 1,
        ci_type = "lines",
        ci_line_col = "black",
        ci_line_lty = 2)
text(x = 2.5, y = 11,
     labels = expression(Y \%~\% 5.31584 \%*\% sin(1.08158 \%*\% X[3]) + 0.51338),
     cex = 0.7)

#------------------------------------------
# 5. Generalized linear model (glm)
#------------------------------------------
mod_glm <- glm(binary_y ~ x1 + x4 + offset(log(offset_var)), 
               family = binomial(link="cloglog"),
               data = sim_data)
easyViz(model = mod_glm, data = sim_data, predictor = "x1",
        fix_values = list(x4="b", offset_var=1),
        xlab = "Predictor x1", 
        ylab = "Response y",
        binary_data_type = "binned",
        point_col = "black",
        ci_polygon_col = "red")

easyViz(model = mod_glm, data = sim_data, predictor = "x4",
        bty = "n",
        xlab = "Predictor x4", 
        ylab = "Response y",
        binary_data_type = "plain",
        jitter_data_points = TRUE,
        point_col = "black",
        point_pch = "|",
        point_cex = 0.5)

mod_glm2 <- glm(y1/y3 ~ x1 + x4, weights = y3, 
                family = binomial(link="logit"), 
                data = sim_data)
easyViz(model = mod_glm2, data = sim_data, predictor = "x1",
        pred_on_top = TRUE,
        xlab = "Predictor x1", 
        ylab = "Response y",
        point_col = "black",
        ci_polygon_col = "red")

#------------------------------------------
# 6. Negative binomial GLM (glm.nb)
#------------------------------------------
mod_glm_nb <- glm.nb(count_y ~ x2, 
                     data = sim_data)
easyViz(model = mod_glm_nb, data = sim_data, predictor = "x2",
        font_family = "mono",
        bty = "L",
        plot_args = list(main = "NB model"),
        xlab = "Predictor x2", 
        ylab = "Response y",
        ci_polygon_col = "blue")

#------------------------------------------
# 7. Linear mixed-effects model (lmer)
#------------------------------------------
mod_lmer <- lmer(y ~ x1 + x4 + (1 | group), 
                 data = sim_data)
easyViz(model = mod_lmer, data = sim_data, predictor = "x1", 
        by="group",
        re.form = NULL,
        bty = "n",
        plot_args = list(xaxp = c(round(min(sim_data$x1),1),
                                  round(max(sim_data$x1),1), 5)),
        ylim = c(-15, 15),
        xlab = "Predictor x1", 
        ylab = "Response y",
        pred_line_col = "green",
        pred_line_lty = 1,
        pred_line_lwd = 1)
oldpar <- par(new = TRUE)
easyViz(model = mod_lmer, data = sim_data, predictor = "x1",
        re.form = NA,
        bty = "n",
        plot_args = list(xaxp = c(round(min(sim_data$x1),1),
                                  round(max(sim_data$x1),1), 5)),
        show_data_points = FALSE,
        xlab = "Predictor x1", 
        ylab = "Response y",
        ylim = c(-15, 15),
        pred_line_col = "red",
        pred_line_lty = 1,
        pred_line_lwd = 2,
        ci_polygon = rgb(0,0,0,0))
par(oldpar)

#------------------------------------------
# 8. Generalized linear mixed model (glmer)
#------------------------------------------
mod_glmer <- glmer(binary_y ~ x1 + x4 + (1 | group), 
                   family = binomial,
                   data = sim_data)
easyViz(model = mod_glmer, data = sim_data, predictor = "x1", 
        by = "group",
        re.form = NULL,
        cat_conditioning = "reference",
        font_family = "serif",
        xlab = "Predictor x1", 
        ylab = "Response y",
        binary_data_type = "binned",
        pred_range_limit = FALSE,
        pred_line_col = "blue",
        pred_line_lty = 1,
        pred_line_lwd = 1)

#------------------------------------------
# 9. GLMM with negative binomial (glmer.nb)
#------------------------------------------
mod_glmer_nb <- glmer.nb(count_y ~ x2 + x4 + (1 | group), 
                         data = sim_data)
easyViz(model = mod_glmer_nb, data = sim_data, predictor = "x2",
        re.form = NA,
        bty = "n",
        xlab = "Predictor x2", 
        ylab = "Response y",
        ylim = c(0, 120),
        point_pch = 1)

#------------------------------------------
# 10. GLMM using glmmTMB
#------------------------------------------
mod_glmmTMB <- glmmTMB(count_y ~ x2 + x4 + (1 | group),
                       ziformula = ~ x2, 
                       family = nbinom2,
                       data = sim_data)
easyViz(model = mod_glmmTMB, data = sim_data, predictor = "x2",
        re.form = NA,
        bty = "n",
        xlab = "Predictor x2", 
        ylab = "Response y",
        ylim = c(0, 120),
        point_pch = 1)

#------------------------------------------
# 11. GAM with random smooth for group
#------------------------------------------
mod_gam <- gam(y ~ s(x3) + s(group, bs = "re"),
               data = sim_data)
easyViz(model = mod_gam, data = sim_data, predictor = "x3",
        re.form = NA,
        las = 0,
        bty = "n",
        xlab = "Predictor x3", 
        ylab = "Response y",
        point_col = "black",
        point_pch = 1,
        ci_polygon_col = rgb(1,0,0,0.5))

}
