#' Add a Marker Axis with Chromosome Names to a Plot of Polarized Genotypes
#'
#' This function adds a marker axis with chromosome names to an existing plot of polarized
#' genotypes. It requires that the plot is already created using \link{plotPolarized}.
#'
#' @param includedSites A character path to a file with columns \code{CHROM} and \code{POS}.
#' @inheritParams diem
#' @param tickDist A numeric indicating the spacing of physical tick marks along a chromosome.
#' @param axisInfo A list with user-defined tick positions and labels for marker axis. See
#'   Details.
#' @details Either \code{axisInfo} or \code{includedSites} must be provided. If 
#' \code{axisInfo = NULL}, the function extracts the necessary information from the 
#' \code{includedSites} file. The \code{includedSites} file should ideally be generated by
#' \link{vcf2diem} to ensure congruence between the plotted genotypes and
#' the respective metadata.
#'
#' Tick mark distances within a chromosome are spaced at intervals of \code{tickDist} 
#' (in bp) and formatted to multiples of millions.
#'
#' The positions in \code{axisInfo} (e.g., \code{ticksPos}, \code{CHROMbreaks}, 
#' \code{CHROMnamesPos}) are in units of marker index (i.e. column positions in the 
#' genotype matrix used in the previous \code{plotPolarized} call).
#'
#' The optional \code{axisInfo} argument must be a list with five named elements:
#' * \code{CHROMbreaks}: Numeric vector of positions defining ticks that separate
#'   chromosomes. 
#' * \code{CHROMnamesPos}: Numeric vector of label positions for chromosome names.
#' * \code{CHROMnames}: Character vector of chromosome names (same length as 
#'   \code{CHROMnamesPos}).
#' * \code{ticksPos}: Numeric vector with tick positions within chromosomes.
#' * \code{ticksNames}: Character vector of labels corresponding to \code{ticksPos}.
#'   Potentially overlapping labels are automatically replaced with \code{""}.
#'
#'
#' The \code{...} arguments accept additional graphical parameters passed to either 
#' \code{axis()}, \code{mtext()}, or \code{circos.*()} functions, depending on plot type. 
#' The following arguments are supported: 
#' 
#' \strong{Rectangular axis arguments:} \code{side}, \code{col.ticks},
#' \code{labels}, \code{las}, \code{tick}, \code{line}, \code{tcl}, \code{cex},
#' \code{cex.axis}, \code{pos}, \code{outer}, \code{font}, \code{lty}, \code{lwd},
#' \code{lwd.ticks}, \code{hadj}, \code{padj}, \code{gap.axis}, \code{xpd}.
#'
#' \strong{Circular axis arguments (used in iris/circular plots):}
#' \code{major.tick.length}, \code{lwd}, \code{labels.cex}, \code{cex}, \code{niceFacing},
#' \code{labels.niceFacing}, \code{facing}, \code{labels.facing}, \code{track.height}, 
#' \code{adj}.
#'
#' In rectangular plots, chromosome names are drawn using \code{mtext()}, with size 
#' scaled by \code{cex * 1.1}. Tick labels for positions use the same \code{cex}.
#'
#' In circular plots (types \code{"iris"} or \code{"circular"}), tick positions and 
#' chromosome names  are drawn using \code{circos.axis()} and \code{circos.text()}, 
#' respectively. Their sizes are controlled separately:
#' * \code{labels.cex} controls the size of tick position numbers,
#' * \code{cex} controls the size of chromosome name labels.
#'
#' Similarly, text facing is controlled independently:
#' * \code{labels.niceFacing} affects the tick labels,
#' * \code{niceFacing} affects chromosome names.
#'
#' The default alignment for chromosome names is set by \code{adj = c(1, 0.5)}, meaning 
#' right-aligned and vertically centered. 
#'
#' To optimize the marker axis format in iris plots, it is recommended to test with only
#' one sample plotted to save processing time.
#'
#' @return Returns an invisible \code{axisInfo} list with the tick positions and labels
#' for the marker axis.
#' @seealso 
#' \code{\link{plotPolarized}} to generate the base plot.
#' @importFrom utils modifyList
#' @importFrom grDevices dev.cur
#' @importFrom graphics mtext
#' @import circlize
#' @export
#' @examples
#' \dontrun{
#' # Run this example in a working directory with write permissions
#' myo <- system.file("extdata", "myotis.vcf", package = "diemr")
#' vcf2diem(myo, "myo")
#' inds <- 1:14
#' fit <- diem("myo-001.txt", ChosenInds = inds, ploidy = FALSE)
#' gen <- importPolarized("myo-001.txt", fit$markerPolarity, inds)
#' h <- apply(gen, 1, function(x) pHetErrOnStateCount(sStateCount(x)))[1, ]
#' plotPolarized(gen, h, xlab = "")
#' plotMarkerAxis("myo-includedSites.txt", tickDist = 100)
#'
#' plotPolarized(gen, h, type = "iris")
#' plotMarkerAxis("myo-includedSites.txt", tickDist = 100)
#' }
plotMarkerAxis <- function(includedSites, ChosenSites = "all", tickDist = 1e+06, axisInfo = NULL, ...) {
  if (dev.cur() == 1) {
    stop("Plot polarized genotypes with plotPolarized first.")
  }
  if (is.null(axisInfo)) {
    axisInfo <- markerAxis(
      includedSites = includedSites,
      ChosenSites = ChosenSites, tickDist = tickDist
    )
  }
  if (any(missing <- !(req <- c("ticksPos", "ticksNames", "CHROMbreaks", "CHROMnamesPos", "CHROMnames")) %in% names(axisInfo))) {
    stop("The 'axisInfo' is missing: ", paste(req[missing], collapse = ", "))
  }


  userArgs <- list(...)
  plottingArgs <- utils::modifyList(list(
    # axis defaults
    side = 1,
    las = 1,
    tcl = -.5,
    cex = 1,
    line = 0,
    # circos.axis defaults
    major.tick.length = 10,
    lwd = 2, 
    labels.cex = .8,
    niceFacing = TRUE,
    labels.niceFacing = TRUE,
    facing = "clockwise",
    labels.facing = "clockwise",
    track.height = .2,
    adj = c(1, .5)
  ), userArgs)


  acceptedAxisArgs <- c(
    "side", "col.ticks", "labels", "las", "tick", "line", "tcl", "cex", "cex.axis",
    "pos", "outer", "font", "lty", "lwd", "lwd.ticks", "hadj", "padj", "gap.axis",
    "xpd"
  )
  acceptedCirclizeArgs <- c(
    "major.tick.length", "lwd", "cex", "labels.cex", "niceFacing", "labels.niceFacing",
    "facing", "labels.facing", "track.height", "adj"
  )
  
  axisArgs <- plottingArgs[names(plottingArgs) %in% acceptedAxisArgs]
  circlizeArgs <- plottingArgs[names(plottingArgs) %in% acceptedCirclizeArgs]



  # check how to draw axis
  isIris <- !any(grepl("No track", capture.output(circlize::circos.info(plot = FALSE))))

  if (!isIris) {
    # ticks of physical distances
    do.call(axis, utils::modifyList(
      axisArgs,
      list(at = axisInfo$ticksPos, labels = axisInfo$ticksNames, cex = axisArgs$cex)
    ))
    # ticks for chromosome separators
    do.call(axis, utils::modifyList(
      axisArgs,
      list(at = axisInfo$CHROMbreaks, tcl = axisArgs$tcl * 2.2, labels = FALSE)
    ))
    # chromosome names
    do.call(mtext, utils::modifyList(
      axisArgs,
      list(
        at = axisInfo$CHROMnamesPos,
        text = axisInfo$CHROMnames,
        line = axisArgs$line + 2.5,
        cex = axisArgs$cex * 1.1
      )
    ))
  } else {
    # marker axis for iris
    # CHROMbreaks
    circos.axis(
      h = "top", track.index = 1,
      major.at = axisInfo$CHROMbreaks - .5,
      labels = FALSE,
      minor.ticks = 0,
      major.tick.length = convert_y(circlizeArgs$major.tick.length, "mm"),
      lwd = circlizeArgs$lwd
    )

    # minor ticks
    circos.axis(
      h = "top", track.index = 1,
      major.at = axisInfo$ticksPos - .5,
      labels = axisInfo$ticksNames,
      minor.ticks = 0,
      major.tick.length = convert_y(.3 * circlizeArgs$major.tick.length, "mm"),
      lwd = .5 * circlizeArgs$lwd,
      labels.cex = circlizeArgs$labels.cex,
      labels.niceFacing = circlizeArgs$labels.niceFacing,
      labels.facing = circlizeArgs$labels.facing,
    )

    # CHROMbreaks shading
    circos.track(
      ylim = c(0, 1),
      panel.fun = function(x, y) {
        for (i in seq_along(axisInfo$CHROMnames)) {
          circos.rect(
            xleft = axisInfo$CHROMbreaks[i] - .5,
            ybottom = 0,
            xright = axisInfo$CHROMbreaks[i + 1] - .5,
            ytop = 1,
            col = c("white", "grey")[i %% 2 + 1],
            border = NA
          )
          circos.text(axisInfo$CHROMnamesPos[i] - .5,
            y = .9,
            labels = axisInfo$CHROMnames[i],
            facing = circlizeArgs$facing,
            niceFacing = circlizeArgs$niceFacing,
            adj = circlizeArgs$adj,
            cex = circlizeArgs$cex
          )
        }
      }, track.height = circlizeArgs$track.height, bg.border = NA
    )
  }
  return(invisible(axisInfo))
}
