% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{prune}
\alias{prune}
\title{Static pruning of a DGP emulator}
\usage{
prune(object, control = list(), verb = TRUE)
}
\arguments{
\item{object}{an instance of the \code{dgp} class that is generated by \code{dgp()}.}

\item{control}{a list that can supply the following two components to control static pruning of the DGP emulator:
\itemize{
\item \code{min_size}, the minimum number of design points required to trigger pruning. Defaults to 10 times of the input dimensions.
\item \code{threshold}, the \eqn{R^2} value above which a GP node is considered redundant and removable. Defaults to \code{0.97}.
}}

\item{verb}{a bool indicating if trace information will be printed during the function execution. Defaults to \code{TRUE}.}
}
\value{
An updated \code{object} that could be an instance of \code{gp}, \code{dgp}, or \code{bundle} (of GP emulators) class.
}
\description{
This function implements static pruning for a DGP emulator.
}
\details{
See further examples and tutorials at \url{https://mingdeyu.github.io/dgpsi-R/}.
}
\note{
\itemize{
\item The function requires a DGP emulator that has been trained with a dataset comprising a minimum size equal to \code{min_size} in \code{control}.
If the training dataset size is smaller than this, it is recommended that the design of the DGP emulator is enriched and its
structure pruned dynamically using the \code{design()} function. Depending on the design of the DGP emulator, static pruning may not be accurate.
It is thus recommended that dynamic pruning is implemented as a part of a sequential design via \code{design()}.
\item The following slots:
\itemize{
\item \code{loo} and \code{oos} created by \code{\link[=validate]{validate()}}; and
\item \code{results} created by \code{\link[=predict]{predict()}};
}

in \code{object} will be removed and not contained in the returned object.
}
}
\examples{
\dontrun{

# load the package and the Python env
library(dgpsi)

# construct the borehole function over a hypercube
f <- function(x){
  x[,1] <- (0.15 - 0.5) * x[,1] + 0.5
  x[,2] <- exp((log(50000) - log(100)) * x[,2] + log(100))
  x[,3] <- (115600 - 63070) *x[,3] + 63070
  x[,4] <- (1110 - 990) * x[,4] + 990
  x[,5] <- (116 - 63.1) * x[,5] + 63.1
  x[,6] <- (820 - 700) * x[,6] + 700
  x[,7] <- (1680 - 1120) * x[,7] + 1120
  x[,8] <- (12045 - 9855) * x[,8] + 9855
  y <- apply(x, 1, RobustGaSP::borehole)
}

# set a random seed
set_seed(999)

# generate training data
X <- maximinLHS(80, 8)
Y <- f(X)

# generate validation data
validate_x <- maximinLHS(500, 8)
validate_y <- f(validate_x)

# training a DGP emulator with anisotropic squared exponential kernels
m <- dgp(X, Y, share = F)

# OOS validation of the DGP emulator
plot(m, validate_x, validate_y)

# prune the emulator until no more GP nodes are removable
m <- prune(m)

# OOS validation of the resulting emulator
plot(m, validate_x, validate_y)
}
}
