#' Estimate Standardized Regression Coefficients
#' and Generate the Corresponding Sampling Distribution
#' Using the Monte Carlo Method
#'
#' @details The vector of standardized regression coefficients
#'   (\eqn{\boldsymbol{\hat{\beta}}})
#'   is derived from each randomly generated vector of parameter estimates.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{\boldsymbol{\hat{\beta}}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @return Returns an object
#'   of class `betamc` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{args}{Function arguments.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{\boldsymbol{\hat{\beta}}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{\boldsymbol{\hat{\beta}}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{\boldsymbol{\hat{\beta}}}.}
#'     \item{fun}{Function used ("BetaMC").}
#'   }
#'
#' @param object Object of class `mc`, that is,
#'   the output of the `MC()` function.
#' @param alpha Numeric vector.
#'   Significance level \eqn{\alpha}.
#'
#' @examples
#' # Data ---------------------------------------------------------------------
#' data("nas1982", package = "betaMC")
#'
#' # Fit Model in lm ----------------------------------------------------------
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#'
#' # MC -----------------------------------------------------------------------
#' mc <- MC(
#'   object,
#'   R = 100, # use a large value e.g., 20000L for actual research
#'   seed = 0508
#' )
#'
#' # BetaMC -------------------------------------------------------------------
#' out <- BetaMC(mc, alpha = 0.05)
#'
#' ## Methods -----------------------------------------------------------------
#' print(out)
#' summary(out)
#' coef(out)
#' vcov(out)
#' confint(out, level = 0.95)
#'
#' @family Beta Monte Carlo Functions
#' @keywords betaMC std
#' @export
BetaMC <- function(object,
                   alpha = c(0.05, 0.01, 0.001)) {
  stopifnot(
    inherits(
      object,
      "mc"
    )
  )
  if (object$fun == "MCMI") {
    est <- colMeans(
      do.call(
        what = "rbind",
        args = lapply(
          X = object$args$mi_output$lm_process,
          FUN = function(x) {
            return(
              x$betastar
            )
          }
        )
      )
    )
  } else {
    est <- object$lm_process$betastar
  }
  out <- list(
    call = match.call(),
    args = list(
      object = object,
      alpha = alpha
    ),
    thetahatstar = lapply(
      X = object$thetahatstar,
      FUN = function(x) {
        return(
          .BetaStar(
            beta = x$coef,
            sigmay = sqrt(
              x$sigmaysq
            ),
            sigmax = sqrt(
              diag(x$sigmacapx)
            )
          )
        )
      }
    ),
    est = est,
    fun = "BetaMC"
  )
  class(out) <- c(
    "betamc",
    class(out)
  )
  return(
    out
  )
}
