% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/disparity.R
\name{disparity}
\alias{disparity}
\title{Extract backbone using the Disparity Filter}
\usage{
disparity(
  W,
  alpha = 0.05,
  missing.as.zero = FALSE,
  signed = FALSE,
  mtc = "none",
  class = "original",
  narrative = FALSE
)
}
\arguments{
\item{W}{A positively-weighted unipartite graph, as: (1) an adjacency matrix in the form of a matrix or sparse \code{\link{Matrix}}; (2) an edgelist in the form of a three-column dataframe; (3) an \code{\link{igraph}} object.}

\item{alpha}{real: significance level of hypothesis test(s)}

\item{missing.as.zero}{boolean: should missing edges be treated as edges with zero weight and tested for significance}

\item{signed}{boolean: TRUE for a signed backbone, FALSE for a binary backbone (see details)}

\item{mtc}{string: type of Multiple Test Correction to be applied; can be any method allowed by \code{\link{p.adjust}}.}

\item{class}{string: the class of the returned backbone graph, one of c("original", "matrix", "Matrix", "igraph", "edgelist").
If "original", the backbone graph returned is of the same class as \code{W}.}

\item{narrative}{boolean: TRUE if suggested text & citations should be displayed.}
}
\value{
If \code{alpha} != NULL: Binary or signed backbone graph of class \code{class}.

If \code{alpha} == NULL: An S3 backbone object containing (1) the weighted graph as a matrix, (2) upper-tail p-values as a
matrix, (3, if \code{signed = TRUE}) lower-tail p-values as a matrix, (4, if present) node attributes as a dataframe, and
(5) several properties of the original graph and backbone model, from which a backbone can subsequently be extracted
using \code{\link[=backbone.extract]{backbone.extract()}}.
}
\description{
\code{disparity} extracts the backbone of a weighted network using the Disparity Filter.
}
\details{
The \code{disparity} function applies the disparity filter (Serrano et al., 2009), which compares an edge's weight to
its expected weight if a node's total degree was uniformly distributed across all its edges. The graph may be
directed or undirected, however the edge weights must be positive.

When \code{signed = FALSE}, a one-tailed test (is the weight stronger?) is performed for each edge. The resulting backbone
contains edges whose weights are significantly \emph{stronger} than expected in the null model. When \code{signed = TRUE}, a
two-tailed test (is the weight stronger or weaker?) is performed for each edge. The resulting backbone contains
positive edges for those whose weights are significantly \emph{stronger}, and negative edges for those whose weights are
significantly \emph{weaker}, than expected in the null model.

If \code{W} is an unweighted bipartite graph, then the disparity filter is applied to its weighted bipartite projection.
}
\examples{
#A network with heterogeneous (i.e. multiscale) weights
net <- matrix(c(0,10,10,10,10,75,0,0,0,0,
                10,0,1,1,1,0,0,0,0,0,
                10,1,0,1,1,0,0,0,0,0,
                10,1,1,0,1,0,0,0,0,0,
                10,1,1,1,0,0,0,0,0,0,
                75,0,0,0,0,0,100,100,100,100,
                0,0,0,0,0,100,0,10,10,10,
                0,0,0,0,0,100,10,0,10,10,
                0,0,0,0,0,100,10,10,0,10,
                0,0,0,0,0,100,10,10,10,0),10)

net <- igraph::graph_from_adjacency_matrix(net, mode = "undirected", weighted = TRUE)
plot(net, edge.width = sqrt(igraph::E(net)$weight)) #A stronger clique & a weaker clique

strong <- igraph::delete_edges(net, which(igraph::E(net)$weight < mean(igraph::E(net)$weight)))
plot(strong) #A backbone of stronger-than-average edges ignores the weaker clique

bb <- disparity(net, alpha = 0.05, narrative = TRUE) #A disparity backbone...
plot(bb) #...preserves edges at multiple scales
}
\references{
package: {Neal, Z. P. (2022). backbone: An R Package to Extract Network Backbones. \emph{PLOS ONE, 17}, e0269137. \doi{10.1371/journal.pone.0269137}}

disparity filter: {Serrano, M. A., Boguna, M., & Vespignani, A. (2009). Extracting the multiscale backbone of complex weighted networks. \emph{Proceedings of the National Academy of Sciences, 106}, 6483-6488. \doi{10.1073/pnas.0808904106}}
}
