\name{sarima.sim}
\alias{sarima.sim}
\title{ARIMA Simulation
} 
\description{Simulate data from (seasonal) ARIMA models.
}
\usage{
sarima.sim(ar = NULL, d = 0, ma = NULL, sar = NULL, D = 0, sma = NULL, S = NULL, 
            n = 500, rand.gen = rnorm, innov = NULL, burnin = NA, t0 = 0, ...)
}
\arguments{
  \item{ar}{
coefficients of AR component (does not have to be specified)
}
  \item{d}{
order of regular difference (does not have to be specified)
}
  \item{ma}{
coefficients of MA component (does not have to be specified)
}
  \item{sar}{
coefficients of SAR component (does not have to be specified)
}
  \item{D}{
order of seasonal difference (does not have to be specified)
}
  \item{sma}{
coefficients of SMA component (does not have to be specified)
}
  \item{S}{
seasonal period (does not have to be specified)
}
  \item{n}{
desired sample size (defaults to 500)
}
  \item{rand.gen}{
optional; a function to generate the innovations (defaults to normal)
}
  \item{innov}{	
an optional times series of innovations. If not provided, rand.gen is used.
}
  \item{burnin}{
length of burn-in (a non-negative integer). If \code{NA}, or not a positive
integer, or a vector, a reasonable value is selected.  
}
  \item{t0}{
start time (defaults to 0)
}
  \item{\dots}{
additional arguments applied to the innovations.  For \code{rand.gen}, the standard deviation of the innovations generated by \code{rnorm} can be specified by \code{sd} or the mean by \code{mean} (see details and examples).  In addition, \code{rand.gen} may be overridden using a preset sequence of innovations specifying \code{innov} (see details and examples).
}
}
\details{
Will generate a time series of length \code{n} from the specified SARIMA model using simplified input.

The use of the term \code{mean} in \dots refers to the generation of normal innovations.  For example, \code{sarima.sim(ar=.9, mean=5)} will generate data using N(5,1) or 5+N(0,1) innovations, so that the constant in the model is 5 and the mean of the AR model is 5/(1-.9) = 50. In \code{sarima.sim(ma=.9, mean=5)}, however, the model mean is 5 (the constant). Also, a random walk with drift = .1 can be generated by \code{sarima.sim(d=1, mean=.1, burnin=0)}, which is equivalent to \code{cumsum(rnorm(500, mean=.1))}.  The same story goes if \code{sd} is specified; i.e., it's applied to the innovations. Because anything specified in \dots refers to the innovations, a simpler way to generate a non-zero mean is to add the value outside the call; see the examples.

If \code{innov} is used to input the innovations and override \code{rand.gen}, be sure that \code{length(innov)} is at least \code{n + burnin}. If the criterion is not met, the script will return less than the desired number of values and a warning will be given. 
}
\value{
{A time series of length n from the specified SARIMA model with the specified frequency if the model is seasonal and start time t0.}
}
\references{You can find demonstrations of  astsa capabilities at  
\href{https://github.com/nickpoison/astsa/blob/master/fun_with_astsa/fun_with_astsa.md}{FUN WITH ASTSA}.

The most recent version of the package can be found  at \url{https://github.com/nickpoison/astsa/}.

In addition, the News and ChangeLog files are at  \url{https://github.com/nickpoison/astsa/blob/master/NEWS.md}.

The webpages for the texts and some help on using R for time series analysis can be found at
 \url{https://nickpoison.github.io/}.   
}
\author{D.S. Stoffer
}
\note{The model autoregressive polynomial ('AR side' = AR x SAR) is checked for causality and the model moving average polynomial ('MA side' = MA x SMA) is checked invertibility.  The script stops and reports an error at the first violation of causality or invertibility; i.e., it will not report multiple errors.  
}
\examples{
## AR(2) with mean 50 [n = 500 is default]
y = sarima.sim(ar=c(1.5,-.75)) + 50
tsplot(y)

## ARIMA(0,1,1) with drift ['mean' refers to the innovations] 
tsplot(sarima.sim(ma=-.8, d=1, mean=.1))

## SAR(1) example from text
set.seed(666)   # not that 666
sAR = sarima.sim(sar=.9, S=12, n=36)
tsplot(sAR, type='c')
points(sAR, pch=Months, cex=1.1, font=4, col=1:4)

## SARIMA(0,1,1)x(0,1,1)_12 - B&J's favorite
set.seed(101010)
tsplot(sarima.sim(d=1, ma=-.4, D=1, sma=-.6, S=12, n=120))  

## infinite variance t-errors 
tsplot(sarima.sim(ar=.9, rand.gen=function(n, ...) rt(n, df=2) ))

## use your own innovations
dog = rexp(150, rate=.5)*sign(runif(150,-1,1))
tsplot(sarima.sim(n=100, ar=.99, innov=dog, burnin=50))

## generate seasonal data but no P, D or Q - you will receive 
## a message to make sure that you wanted to do this on purpose: 
tsplot(sarima.sim(ar=c(1.5,-.75), n=144, S=12), ylab='doggy', xaxt='n')
mtext(seq(0,144,12), side=1, line=.5, at=0:12)

}
\keyword{ ts}

