% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions_main.R
\name{LoTTA_fuzzy_BIN}
\alias{LoTTA_fuzzy_BIN}
\title{LoTTA_fuzzy_BIN}
\usage{
LoTTA_fuzzy_BIN(
  x,
  t,
  y,
  c_prior,
  jlb = 0.2,
  ci = 0.95,
  trimmed = NULL,
  outcome_prior = list(pr = 1e-04),
  n_min = 25,
  param = c("c", "j", "kl", "kr", "eff", "a0l", "a1l", "a2l", "a3l", "a0r", "a1r", "a2r",
    "a3r", "b1lt", "a1lt", "a2lt", "b2lt", "b1rt", "a1rt", "a2rt", "b2rt", "k1t", "k2t"),
  normalize = TRUE,
  n.chains = 4,
  burnin = 10000,
  sample = 1000,
  adapt = 500,
  inits = NULL,
  method = "parallel",
  seed = NULL,
  ...
)
}
\arguments{
\item{x}{\itemize{
\item is the score data
}}

\item{t}{\itemize{
\item is the treatment allocation data
}}

\item{y}{\itemize{
\item is the binary outcome data
}}

\item{c_prior}{\itemize{
\item specifies the cutoff prior in case of the unknown cutoff or the cutoff point
if the cutoff is known. Takes as value a number if the cutoff is known or a list of values otherwise.
For a continuous prior the list requires the following elements: clb - left end of the interval cub - right end of the interval
in which the scaled and translated beta distribution is defined, alpha (optional) - shape parameter, default value = 1, beta (optional) - shape parameter, default value = 1.
For a discrete prior the list requires the following elements: cstart - first point with positive prior mass, cend - last point with positive prior mass, grid - distance between the consecutive points in the support
weights (optional) - vector of weights assigned to each point in the support, default is vector of 1's (uniform distribution)
}}

\item{jlb}{\itemize{
\item minimum jump size
}}

\item{ci}{\itemize{
\item specifies the probability level 1-alpha for the highest posterior density intervals; default is ci = 0.95
}}

\item{trimmed}{\itemize{
\item takes as a value NULL or a vector of two values. It specifies potential trimming of the data.
If set to NULL no trimming is applied to the data. If a list of two values is provided the data is trimmed
to include data points with the score x in between those values; default is trimmed=NULL
}}

\item{outcome_prior}{\itemize{
\item takes as a value a list with elements 'pr'. 'pr' specifies precision in the normal priors on the coefficients in the
outcome function; default is list('pr'=0.0001)
}}

\item{n_min}{\itemize{
\item specifies the minimum number of data points to which a cubic part of the outcome function is fit to ensure stability of the sampling procedure; default is n_min=25
}}

\item{param}{\itemize{
\item takes as a value a vector with names of the parameters that are to be sampled; default is the list of all parameters
}}

\item{normalize}{\itemize{
\item specifies if the data is to be normalized. The data is normalized as follows. If the prior is continuous: x_normalized=(x-d)/s, where d=(min(x)+max(x))*0.5 and s=max(x)-min(x).
If the prior is discrete: x_normalized=x/s, where s=10^m, where m is chosen so that |max(abs(x))-1| is minimal.
The priors inside the model are specified for the normalized data, in extreme cases not normalizing the data may lead to unreliable results; default is normalize=TRUE
}}

\item{n.chains}{\itemize{
\item specifies the number of chains in the MCMC sampler; default is n.chains=4
}}

\item{burnin}{\itemize{
\item specifies the number of burnin iterations without the adaptive iterations; default is burnin=5000
}}

\item{sample}{\itemize{
\item specifies the number of samples per chain; default is samples=5000
}}

\item{adapt}{\itemize{
\item specifies the number of adaptive iterations in the MCMC sampler; default is adapt=1000
}}

\item{inits}{\itemize{
\item initial values for the sampler. By default the initial values are sampled inside the function. To run LoTTA with a method other than "parallel" inits must be set to NA or to a user defined value.
If the user wants to provide its own values please refer to run.jags manual; default inits=NULL
}}

\item{method}{\itemize{
\item set to default as 'parallel', which allows to sample the chains in parallel reducing computation time.
To read more about possible method values type ?run.jags; default method='parallel'
}}

\item{seed}{\itemize{
\item specifies the seed for replicability purposes; default is seed=NULL
}}

\item{...}{\itemize{
\item other arguments of run.jags function. For more details type ?run.jags
}}
}
\value{
The function returns the list with the elements:
\itemize{
\item Effect_estimate: contains a list with MAP estimate and HDI of the treatment effect, the cutoff location (if unknown) and the discontinuity size in the treatment probability function (compliance rate at c) on the original, unnormalized scale;
\item JAGS_output: contains output of the run.jags function for the normalized data if normalize=TRUE, based on this output mixing of the chains can be assessed;
\item Samples: contains posterior samples of the treatment effect (eff), cutoff location (c) if unknown, and compliance rate (j);
\item Normalized_data: contains a list of the normalized data (if normalized=TRUE) and the parameters used to normalize the data (see arg normalize);
\item Priors: contains a list of the priors' parameters ;
\item Inits contains the list of initial values  and .RNG.seed value
}
}
\description{
Function that fits LoTTA model to the fuzzy RD data with binary outcomes with an either known or unknown/suspected cutoff.
It supports two types of priors on the cutoff location: a scaled beta distribution of the form beta(alpha,beta)(cub-clb)+clb and a discrete distribution with the support of the form cstart+grid i for i=0,...,(cend-cstart)/grid.
The score does NOT have to be normalized beforehand. We recommend NOT to transform the data before imputing it into the function, except for initial trimming of the score which should be done beforehand.
The further trimming for the sensitivity analysis can be done through the function, which ensures that the data is normalized before the trimming.
}
\examples{
# functions to generate the data

ilogit <- function(x) {
  return(1 / (1 + exp(-x)))
}

fun_prob55 <- function(x) {
  P = rep(0, length(x))
  P[x >= 0.] = ilogit((8.5 * x[x >= 0.] - 1.5)) / 10.5 + 0.65 - 0.0007072
  P[x < 0.] = (x[x < 0.] + 1)^4 / 15 + 0.05
  return(P)
}

sample_prob55 <- function(x) {
  P = rep(0, length(x))
  P[x >= 0.] = ilogit((8.5 * x[x >= 0.] - 1.5)) / 10.5 + 0.65 - 0.0007072
  P[x < 0.] = (x[x < 0.] + 1)^4 / 15 + 0.05
  t = rep(0, length(x))
  for (j in 1:length(x)) {
    t[j] = sample(c(1, 0), 1, prob = c(P[j], 1 - P[j]))
  }
  return(t)
}

funB <- function(x) {
  y = x
  x2 = x[x >= 0]
  x1 = x[x < 0]
  y[x < 0] = 1 / (1 + exp(-2 * x1)) - 0.5 + 0.4
  y[x >= 0] = (log(x2 * 2 + 1) - 0.15 * x2^2) * 0.6 - 0.20 + 0.4
  return(y)
}

funB_sample_binary <- function(x) {
  y = x
  P = funB(x)
  for (j in 1:length(x)) {
    y[j] = sample(c(1, 0), 1, prob = c(P[j], 1 - P[j]))
  }
  return(y)
}

## Toy example - for the function check only! ##
N=100
x = sort(runif(N, -1, 1))
t = sample_prob55(x)
y = funB_sample_binary(x)
c_prior=0 # known cutoff c=0

# running LoTTA model on fuzzy RDD with binary outcomes;
out = LoTTA_fuzzy_BIN(x,t,y,c_prior,burnin = 50,sample = 50,adapt=10,n.chains=1
,method = 'simple',inits = NA)

## Use case example ##
\donttest{
  N=500
  x = sort(runif(N, -1, 1))
  t = sample_prob55(x)
  y = funB_sample_binary(x)
  # comment out to try different priors:
   c_prior=list(clb=-0.25,cub=0.25) # uniform prior on the interval [-0.25,0.25]
  # c_prior=list(cstart=-0.25,cend=0.25,grid=0.05) # uniform discrete prior
  # on -0.25, -0.2, ..., 0.25
  # c_prior=0 # known cutoff c=0

  # running LoTTA model on fuzzy RDD with binary outcomes and unknown cutoff;
  # cutoff = 0, compliance rate = 0.55, treatment effect = -0.3636364
  # remember to check convergence and adjust burnin, sample and adapt if needed
  out = LoTTA_fuzzy_BIN(x,t,y,c_prior,burnin = 10000,sample = 5000,adapt=1000)

  # print effect estimate:
  out$Effect_estimate
  # print JAGS output to asses convergence (the output is for normalized data)
  out$JAGS_output
  # plot posterior fit of the outcome function
  LoTTA_plot_outcome(out,nbins = 60)
  # plot posterior fit of the treatment probablity function
  LoTTA_plot_treatment(out,nbins = 60)
  # plot dependence of the treatment effect on the cutoff location
  LoTTA_plot_effect(out)

 }
}
