% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DBH.R
\name{DBH}
\alias{DBH}
\title{The Discrete Burr Hatke family}
\usage{
DBH(mu.link = "logit")
}
\arguments{
\item{mu.link}{defines the mu.link, with "logit" link as the default for the mu parameter. Other links are "probit" and "cloglog"'(complementary log-log)}
}
\value{
Returns a \code{gamlss.family} object which can be used
to fit a Discrete Burr-Hatke distribution
in the \code{gamlss()} function.
}
\description{
The function \code{DBH()} defines the Discrete Burr Hatke distribution, one-parameter
discrete distribution, for a \code{gamlss.family} object to be used in GAMLSS fitting
using the function \code{gamlss()}.
}
\details{
The Discrete Burr-Hatke distribution with parameters \eqn{\mu} has a support
0, 1, 2, ... and density given by

\eqn{f(x | \mu) = (\frac{1}{x+1}-\frac{\mu}{x+2})\mu^{x}}

The pmf is log-convex for all values of \eqn{0 < \mu < 1}, where \eqn{\frac{f(x+1;\mu)}{f(x;\mu)}}
is an increasing function in \eqn{x} for all values of the parameter \eqn{\mu}.

Note: in this implementation we changed the original parameters \eqn{\lambda} for \eqn{\mu},
we did it to implement this distribution within gamlss framework.
}
\examples{
# Example 1
# Generating some random values with
# known mu
y <- rDBH(n=1000, mu=0.74)

library(gamlss)
mod1 <- gamlss(y~1, family=DBH,
               control=gamlss.control(n.cyc=500, trace=FALSE))

# Extracting the fitted values for mu
# using the inverse logit function
inv_logit <- function(x) exp(x) / (1+exp(x))
inv_logit(coef(mod1, parameter="mu"))

# Example 2
# Generating random values under some model

# A function to simulate a data set with Y ~ DBH
gendat <- function(n) {
  x1 <- runif(n)
  mu    <- inv_logit(-3 + 5 * x1)
  y <- rDBH(n=n, mu=mu)
  data.frame(y=y, x1=x1)
}

datos <- gendat(n=150)

mod2 <- NULL
mod2 <- gamlss(y~x1, family=DBH, data=datos,
               control=gamlss.control(n.cyc=500, trace=FALSE))

summary(mod2)

# Example 3
# number of carious teeth among the four deciduous molars.
# Taken from EL-MORSHEDY (2020) page 74364.

y <- rep(0:4, times=c(64, 17, 10, 6, 3))

mod3 <- gamlss(y~1, family=DBH,
               control=gamlss.control(n.cyc=500, trace=FALSE))

# Extracting the fitted values for mu
# using the inverse link function
inv_logit <- function(x) 1/(1 + exp(-x))
inv_logit(coef(mod3, what="mu"))
}
\references{
\insertRef{el2020discrete}{DiscreteDists}
}
\seealso{
\link{dDBH}.
}
\author{
Valentina Hurtado Sepulveda, \email{vhurtados@unal.edu.co}
}
