% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BuyseMultComp.R
\name{BuyseMultComp}
\alias{BuyseMultComp}
\title{Adjustment for Multiple Comparisons}
\usage{
BuyseMultComp(
  object,
  cluster = NULL,
  linfct = NULL,
  rhs = NULL,
  endpoint = NULL,
  statistic = NULL,
  cumulative = TRUE,
  conf.level = NULL,
  band = TRUE,
  global = FALSE,
  alternative = NULL,
  transformation = NULL,
  ...
)
}
\arguments{
\item{object}{A BuyseTest object or a list of BuyseTest objects. All objects should contain the same endpoints.}

\item{cluster}{[character] name of the variable identifying the observations in the dataset used by each BuyseTest model.
Only relevant when using a list of BuyseTest objects to correctly combine the influence functions.
If NULL, then it is assumed that the BuyseTest objects correspond to different groups of individuals.}

\item{linfct}{[numeric matrix] a contrast matrix of size the number of endpoints times the number of BuyseTest models.}

\item{rhs}{[numeric vector] the values for which the test statistic should be tested against. Should have the same number of rows as \code{linfct}.}

\item{endpoint}{[character or numeric vector] the endpoint(s) to be considered.}

\item{statistic}{[character] the statistic summarizing the pairwise comparison:
\code{"netBenefit"} displays the net benefit, as described in Buyse (2010) and Peron et al. (2016)),
\code{"winRatio"} displays the win ratio, as described in Wang et al. (2016),
\code{"favorable"} displays the proportion in favor of the treatment (also called Mann-Whitney parameter), as described in Fay et al. (2018).
\code{"unfavorable"} displays the proportion in favor of the control.
Default value read from \code{BuyseTest.options()}.}

\item{cumulative}{[logical] should the summary statistic be cumulated over endpoints?
Otherwise display the contribution of each endpoint.}

\item{conf.level}{[numeric] confidence level for the confidence intervals.
Default value read from \code{BuyseTest.options()}.}

\item{band}{[logical] Should confidence intervals and p-values adjusted for multiple comparisons be computed.}

\item{global}{[logical] Should global test (intersection of all null hypotheses) be made?}

\item{alternative}{[character] the type of alternative hypothesis: \code{"two.sided"}, \code{"greater"}, or \code{"less"}.
Default value read from \code{BuyseTest.options()}.}

\item{transformation}{[logical]  should the CI be computed on the logit scale / log scale for the net benefit / win ratio and backtransformed.
Otherwise they are computed without any transformation.
Default value read from \code{BuyseTest.options()}. Not relevant when using permutations or percentile bootstrap.}

\item{...}{argument passsed to the function \code{transformCIBP} of the riskRegression package.}
}
\value{
An S3 object of class \code{BuyseMultComp}.
}
\description{
Adjust p-values and confidence intervals estimated via GPC for multiple comparisons.
}
\details{
Simulateneous confidence intervals and adjusted p-values are computed using a single-step max-test approach via the function \code{transformCIBP} of the riskRegression package.
This corresponds to the single-step Dunnett described in Dmitrienko et al (2013) in table 2 and section 7.
}
\examples{
#### simulate data ####
set.seed(10)
df.data <- simBuyseTest(1e2, n.strata = 3)

#### adjustment for all univariate analyses ####
ff1 <- treatment ~ TTE(eventtime, status = status, threshold = 0.1)
ff2 <- update(ff1, .~. + cont(score, threshold = 1))
BT2 <- BuyseTest(ff2, data= df.data, trace = FALSE)

## (require riskRegression >= 2021.10.04 to match)
confint(BT2, cumulative = FALSE) ## not adjusted
confintAdj <- BuyseMultComp(BT2, cumulative = FALSE, endpoint = 1:2) ## adjusted
confintAdj
if(require(lava)){
cor(lava::iid(confintAdj)) ## correlation between test-statistic
}

#### 2- adjustment for multi-arm trial ####
## case where we have more than two treatment groups
## here strata will represent the treatment groups
df.data$strata <- as.character(df.data$strata)
df.data$id <- paste0("Id",1:NROW(df.data)) ## define id variable

BT1ba <- BuyseTest(strata ~ TTE(eventtime, status = status, threshold = 1),
                   data= df.data[strata \%in\% c("a","b"),], trace = FALSE)
BT1ca <- BuyseTest(strata ~ TTE(eventtime, status = status, threshold = 0.1),
                   data= df.data[strata \%in\% c("a","c"),], trace = FALSE)
BT1cb <- BuyseTest(strata ~ TTE(eventtime, status = status, threshold = 0.1),
                   data= df.data[strata \%in\% c("b","c"),], trace = FALSE)
rbind("b-a" = confint(BT1ba),
      "c-a" = confint(BT1ca),
      "c-b" = confint(BT1cb)) ## not adjusted
confintAdj <- BuyseMultComp(list("b-a" = BT1ba, "c-a" = BT1ca, "c-b" = BT1cb),
                            cluster = "id", global = TRUE)
confintAdj
if(require(lava)){
cor(lava::iid(confintAdj))
}
}
\references{
Dmitrienko, A. and D'Agostino, R., Sr (2013), Traditional multiplicity adjustment methods in clinical trials. Statist. Med., 32: 5172-5218. https://doi.org/10.1002/sim.5990
}
\keyword{htest}
